/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.introspection;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.stp.core.sca.SCAObject;

/**
 * Provides a simple interface for non-EMF models to normalize their content
 * into an EMF structure.
 * 
 * The following pattern should be followed for Component Types:
 * 
 * 
 * For component type, this should start be of the following pattern:
 * 
 * For each Service:
 * 
 * startFeature: ComponentType>Service: feature:{@link org.eclipse.stp.core.sca.SCAPackage#getComponentType_Services()}
 * type: {@link org.eclipse.stp.core.sca.SCAPackage#getService()}
 * substitutionGroupFeature:
 * {@link org.eclipse.stp.core.sca.SCAPackage#getComponentType_Services()} * -
 * Here the substitutionGroup feature is the same as the feature since
 * substituion of this type is not allowed. attribute()
 * ComponentType>Service>_Name:{@link org.eclipse.stp.core.sca.SCAPackage#getService_Name()}
 * startFeature() ComponentType>Service>Interface: feature:{@link org.eclipse.stp.core.sca.SCAPackage#getService_Interface()}
 * type: {@link org.eclipse.stp.core.sca.SCAPackage#getJavaInterface()}
 * substitutionGroupFeature:
 * {@link org.eclipse.stp.core.sca.SCAPackage#getSCACoreRoot_InterfaceJava()}
 * 
 * attribute( ... isUniqueKey=true)
 * {@link org.eclipse.stp.core.sca.SCAPackage#getJavaInterface_Interface()};
 * endFeature() ComponentType>Service>Interface:
 * {@link org.eclipse.stp.core.sca.SCAPackage#getService_Interface()}
 * endFeature() ComponentType>Service:{@link org.eclipse.stp.core.sca.SCAPackage#getComponentType_Services()}.
 * 
 * @since 1.0
 * 
 */
public interface IDynamicModelIntrospector {

   /**
    * Indicate the BATCH mode. The current state of the model will be announced,
    * and the IDynamicModelIntrospector will handle the merge.
    */
   int BATCH              = 1;

   /**
    * Indicate the feature being started was added. Describe all attributes and
    * contained features.
    */
   int INCREMENTAL_ADD    = 2;

   /**
    * Indicate the feature being started has changed in some way. Describe the
    * set of unique attributes, and then any modified attribute values or
    * contained features.
    */
   int INCREMENTAL_UPDATE = 4;

   /**
    * Indicate the feature being started has been removed. Describe the set of
    * unique attributes or features that contain unique attributes to identify
    * the model element to remove.
    */
   int INCREMENTAL_REMOVE = 8;

   /**
    * Should be called before the introspection begins (e.g. at the beginning of
    * a parse on the implementatation model).
    * 
    * @param feature
    *           The root feature of the given root. In general, this is an
    *           element from the document root.
    * @param root
    *           The target that will be (potentially) modified.
    * 
    */
   void startIntrospection(EStructuralFeature feature, SCAObject root);

   /**
    * Indicate which feature is being parsed, the type of that feature, and the
    * serialization feature for that type.
    * 
    * <p>
    * For example: <code>
    * SCAPackage pkg = SCAPackage.eINSTANCE;
    * modelIntrospector.startFeature(
    * 		pkg.getService_Interface(), // we are parsing an interface for a service
    * 		pkg.getJavaInterface(), // its type is com.ibm.ccl.soa.core.JavaInterface
    * 		pkg.getSCACoreRoot_InterfaceJava()); // its serialization format is "interface.java"
    * </code>
    * <p>
    * 
    * 
    * @param feature
    *           The feature that is being updated
    * @param type
    *           The type of the value that will be set on the feature
    * @param substitutionGroupFeature
    *           The feature that this type will be serialized as
    */
   void startFeature(EStructuralFeature feature, EClass type,
         EStructuralFeature substitutionGroupFeature);

   /**
    * Indicate which feature is being parsed, the type of that feature, and the
    * serialization feature for that type.
    * 
    * <p>
    * For example: <code>
    * SCAPackage pkg = SCAPackage.eINSTANCE;
    * modelIntrospector.startFeature(
    * 		pkg.getService_Interface(), // we are parsing an interface for a service
    * 		pkg.getJavaInterface(), // its type is com.ibm.ccl.soa.core.JavaInterface
    * 		pkg.getSCACoreRoot_InterfaceJava()); // its serialization format is "interface.java"
    * </code>
    * <p>
    * 
    * 
    * @param feature
    *           The feature that is being updated
    * @param type
    *           The type of the value that will be set on the feature
    * @param substitutionGroupFeature
    *           The feature that this type will be serialized as
    * @param announcementType
    *           Should be one of {@link #BATCH}, {@link #INCREMENTAL_ADD},
    *           {@link #INCREMENTAL_UPDATE}, or {@link #INCREMENTAL_REMOVE}
    */
   void startFeature(EStructuralFeature feature, EClass type,
         EStructuralFeature substitutionGroupFeature, int announcementType);

   /**
    * Indicate the attribute being parsed and the current value.
    * 
    * @param attribute
    *           The attribute of the current feature that has a value
    * @param value
    *           The current value of the attribute
    * @param isUniqueKey
    *           True marks the attribute as one of possibly many that identifies
    *           the uniqueness of this modele element.
    */
   void attribute(EAttribute attribute, Object value, boolean isUniqueKey);

   /**
    * Indicate when a feature is fully parsed.
    * 
    * @param feature
    *           The feature that has been completed.
    */
   void endFeature(EStructuralFeature feature);

   /**
    * Indicate the parsing is complete. Apply changes (if any).
    * 
    */
   void endIntrospection();

   /**
    * Delete the untouched nodes. Anything not announced should be removed.
    * 
    * @param feature
    *           The feature that has been updated.
    */
   void removeUntouched(EStructuralFeature feature);

   /**
    * Instruct the parser to dispose of any resources it may be holding onto as
    * a result of a startIntrospection() ... endIntrospection() sequence.
    */
   void reset();

}
