/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.create.operations;

import java.util.Collections;
import java.util.Set;

import org.apache.xerces.util.XMLChar;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.operations.AbstractScribblerDataModelProvider;
import org.eclipse.stp.core.internal.Messages;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.resources.SOAConstants;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

public class ComponentTypeCreationDataModelProvider extends
      AbstractScribblerDataModelProvider implements
      IComponentTypeCreationDataModelProperties {

   public IDataModelOperation getDefaultOperation() {
      return new ComponentTypeCreationOperation(model);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.frameworks.datamodel.IDataModelProvider#getPropertyNames()
    */
   public Set getPropertyNames() {
      Set propertyNames = super.getPropertyNames();
      propertyNames.add(NAME);
      propertyNames.add(CONTAINER);
      propertyNames.add(REFERENCES);
      propertyNames.add(PROPERTIES);
      propertyNames.add(SERVICES);

      return propertyNames;
   }

   public Object getDefaultProperty(String propertyName) {
      if (propertyName.equals(REFERENCES))
         return Collections.EMPTY_LIST;
      else if (propertyName.equals(PROPERTIES))
         return Collections.EMPTY_LIST;
      else if (propertyName.equals(SERVICES))
         return Collections.EMPTY_LIST;

      return super.getDefaultProperty(propertyName);
   }

   public IStatus validate(String propertyName) {
      if (propertyName.equals(CONTAINER))
         return validateContainer();
      if (propertyName.equals(NAME))
         return validateName();

      return super.validate(propertyName);
   }

   private IStatus validateName() {
      IStatus returnStatus = OK_STATUS;
      // Multiple check needs to be perfomed.

      // First Check for empty
      String name = getStringProperty(NAME);
      if (name == null || name.length() <= 0)
         returnStatus = STPCorePlugin.createErrorStatus(
               Messages.nameBlank_errorText, null);

      if (!returnStatus.isOK())
         return returnStatus;

      // Second check for valid characters
      returnStatus = validateNameForValidCharaters(name);
      if (!returnStatus.isOK())
         return returnStatus;

      // Third check if the name entered is unique within the container.
      IFile file = null;
      IContainer container = (IContainer) getProperty(CONTAINER);
      IJavaElement javaElement = JavaCore.create(container);
      // This could either be a PackageFragmentRoot or IProject in which case we
      // default.
      if (javaElement.getElementType() == IJavaElement.PACKAGE_FRAGMENT) {
         file = container.getFile(new Path(name + '.'
               + SOAConstants.COMPONENT_TYPE_EXT));
         if (file != null && file.exists())
            returnStatus = STPCorePlugin.createErrorStatus(NLS.bind(
                  Messages.nameNotValid_errorText, name), null);
      }

      if (javaElement.getElementType() == IJavaElement.JAVA_PROJECT) {
         file = container
               .getFile(new Path(
                     "sca/java/impl/" + name + '.' + SOAConstants.COMPONENT_TYPE_EXT)); //$NON-NLS-1$
         if (file != null && file.exists())
            returnStatus = STPCorePlugin.createErrorStatus(NLS.bind(
                  Messages.nameNotValid_errorText, name), null);
      }

      return returnStatus;
   }

   // The following characters are restricted characters of
   // org.eclipse.emf.common.util.URI which is
   // used while loading the extension model (.modulex) file.
   protected static final String URI_RESTRICTED_CHARACTERS = ":?#@!"; //$NON-NLS-1$

   private IStatus validateNameForValidCharaters(String name) {
      if (!XMLChar.isValidNCName(name)) {
         return STPCorePlugin.createErrorStatus(NLS.bind(
               Messages.nameNotValid_errorText, name), null);
      }
      for (int i = 0; i < URI_RESTRICTED_CHARACTERS.length(); i++) {
         if (name.indexOf(URI_RESTRICTED_CHARACTERS.charAt(i)) != -1) {
            return STPCorePlugin.createErrorStatus(NLS.bind(
                  Messages.portionOfNameNotValid_errorText, name, String
                        .valueOf(URI_RESTRICTED_CHARACTERS.charAt(i))), null);
         }
      }

      return Status.OK_STATUS;

   }

   private IStatus validateContainer() {
      Object object = getProperty(CONTAINER);
      if (object == null || !(object instanceof IContainer))
         return STPCorePlugin.createErrorStatus(
               Messages.invalidObjectPassed_errorText, null);

      if (!((IContainer) object).exists())
         return STPCorePlugin.createErrorStatus(
               Messages.containerNotExisting_errorText, null);

      return OK_STATUS;
   }
}
