/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.sca;

import java.util.List;

/**
 * This interface represents a source for a conceptual wire. A wire is a
 * conceptual mapping of a source element in the model to a target element.
 * 
 * <p>
 * The {@link #getName()} API will return the soure URI to this source.
 * </p>
 * 
 * <p>
 * Use the {@link org.eclipse.stp.core.sca.SCAObject#getFlags()} method to get
 * the flags for this target and then you can use the
 * {@link org.eclipse.stp.core.sca.Flags#isBoundaryElement(int)} method to
 * determine if the target is a boundary target element. Boundary elements are
 * EntryPoint and ExternalService elements.
 * </p>
 * <p>
 * You can obtain a WireSource by using APIs on Composite or Component.
 * </p>
 * <p>
 * If the element returns true for {@link #isComponentOwned()} you may cast down
 * to {@link org.eclipse.stp.core.sca.ComponentWireHandle} for additional APIs.
 * </p>
 * <p>
 * This interface is not intended to be implemented by clients.
 * </p>
 * 
 * @see org.eclipse.stp.core.sca.WireTarget
 * @see org.eclipse.stp.core.sca.Reference
 * @see org.eclipse.stp.core.sca.EntryPoint
 * @see org.eclipse.stp.core.sca.ExternalService
 * @see org.eclipse.stp.core.sca.Composite#resolveWireSource(String)
 * @see org.eclipse.stp.core.sca.Component#getWireSource(Reference)
 * @see org.eclipse.stp.core.sca.Component#getWireSource(String)
 * @since 1.0
 */
public interface WireSource extends SCAObject {

   /**
    * Return an Interface that is assigned to this source of a wire.
    * 
    * @return Interface
    */
   Interface getInterface();

   /**
    * NOT IMPLEMENTED.
    * 
    * @param wireTarget
    *           The wireTarget to test for compatibility.
    * @return True if and only if the Interface of the given wireTarget is
    *         compatible with this WireSource.
    */
   boolean matches(WireTarget wireTarget);

   /**
    * This method is used as a convenience to create a wire to the WireTarget
    * that is passed. This method will not perform any validation to ensure that
    * the wire is valid.
    * 
    * @param wireTarget
    */
   void createWireToTarget(WireTarget wireTarget);

   /**
    * This method is used as a convenience to delete a wire to the WireTarget
    * that is passed. This method will not perform any validation to ensure that
    * the wire is valid.
    * 
    * @param wireTarget
    *           The wire target to delete from this wire source's inlined wires.
    * @return True if the wire was deleted and false otherwise.
    */
   boolean deleteWireToTarget(WireTarget wireTarget);

   /**
    * This is a convenience method to return an unmodifiable list of WireTarget
    * elements which are actually wired to this WireSource.
    * 
    * @return List of {@link WireTarget}
    */
   List getWiredTargets();

   /**
    * Return true if this source was resolved to an element in the SCA model. If
    * false is returned you can use the {@link SCAObject#getName()} method to
    * get the original source URI.
    * 
    * @return boolean
    */
   boolean isResolved();

   /**
    * Return true if this WireSource originates from a {@link Component}.
    * 
    * @return boolean
    */
   boolean isComponentOwned();

}
