/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Interface;
import org.eclipse.stp.core.sca.JavaInterface;
import org.eclipse.stp.core.sca.Property;
import org.eclipse.stp.core.sca.Reference;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.stp.core.sca.Service;
import org.eclipse.stp.core.sca.WSDLPortType;

public class SOACoreUtils {
   public static final SOACoreUtils INSTANCE = new SOACoreUtils();

   private SOACoreUtils() {
      super();
   }

   /**
    * 
    * @param services
    *           A list of Service objects
    * @return A list of Interface(s) from the list of Services that are Java
    *         types
    */
   public List getAllJavaInterfaces(List services) {
      if (services == null)
         return Collections.EMPTY_LIST;

      List allJavaInterfaces = new ArrayList();

      for (Iterator iter = services.iterator(); iter.hasNext();) {
         Service service = (Service) iter.next();
         Interface intf = service.getInterface();
         if (intf instanceof JavaInterface) {
            allJavaInterfaces.add(intf);
         }
      }

      return allJavaInterfaces;
   }

   /**
    * 
    * @param services
    *           A list of Service objects
    * @return A list of Interface(s) from the list of Services that are WSDL
    *         types
    */
   public List getAllWSDLInterfaces(List services) {
      if (services == null)
         return Collections.EMPTY_LIST;

      List allWSDLInterfaces = new ArrayList();
      for (Iterator iter = services.iterator(); iter.hasNext();) {
         Service service = (Service) iter.next();
         Interface intf = service.getInterface();
         if (intf instanceof WSDLPortType) {
            allWSDLInterfaces.add(intf);
         }
      }

      return allWSDLInterfaces;
   }

   /**
    * 
    * @param references
    *           A list of Reference(s)
    * @return
    */
   public List getAllJavaReferenceInterfaces(List references) {
      if (references == null)
         return Collections.EMPTY_LIST;

      List allJavaReferenceInterfaces = new ArrayList();
      Iterator referenceListItr = references.iterator();
      while (referenceListItr.hasNext()) {
         Reference reference = (Reference) referenceListItr.next();

         Interface intf = reference.getInterface();
         if (intf instanceof JavaInterface) {
            allJavaReferenceInterfaces.add(intf);
         }
      }

      return allJavaReferenceInterfaces;
   }

   /**
    * 
    * @param references
    *           A list of Reference(s)
    * @return
    */
   public List getAllWSDLReferences(List references) {
      if (references == null)
         return Collections.EMPTY_LIST;

      List allWSDLReferences = new ArrayList();

      Iterator referenceListItr = references.iterator();
      while (referenceListItr.hasNext()) {
         Reference reference = (Reference) referenceListItr.next();

         Interface intf = reference.getInterface();
         if (intf instanceof WSDLPortType)
            allWSDLReferences.add(reference);

      }
      return allWSDLReferences;
   }

   /**
    * 
    * @param references
    *           A list of Reference(s)
    * @return
    */
   public List getAllWSDLReferenceInterfaces(List references) {
      if (references == null)
         return Collections.EMPTY_LIST;

      List allWSDLReferenceInterfaces = new ArrayList();
      Iterator referenceListItr = references.iterator();
      while (referenceListItr.hasNext()) {
         Reference reference = (Reference) referenceListItr.next();

         Interface intf = reference.getInterface();
         if (intf instanceof WSDLPortType)
            allWSDLReferenceInterfaces.add(intf);
      }

      return allWSDLReferenceInterfaces;
   }

   public SCAObject getSCAObjectByName(String name,
         ComponentType componentType, Class type) {
      SCAObject scaObject = null;

      List candidates = null;
      if (type == Reference.class) {
         candidates = componentType.getReferences();
      } else if (type == Property.class) {
         candidates = componentType.getProperties();
      } else if (type == Service.class) {
         candidates = componentType.getServices();
      }

      for (int i = 0; i < candidates.size() && scaObject == null; i++) {
         SCAObject candidate = (SCAObject) candidates.get(i);
         if (candidate.getName().equals(name))
            scaObject = candidate;
      }

      return scaObject;
   }
}
