/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.ui.contribution;

import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

/**
 * This class allows an editor to display and update an SOA model.
 * <p>
 * A typical calling sequence is as follows:
 * <ul>
 * <li><code>{@link #createControls(Composite, Object) ISOAPropertiesContribution.createControls()}</code></li>
 * <li><code>{@link #setInput(Object, IEditorHandler) ISOAPropertiesContribution.setInput()}</code></li>
 * <li><code>{@link #aboutToBeShown() ISOAPropertiesContribution.aboutToBeShown()}</code></li>
 * <li><code>{@link #refresh() ISOAPropertiesContribution.refresh()}</code></li>
 * <li><code>{@link #aboutToBeHidden() ISOAPropertiesContribution.aboutToBeHidden()}</code></li>
 * <li><code>{@link #dispose() ISOAPropertiesContribution.dispose()}</code></li>
 * </ul>
 * </p>
 * <p>
 * Inbetween the calls to:
 * </p>
 * <ul>
 * <li><code>{@link #aboutToBeShown() ISOAPropertiesContribution.aboutToBeShown()}</code>, and</li>
 * <li><code>{@link #aboutToBeHidden() ISOAPropertiesContribution.aboutToBeHidden()}</code></li>
 * </ul>
 * <p>
 * the following may be called:
 * </p>
 * <ol>
 * <li><code>{@link #refresh() ISOAPropertiesContribution.refresh()}</code>
 * <p>
 * Requires the controls to be updated with the current input
 * </p>
 * </li>
 * <li><code>{@link #requiresNewControls(Object) ISOAPropertiesContribution.requiresNewControls()}</code>
 * <p>
 * Controls may be reused to display objects of compatible types.
 * Calls to this method can be used to determine if the current set of controls
 * are sufficient to display the new object of compatible types.  If this method
 * returns <code>false</code>, then the following sequence of calls are made:
 * </p>
 * <ul>
 * <li><code>{@link #setInput(Object, IEditorHandler) ISOAPropertiesContribution.setInput()}</code></li>
 * <li><code>{@link #aboutToBeShown() ISOAPropertiesContribution.aboutToBeShown()}</code></li>
 * <li><code>{@link #refresh() ISOAPropertiesContribution.refresh()}</code></li>
 * </ul>
 * <p>
 * If this method returns <code>true</code>, then the following sequence of calls
 * are made:
 * </p> 
 * <ul>
 * <li><code>{@link #aboutToBeHidden() ISOAPropertiesContribution.aboutToBeHidden()}</code></li>
 * <li><code>{@link #dispose() ISOAPropertiesContribution.dispose()}</code></li>
 * <li><code>{@link #createControls(Composite, Object) ISOAPropertiesContribution.createControls()}</code></li>
 * <li><code>{@link #setInput(Object, IEditorHandler) ISOAPropertiesContribution.setInput()}</code></li>
 * <li><code>{@link #aboutToBeShown() ISOAPropertiesContribution.aboutToBeShown()}</code></li>
 * <li><code>{@link #refresh() ISOAPropertiesContribution.refresh()}</code></li>
 * </ul>
 * </ol>
 * 
 */

public interface ISOAPropertiesContribution {

	/**
	 * Creates the controls for the PropertyPage.
	 * 
	 * @param composite
	 *            the composite for the property page.
	 * @param object
	 *            object to create controls for.  The controls should not be populated with data until
	 *            a future call to {@link #refresh() } is made.
	 */
	public void createControls(Composite composite,
			Object object);

	/**
	 * Returns whether a new set of controls needs to be created to display this new object.
	 *   
	 * @param newObject
	 *          the object about to be displayed in the controls
	 * @return
	 *          Returns <code>true</code> if the current set of controls is not sufficient to
	 *          display this new object and a new set of controls needs to be built.  Returns <code>false</code>
	 *          if the current set of controls can be reused to display the new object.
	 * 
	 */
	public boolean requiresNewControls(Object newObject);

	/**
	 * Identifies the selected object whose properties are about to be shown.
	 * 
	 * @param object
	 *          the selected model object    
	 * @param editorHandler
	 *            object used to report errors to the user and to handle updates to the model
	 */
	public void setInput(Object object, IEditorHandler editorHandler);

	/**
	 * Notifies the contributor that its controls are about to be shown. It is
	 * expected that contributors enable domain-related functions in this
	 * method, most commonly by adding listeners.
	 * <p>
	 * Since the controls are not visible, the contributor should wait for the
	 * {@link #refresh() } before updating the property page controls.
	 * </p>
	 */
	public void aboutToBeShown();

	/**
	 * Notifies the contributor that its controls are about to be hidden. It is
	 * expected that contributors disable domain-related functions in this
	 * method, most commonly by removing listeners.
	 */
	public void aboutToBeHidden();

	/**
	 * Dispose this property page.
	 */
	public void dispose();

	/**
	 * Refresh the contents of the controls displayed in the property pages.
	 */
	public void refresh();
	
	/**
	 * Create and initialize an instance of the type.  The type of object to create may be
	 * an SCAObject or an object contained within an SCAObject.
	 * 
	 *  @param container - object in the SOA model to which the new instance is to be added
	 *  @param targetTypeNamespace - the namespace of the type of object to create
	 *  @param targetType - type of object to create
	 *  @return
	 *          a new instance of type <code>targetType</code>
	 */
	public Object createInstance(Object container, String targetTypeNamespace, String targetType);
	
	/**
	 * Returns the icon for the type of the contribution.
	 * 
	 * @return the icon 
	 */
	public Image getIcon();
	
	/**
	 * Returns a short description of the object.
	 * 
	 * @param object
	 *            the object whose short description is to be returned
	 * @return
	 *            the short description
	 */
	public String getShortDescription(Object object);
	
	/**
	 * Returns a long description of the object
	 * 
	 * @param object
	 *            the object whose short description is to be returned
	 * @return
	 *            the long description
	 */
	public String getLongDescription(Object object);
	
	/**
	 * Returns the name for the type.
	 *  @return
	 *        a string displayed to the user for this type
	 */
	public String getTypeName();
	
	/**
	 * Return the descriptors for the tabs required by the contribution.  This method only applies to
	 * Component Implementations, ImportBindings and ExportBindings.
	 * 
	 * @return the list of tab descriptors
	 */
	public ITabDescriptor[] getTabDescriptors();
}
