/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/** This utility class contains helper functions for dealing with inputstreams.
 */
public final class InputStreamHelper {
    static final int BLOCK_SIZE = 4096;
    
    private InputStreamHelper() {}

    /** Reads everything from an <tt>InputStream</tt> into an <tt>OutputStream</tt>.
     * @param r The <tt>InputStream</tt> to be read from.
     * @param w The <tt>OutputStream</tt> to write to.
     * @throws IOException In case of trouble reading/writing.
     */
    public static void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];

        try {
            int length = 0;
            int offset = 0;

            while ((length = r.read(bytes, offset, bytes.length - offset)) != -1) {
                offset += length;

                if (offset == bytes.length) {
                    w.write(bytes, 0, bytes.length);
                    offset = 0;
                }
            }
            if (offset != 0) {
                w.write(bytes, 0, offset);
            }
        } finally {
            bytes = null;
        }
    }

    /** Reads everything from an <tt>InputStream</tt> into a byte array.
     * @param r The <tt>InputStream</tt> to read from. 
     * @return The byte array containing the contents of the input stream.
     * @throws IOException In case of trouble reading.
     */
    public static byte[] drain(InputStream r) throws IOException {
        int avail = r.available();
        ByteArrayOutputStream bytes;
        if (avail > 0) {
            bytes = new ByteArrayOutputStream(avail);
        } else {
            bytes = new ByteArrayOutputStream();
        }
        drain(r, bytes);
        return bytes.toByteArray();
    }
 
    public static void close(InputStream is) {
        if (is != null) {
            try {
                is.close();
            } catch (IOException ioe) {
                // can't really do much with this...
                ioe.printStackTrace();
            }
        }
    }

    public static void close(OutputStream os) {
        if (os != null) {
            try {
                os.close();
            } catch (IOException ioe) {
                // can't really do much with this...
                ioe.printStackTrace();
            }
        }
    }
}
