/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.util.HashMap;
import java.util.Map;



import org.eclipse.stp.xef.util.QNameHelper;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Event;

public class QNameControlAccessor implements ControlAccessor {
    private final Combo combo;
    private final Map<String, String> map;
    
    QNameControlAccessor(Combo c, Map<String, String> m) {
        combo = c;
        map = m;
    }

    public String getValue() {
        return combo.getText();
    }

    public void setValue(String value) {
        if (combo.indexOf(value) == -1 && value.length() != 0) {
            combo.add(value, 0);
        }
        combo.select(combo.indexOf(value));
        combo.notifyListeners(SWT.Selection, new Event());
    }
    
    public String convertToFullName(String qname) {
        String fullName = qname;
        String prefix = QNameHelper.getPrefixFromQName(qname);
        String local = QNameHelper.getLocalFromQName(qname);
        if (map != null && map.get(prefix) != null) {
            fullName = QNameHelper.convertToFullName(map.get(prefix), local);
        }
        return fullName;        
    }
    
    public String[] getQNameValues(String[] values) {
        if (values == null || values.length == 0) {
            return new String[] {};
        }
        
        String[] qnameValues = new String[values.length];
        for (int i = 0; i < values.length; i++) {
            qnameValues[i] = convertToQName(values[i]);
        }
        return qnameValues;
    }
    
    public String convertToQName(String fullName) {
        String prefixName = fullName;
        String local = QNameHelper.getLocalNameFromFullName(fullName);
        String namespace = QNameHelper.getNamespaceFromFullName(fullName);
        if (map != null) {
            for (String key : map.keySet()) {
                if (namespace.equals(map.get(key))) {
                    prefixName = QNameHelper.convertToQName(key, local);
                    break;
                }
            }
        }
        return prefixName;
    }

    @SuppressWarnings("unchecked")
    public static Map<String, String> createNamespaceMap(Map existingNamespaces, String[] values) {
        Map<String, String> map = new HashMap<String, String>(existingNamespaces);        
        if (values != null) {
            for (String value : values) {
                addPrefixToMap(map, value);
            }
        }
        return map;
    }
    
    private static void addPrefixToMap(Map<String, String> map,
                                       String fullName) {
        if (map == null) {
            return;
        }
        
        String uri = QNameHelper.getNamespaceFromFullName(fullName);
        if (uri == null ||uri.length() == 0) {
            return;
        }

        // if a mapping to this uri already exists, use that prefix
        for (String key : map.keySet()) {
            if (uri.equals(map.get(key))) {
                return;
            }
        }
        
        // setup new prefix mapping
        String prefixBase = "ns";
        int count = 1;
        
        String prefix = prefixBase + String.valueOf(count);
        while (!validPrefix(prefix, uri, map)) {
            prefix = prefixBase + String.valueOf(++count);
        }
        
        map.put(prefix, uri);
        return;
    }
    
    @SuppressWarnings("unchecked")
    private static boolean validPrefix(String prefix, String uri, Map map) {
        if (map.containsKey(prefix)) {
            return false;
        }
        
        return true;
    }

}
