/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.dialogs;

import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.ILogicalPackage;
import org.eclipse.stp.soas.deploy.core.IPackage;
import org.eclipse.stp.soas.deploy.core.IPackageConstructorExtension;
import org.eclipse.stp.soas.deploy.core.IPhysicalPackage;
import org.eclipse.stp.soas.deploy.core.IPhysicalPackageExtension;
import org.eclipse.stp.soas.deploy.core.ISupportedServerType;
import org.eclipse.stp.soas.deploy.core.IVersion;
import org.eclipse.stp.soas.deploy.core.ServerType;
import org.eclipse.stp.soas.deploy.core.utils.DeploymentUtil;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployFileFactory;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.ui.ConnectionProfileTreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerType;
import org.eclipse.wst.server.core.ServerCore;

/**
 * @author rcernich
 * 
 * Created on Feb 24, 2004
 */
public class ServerSelectionDialogPage extends DialogPage {

	public static final String TITLE;
	public static final String ICON_PATH;
	public static final ImageDescriptor ICON_DESC;

	public static final String LABEL_ADD_SERVER;
	public static final int MIN_LIST_HEIGHT;
	public static final int INDENT;

	private ConnectionProfileTreeViewer mCPViewer;
	private IPackage mPackage;
	private DeployPackage mDeployPackage;
	private Root mRoot;
	private DeployServer mSelectedServer;

	/**
	 * 
	 */
	public ServerSelectionDialogPage() {
		this(TITLE, ICON_DESC);
	}

	/**
	 * @param title
	 * @param image
	 */
	public ServerSelectionDialogPage(String title, ImageDescriptor image) {
		super(title, image);
		setNeedsApplyButton(false);
	}

	public void init(DeployPackage pkg, DeployServer server) {
		mDeployPackage = pkg;
		mRoot = mDeployPackage.getRoot();
		mSelectedServer = server;
		mPackage = org.eclipse.stp.soas.deploy.core.Utilities
				.adaptToIPackage(mDeployPackage);
	}

	public void init(DeployPackage deployPackage, IPackage iPackage,
			DeployServer server) {
		mDeployPackage = deployPackage;
		mRoot = mDeployPackage.getRoot();
		mSelectedServer = server;
		mPackage = iPackage;
	}

	public DeployServer getSelectedDeployServer() {
		return mSelectedServer;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.internal.deploy.ui.dialogs.DialogPage#performOk()
	 */
	public boolean performOk() {
		IServer selectedServer = (IServer) ((IStructuredSelection) mCPViewer
				.getViewer().getSelection()).getFirstElement();
		String selectedServerName = selectedServer.getName();
		String serverId = selectedServer.getId();
		
		mSelectedServer = null;
		for (Iterator it = mRoot.getServer().iterator(); mSelectedServer == null
				&& it.hasNext();) {
			DeployServer server = (DeployServer) it.next();
			if (selectedServerName.equals(server.getProfileName())) {
				mSelectedServer = server;
			}
		}
		if (mSelectedServer == null) {
			mSelectedServer = DeployFileFactory.eINSTANCE
					.createDeployServer(selectedServerName);
			mSelectedServer.setServerId(serverId);
		}
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		Composite content = new Composite(parent, SWT.NULL);
		content.setLayout(new GridLayout(1, false));
		content.setLayoutData(new GridData(GridData.FILL_BOTH));

		Tree tree = new Tree(content, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		mCPViewer = new ConnectionProfileTreeViewer(tree);
		mCPViewer.getViewer().setAutoExpandLevel(3);
		GridData data = new GridData(GridData.FILL_BOTH);
		data.heightHint = tree.getItemHeight() * MIN_LIST_HEIGHT;
		tree.setLayoutData(data);
		mCPViewer.getViewer().addSelectionChangedListener(
				new ISelectionChangedListener() {

					public void selectionChanged(SelectionChangedEvent event) {
						handleCPSelectionChanged();
					}
				});
		mCPViewer.getViewer().addDoubleClickListener(
				new IDoubleClickListener() {

					public void doubleClick(DoubleClickEvent event) {
						handleCPDoubleClick();
					}
				});
		mCPViewer.getViewer().addFilter(new ViewerFilter() {

			public boolean select(Viewer viewer, Object parentElement,
					Object element) {
				boolean retVal = false;
				if (element instanceof IServer) {
					retVal = true;
					String serverId = ((IServer) element)
							.getId();
					if (mSelectedServer == null
							|| !mSelectedServer.getServerId().equals(
									serverId)) {
						for (Iterator it = mDeployPackage
								.getTargetConfiguration().iterator(); retVal
								&& it.hasNext();) {
							DeployServer server = ((DeployConfiguration) it
									.next()).getTargetServer();
							retVal = server == null
									|| !server.getServerId().equals(
											serverId);
						}
					}
				}
				else if (element instanceof IServerType){
					
					retVal = DeploymentUtil.getServerMap().containsKey(element);
				}
				return retVal;
			}
		});
		if (mPackage != null) {
			if (mPackage instanceof IPhysicalPackage) {
				// Physical packages may be deployable to multiple server types
				IFile packageFile = mPackage.getFile();
				List packageExtensions = DeploymentExtensionManager
						.getInstance().getPhysicalPackageExtensions(
								mPackage.getFile().getFileExtension());
				final List packages = new ArrayList(packageExtensions.size());
				for (Iterator it = packageExtensions.iterator(); it.hasNext();) {
					IPhysicalPackageExtension ippe = (IPhysicalPackageExtension) it
							.next();
					if (ippe.supportsFile(packageFile)) {
						packages.add(ippe.adaptFile(packageFile));
					}
				}
			}
			else {
				mCPViewer.getViewer().addFilter(new ViewerFilter() {

					public boolean select(Viewer viewer, Object parentElement,
							Object element) {
						boolean retVal = true;
						if (element instanceof IServer) { 
							
							IServer server = (IServer)element;
							IVersion stpVersion = DeploymentUtil.getSTPVersionFromWTPServer(server);
							org.eclipse.stp.soas.deploy.core.IServerType serverType = new ServerType(server.getServerType().getId(), stpVersion);
							retVal = null != DeploymentExtensionManager.getInstance().getPackageConstructor((ILogicalPackage)mPackage, serverType);
							
						}
						else {
							retVal = true;
						}
						return retVal;
					}
				});
			}
		}

		initControls();

		setControl(content);
	}

	private List getServerTypeIdByLogicalPackage(ILogicalPackage pkg){
		
		List<String> serverTypes = new ArrayList<String>();
		List constructors = DeploymentExtensionManager.getInstance().getPackageConstructor(pkg);
		for(Iterator<IPackageConstructorExtension> it = constructors.iterator(); it.hasNext();){
			IPackageConstructorExtension pce = it.next(); 
			for(Iterator<ISupportedServerType> itSerType = pce.getSupportedServerType().iterator(); itSerType.hasNext();){
				serverTypes.add(itSerType.next().getServerDefinition().getID());
			}
		}
		return serverTypes;
		
	}
	
	private void initControls() {
		
		mCPViewer.getViewer().setInput(ServerCore.getServerTypes());
		updateControls();

		validate();

		setErrorMessage(null);
	}

	private void updateControls() {
	}

	private void validate() {
		validateExistingServer();
	}

	private void validateExistingServer() {
		if (mCPViewer.getViewer().getSelection().isEmpty()
				|| !(((IStructuredSelection) mCPViewer.getViewer()
						.getSelection()).getFirstElement() instanceof IServer)) {
			setErrorMessage(DeployCorePlugin
					.getDefault()
					.getResourceString(
							"ServerSelectionDialogPage.error.selectTargetServer")); //$NON-NLS-1$
			setIsValid(false);
		}
		else {
			setErrorMessage(null);
			setIsValid(true);
		}
	}

	private void handleCPDoubleClick() {
		updateControls();

		validate();

		if (isValid()) {
			getContainer().closeOK();
		}
	}

	private void handleCPSelectionChanged() {
		validateExistingServer();
	}

	static {
		TITLE = DeployCorePlugin.getDefault().getResourceString(
				"ServerSelectionDialogPage.title.selectTargetServer"); //$NON-NLS-1$
		ICON_PATH = "icons/blank_wizard_graphic.gif"; //$NON-NLS-1$
		URL url = null;
		try {
			url = DeployCorePlugin.getDefault().getBundle().getEntry(
					ICON_PATH);
		}
		catch (RuntimeException e) {
			e.printStackTrace();
		}
		if (url == null) {
			ICON_DESC = ImageDescriptor.getMissingImageDescriptor();
		}
		else {
			ICON_DESC = ImageDescriptor.createFromURL(url);
		}
		LABEL_ADD_SERVER = DeployCorePlugin.getDefault()
				.getResourceString("ServerSelectionDialogPage.LABEL.addServer"); //$NON-NLS-1$
		MIN_LIST_HEIGHT = 15;
		INDENT = 25;
	}

}