/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.api;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;

import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.ClassDefiner;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * 
 * @author amiguel
 *
 * The engine executable object.  Can be passed into an engine instance to be run.
 */
public class Program {
	
//
// Message types
//
private static int PROGRAM_START = 99;
private static int PROGRAM_END = 100;

private static int PROGRAM_CLASS = 101;
//V1 byte[] class file

private static int PROGRAM_HOST = 102;
//V1 String host

private static int PROGRAM_EXTRA_CLASS = 103;

//private ArrayList hosts = new ArrayList();
private ArrayList addresses = new ArrayList();
private ArrayList daemonaddresses = new ArrayList();
private byte[] the_class = null; 
private Class real_class = null;

private Object build_LOCK = new Object();

//private boolean load_extras = true;
private ArrayList extra_classes = new ArrayList();

private static StreamUtils su = new StreamUtils();

private JARDependency[] dependencies;

	/**
	 * Get the JAR dependencies which this Program requires
	 * @return an array of JAR dependencies which this Program requires
	 */
	public JARDependency[] getDependencies() {
		return dependencies;
	}

	private Program() {
	}
	
	public Program (byte[] main_class, byte[][] classes, JARDependency[] deps) {
		the_class = main_class;
		dependencies = deps;
		for (int i = 0; i < classes.length; i++) {
			extra_classes.add(classes[i]);	
		}
	}

	private void loadExtras(ClassDefiner definer) throws Exception {
		ArrayList left = new ArrayList();
		left.addAll(extra_classes);
		
		//still classes to load
		while (left.size() > 0) {
			StringBuffer errors = new StringBuffer();
			boolean cannot_ignore = false;
			
			int lsize = left.size();
			for (int i = 0; i < left.size(); i++) {
				byte[] dat = (byte[])left.get(i);
				try {
					definer.define(dat,0,dat.length);
//					if (loader == null) {
//						ClassDefiner.defineSystem(dat,0,dat.length);
//					} else {
//						ClassDefiner.defineSpecific(dat,0,dat.length,loader);
//					}
					left.remove(i--);
				} catch (LinkageError e) {
					errors.append("\nIGNORED LINKAGE ERROR: "+e);
//					Logger.warning("ignored linkage error "+e);
				} catch (Throwable e) {
					cannot_ignore = true;
					errors.append("\n"+e);
//					Logger.warning("failed attempt to load class",e);
				}
			}
			//no new loaded classes
			if (lsize == left.size()) {
				if (cannot_ignore) {
					throw new Exception(left.size()+" extra classes failed to load - "+errors);
				} else {
					if (errors.length() > 0) {
						Logger.warning("Ignored the following errors:\n"+errors);
					}
					left.clear();
				}
			}
		}
		
		//TODO I could do extra_classes=new ArrayList() here which would allow the now loaded classes to be garbage collected?
		
	}

	private Class build(byte[] class_bytes, ClassDefiner definer) throws Exception {
		return definer.define(class_bytes,0,class_bytes.length);
	}
	
	public void addHostAddress(SessionAddress address, SessionAddress daemonaddress) {
		addresses.add(address);
		daemonaddresses.add(daemonaddress);
	}	
	
	public int getAddressCount() {
		return addresses.size();	
	}
	
	public SessionAddress getHostAddress(int index) {
		return (SessionAddress)addresses.get(index);	
	}
	public SessionAddress getDaemonAddress(int index) {
		return (SessionAddress)daemonaddresses.get(index);	
	}

	public Class getProgramClass(URL[] dependencies) throws Exception {
		synchronized(build_LOCK) {
			if (real_class == null) {
				ClassDefiner definer = new ClassDefiner(dependencies);
				
				real_class = build(the_class,definer);
//				if (load_extras) {
					loadExtras(definer);
//				}

/*					
				real_class = build(the_class);
//				if (load_extras) {
					loadExtras(null);
//				}
 */
			}
		}
		return real_class;	
	}

	public Class getProgramClass(URL[] dependencies, ClassLoader loader) throws Exception {
		synchronized(build_LOCK) {
			if (real_class == null) {
				ClassDefiner definer = new ClassDefiner(dependencies,loader);
				
				real_class = build(the_class,definer);
//				if (load_extras) {
					loadExtras(definer);
//				}
			}
		}
		return real_class;
	}

	public ArrayList getAddresses() {
		return addresses;
	}
	public ArrayList getDaemonAddresses() {
		return daemonaddresses;
	}
//	public ArrayList getHosts() {
//		return hosts;	
//	}

	private String strip(String source) {
		
		StringBuffer prog = new StringBuffer();
		
		int n = 0;
		while (true) {
			n = source.indexOf("//@@START_STRIP",n);
			if (n == -1) {
				//no more tokens, append the rest of the string and quit
				prog.append(source);
				break;
			}	
			int n_end = source.indexOf("//@@END_STRIP",n);	
			
			prog.append(source.substring(0,n));
			source = source.substring(n_end);
		}

		return prog.toString();
	}
/*
	private byte[] compile(String source, ClassLoader classLoader) throws Exception {

		source = strip(source);

		byte[] c = ScapaCompiler.compileToBytes(source,classLoader);
		
		return c;
	}
*/	
	private String readAll(InputStream source) throws Exception {
		byte[] buf = new byte[1024];
		StringBuffer sbuf = new StringBuffer();
		int n = 0;
		
		while (n != -1) {
			n = source.read(buf,0,buf.length);
			if (n > 0) {
				sbuf.append(new String(buf,0,n));
			}
		}
		
		return sbuf.toString();		
	}

	public byte[] writeProgram() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		writeProgram(bout);
		return bout.toByteArray();
	}

	public void writeProgram(OutputStream out) throws IOException {
		try {
			Message m;
			
			m = new Message(PROGRAM_START);
			MessageUtils.writeMessage(out,m);
	
			m = new Message(PROGRAM_CLASS);
			m.append(the_class);
			MessageUtils.writeMessage(out,m);
			
			for (int i = 0; i < extra_classes.size(); i++) {
				m = new Message(PROGRAM_EXTRA_CLASS);
				m.append((byte[])extra_classes.get(i));
				MessageUtils.writeMessage(out,m);	
			}
			
			for (int i = 0; i < addresses.size(); i++) {
				m = new Message(PROGRAM_HOST);
				m.append(SessionAddress.toString((SessionAddress)addresses.get(i)));
				m.append(SessionAddress.toString((SessionAddress)daemonaddresses.get(i)));
				MessageUtils.writeMessage(out,m);
			}
	//		for (int i = 0; i < hosts.size(); i++) {
	//			m = new Message(PROGRAM_HOST);
	//			m.append((String)hosts.get(i));
	//			MessageUtils.writeMessage(out,m);
	//		}
			
			m = new Message(PROGRAM_END);
			MessageUtils.writeMessage(out,m);
		} catch (IOException e) {
			throw e;
		} catch (Exception e) {
			throw new IOException(""+e);
		}
		
	}

	public static Program readProgram(byte[] in) throws IOException {
		return readProgram(new ByteArrayInputStream(in));
	}

	public static Program readProgram(InputStream in) throws IOException {
		try {
			
			Program p = new Program();
	
			Message m = MessageUtils.readMessage(in);
			if (m.getType() != PROGRAM_START) {
				throw formatE("did not find program start message");	
			}
	
			m = MessageUtils.readMessage(in);
			while (m.getType() != PROGRAM_END) {
	
				int t = m.getType();
	
				//
				// Check message type here
				//
				// Everything here is done in chunks, if a chunk isnt recognised then it
				// could either be ignored or an exception could be thrown
				//
	
				if (t == PROGRAM_CLASS) {
					p.the_class = (byte[])m.get(0);
	//				p.real_class = p.build(p.the_class);				
	
				} else if (t == PROGRAM_EXTRA_CLASS) {
					p.extra_classes.add((byte[])m.get(0));
					
				} else if (t == PROGRAM_HOST) {
					SessionAddress host = SessionAddress.fromString((String)m.get(0));
					SessionAddress daemon = null;
					if (m.length() > 0) {
						//V2
						daemon = SessionAddress.fromString((String)m.get(1));
					} else {
						daemon = (SessionAddress)host.clone();
						daemon.setRequiresEncryption(false);
						daemon.setPassword(null);
					}
					p.addHostAddress(host,daemon);
	//			} else if (t == PROGRAM_HOST) {
	//				p.addHost( (String)m.get(0) );
	
				} else {
					Logger.error("PROGRAM: unrecognised message type "+m);
					
				}
					
				m = MessageUtils.readMessage(in);
			}
			/*
			try {
				p.loadExtras();
			} catch (Exception e) {
				throw formatE(e);
			}
	*/			
			return p;
		
		} catch (IOException e) {
			throw e;
		} catch (Exception e) {
			throw new IOException(""+e);
		}
	}

	private static IOException formatE(Object o) {
			return new IOException("Program:"+o.toString());
	}

	private static void debug(Object o) {
		Logger.info("Program:"+o.toString());	
	}

}