/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Stack;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.bpel.BPELTranslatorException;
import org.w3c.dom.Node;

/**
 * 
 * @author amiguel
 *
 * Used to manage scopes when translating BPEL source files into engine Progams.
 */
public class ScopeStack {
//
// Scope stuff
//
class Scope {
	String name = "SCOPE_ERROR";
	boolean isolated;
	HashMap variables = new HashMap();
	HashMap links = new HashMap();
	HashMap correlationSets = new HashMap();
	public String toString() {
		return name;
	}
	
	HashMap isolatedVariables = new HashMap();
	
	int logicalStart;
	int logicalEnd;
}

int SCOPEID = 0;

Util compiler_util;

Stack scopes = new Stack();

int LOGICAL = 1;

	public ScopeStack(Util compiler_util) {
		this.compiler_util = compiler_util;
	}

	public void pushScope(String idFetchString, boolean isolated, Node source) throws BPELTranslatorException {
		pushScope(idFetchString,isolated,source,false);
	}
	public void pushScope(String idFetchString, boolean isolated, Node source, boolean ignoreParentIsolation) throws BPELTranslatorException {
		Scope scope = new Scope();
		scope.name = "\"+"+idFetchString+"+\"S"+SCOPEID++;
		scope.isolated = isolated;
		scope.logicalStart = LOGICAL++;
		
		if (isolated && !ignoreParentIsolation) {
			for (int i = 0; i < scopes.size() ;i++) {
				Scope tmp = (Scope)scopes.get(i);
				if (tmp.isolated) {
					throw new BPELTranslatorException(compiler_util,source,"Cannot nest isolated scopes (this scope has a parent which is also isolated)");
				}
			}
		}
		
		scopes.push(scope);
	}
	
	public void popScope(Node source) throws BPELTranslatorException {
		if (scopes.isEmpty()) throw new BPELTranslatorException(compiler_util,source,"Tried to pop scope, but was already at maximum");
		Scope tmp = (Scope)scopes.pop();
		tmp.logicalEnd = LOGICAL++;
	}
	
	public int getScopeLogicalStart() throws BPELTranslatorException {
		return ((Scope)scopes.peek()).logicalStart;
	}
	public int getScopeLogicalEnd() throws BPELTranslatorException {
		return LOGICAL;
	}
	
//	public String getScope() throws BPELException {
//		if (scopes.isEmpty()) throw new BPELException("Tried to get scope, but no scope was found");
//		return scopes.peek().toString();
//	}

	public String[] getIsolatedVariableNames(Node source) throws BPELTranslatorException {
		if (scopes.isEmpty()) throw new BPELTranslatorException(compiler_util,source,"Tried to get isolated variable names for scope but was already at maximum");
		Scope scope = (Scope)scopes.peek();
		ArrayList names = new ArrayList(scope.isolatedVariables.keySet());
		String[] array = new String[names.size()];
		names.toArray(array);
		return array;
	}
	
	public void addVariable(String name, String qtype) {
		Scope scope = (Scope)scopes.peek();
		scope.variables.put(name,qtype);
	}

	public String getVariableName(String name, Node source) throws BPELTranslatorException {
		Scope owner = null;
//System.out.println("GET VARIABLE NAME "+name);		
		for (int i = scopes.size()-1; i >= 0; i--) {
			Scope scope = (Scope)scopes.get(i);
//			if (owner == null) {
				if (scope.variables.get(name) != null) {
					owner = scope;
					break;
				} else {
					if (scope.isolated) {
						//this sub-scope is isolated, it must lock all variables it accesses
						scope.isolatedVariables.put(name,name);
					}
				}
//			} else {
//			}
		}

		if (owner != null) return owner.name+":"+name;

		throw new BPELTranslatorException(compiler_util,source,"variable \""+name+"\" not found in any enclosing scope");
	}
	
	public String getVariableQtype(String name, Node source) throws BPELTranslatorException {
		Object qtype = null;
		
		for (int i = scopes.size()-1; i >= 0; i--) {
			Scope scope = (Scope)scopes.get(i);
			if (qtype == null) {
				qtype = scope.variables.get(name);
				
				if (qtype != null) {
					if (scope.isolated) {
						//this sub-scope is isolated, it must lock all variables it accesses
						scope.isolatedVariables.put(name,name);
					}
					break;
				}
			}
		}

		if (qtype != null) return (String)qtype;

		throw new BPELTranslatorException(compiler_util,source,"variable \""+name+"\" not found in any enclosing scope");
	}

	public void addCorrelationSet(String name, String[] qproperties) {
		Scope scope = (Scope)scopes.peek();
		scope.correlationSets.put(name,qproperties);
	}
	public String getCorrelationSetName(String name, Node source) throws BPELTranslatorException {
		for (int i = scopes.size()-1; i >= 0; i--) {
			Scope scope = (Scope)scopes.get(i);
			Object test = scope.correlationSets.get(name);
			if (test != null) return scope.name+":"+name;
		}
		throw new BPELTranslatorException(compiler_util,source,"correlation set \""+name+"\" not found in any enclosing scope");
	}
	public String[] getCorrelationSetProperties(String name, Node source) throws BPELTranslatorException {
		for (int i = scopes.size()-1; i >= 0; i--) {
			Scope scope = (Scope)scopes.get(i);
			Object properties = scope.correlationSets.get(name);
			if (properties != null) return (String[])properties;
		}
		throw new BPELTranslatorException(compiler_util,source,"correlation set \""+name+"\" not found in any enclosing scope");
	}
	
	public void addLink(String name) {
		Scope scope = (Scope)scopes.peek();
		scope.links.put(name,name);
	}

	public String getLinkName(String name, Node source) throws BPELTranslatorException {
		for (int i = scopes.size()-1; i >= 0; i--) {
			Scope scope = (Scope)scopes.get(i);
			Object test = scope.links.get(name);
			if (test != null) return scope.name+":"+name;
		}
	for (int i = 0; i < scopes.size(); i++) {
		System.out.println(scopes.get(i));
		Scope scope = (Scope)scopes.get(i);
		Iterator it = scope.links.keySet().iterator();
		while (it.hasNext()) {
			String key = (String)it.next();
			System.out.println("  "+key+" = "+scope.links.get(key));
		}
	}
		throw new BPELTranslatorException(compiler_util,source,"link \""+name+"\" not found in any enclosing flow");
	}
}