/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Stack;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.ScopeStack;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.xpath.internal.XPathParser;
import org.eclipse.stp.b2j.core.xpath.internal.XPathVisitor;
import org.w3c.dom.Node;

/**
 * 
 * @author amiguel
 *
 * Builds each XPATH expression into a tree of Nodes that can then be
 * more easily operated on and reduced for optimisation.
 */
public class XPATHTreeBuilder extends XPathVisitor {

	NamespaceTranslator nt;
	
	public XPATHTreeBuilder(NamespaceTranslator nt) {
		this.nt = nt;
	}
	
	public static void main(String[] args) {
		
		ScopeStack sstack = new ScopeStack(null);
//		HashMap vtoq = new HashMap();
		
		String[] expressions = new String[] {
			"1 + 4 + 5 + (14 div 7)",
			"5 + (14 div 7)",
			"14 div 7",
			//how do we perform against this?
			//"4 + 5",
			"1 + 4",
			"-1000",
			"(100 + 7) > (100 + 5)",
			"momally($myvar)",
			"$myvar/one/two[777]",
			"bpws:getVariableData(\"myvar\",\"one\",\"two[777]\")",
			"bpws:getVariableData(\"myvar\",\"one\")",
			"bpws:getVariableData(\"myvar\")",
		};
		
		try {
		sstack.pushScope("SCOPE",false,null);
		} catch (Exception e) {
		e.printStackTrace();
		}
		sstack.addVariable("myvar","VARTYPE");
//		vtoq.put("myvar","VARTYPE");
/*
		try {
			for (int i = 0; i < expressions.length; i++) {
				XPATHTreeTranslator res = new XPATHTreeTranslator();
				String resolved = res.resolveToAny("METHODNAME",vtoq,expressions[i],sstack);
				System.out.println("EXPRESSION "+expressions[i]+":");
				System.out.println(resolved);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
*/

		try {
			XPATHTreeBuilder res = new XPATHTreeBuilder(new NamespaceTranslator(""));
			for (int i = 0; i < expressions.length; i++) {
				res.resolveToTree(expressions[i],sstack,null);
			}
			XPathNode roots = res.roots;
			
			System.out.println("Original Expressions:");
			printRoots(roots);

			XPATHTreePrecomputer precomp = new XPATHTreePrecomputer();
			precomp.precompute(roots);

			System.out.println("Precomputed Expressions:");
			printRoots(roots);

			
		} catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	public XPathNode getRootExpressionsContainer() {
		return roots;
	}
	
	private static void printRoots(XPathNode roots) {
		for (int i = 0; i < roots.getChildCount(); i++) {
			System.out.println(roots.getChild(i));
		}
	}
	
	XPathNode roots = new XPathNode(XPathNode.TYPE_ROOTS_CONTAINER);
	
	Stack nodestack = new Stack();
	
	ArrayList exceptions = new ArrayList();
	
	private void checkForExceptions(String expression, Node source) throws Exception {
		for (int i = 0; i < exceptions.size(); i++) {
			((Exception)exceptions.get(i)).printStackTrace();
			throw XPATHTranslatorException.parserError((Exception)exceptions.get(i),expression,source);
		}
	}
	private void error(String s) {
		exceptions.add(new Exception(s));
	}
	
	public void resolveToTree(String expression, ScopeStack scopes, Node source) throws Exception {
		try {
		
			XPathParser parser = new XPathParser(new StringReader(expression));
			parser.XPath(this);
	
			checkForExceptions(expression,source);
			
			XPathNode root = new XPathNode(XPathNode.TYPE_ROOT_EXPRESSION);
			while(!nodestack.isEmpty()) {
				root.insertChild((XPathNode)nodestack.pop());
			}
			roots.addChild(root);
		} catch (Exception e) {
			//if we have an all out failure then check for any nice descriptive parser errors
			//before throwing the uncaught one
			checkForExceptions(expression,source);
			throw XPATHTranslatorException.parserError(e,expression,source);
		}
	}

	public void prePredicateEval() {
		XPathNode node = new XPathNode(XPathNode.TYPE_PREDICATE_START);

		nodestack.push(node);
	}

	public void postPredicateEval() {
		XPathNode node = new XPathNode(XPathNode.TYPE_PREDICATE);
		
		XPathNode arg = (XPathNode)nodestack.pop();
		while (arg.getType()  != XPathNode.TYPE_PREDICATE_START) {
			node.insertChild(arg);
			arg = (XPathNode)nodestack.pop();
		}
		
		nodestack.push(node);
	}

	public void processNumber() {
		String number = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_NUMBER);
		node.addArg(number);

		nodestack.push(node);
	}

	public void processLiteral() {
		String literal = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_LITERAL);
		node.addArg(literal);

		nodestack.push(node);
	}

	public void processSlash() {}

	public void processAbsoluteSlash() {}

	public void processAbbreviatedAbsoluteSlash() {}

	public void processAbbreviatedRelativeSlash() {}

	public void setContextNode() {}

	public void evaluateAdditiveExpr() {
		String operator = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_ADDITIVE);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());

		nodestack.push(node);
	}

	public void evaluateMultiplicativeExpr() {
		String operator = String.valueOf(pop());
		
		XPathNode node = new XPathNode(XPathNode.TYPE_MULTIPLICATIVE);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());

		nodestack.push(node);
	}

	public void evaluateUnaryExpr() {
		String operator = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_UNARY);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());

		nodestack.push(node);
	}

	public void evaluateRelationalExpr() {
		String operator = String.valueOf(pop());
		
		XPathNode node = new XPathNode(XPathNode.TYPE_RELATIONAL);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());

		nodestack.push(node);
	}

	public void evaluateEqualityExpr() {
		String operator = String.valueOf(pop());
		
		XPathNode node = new XPathNode(XPathNode.TYPE_EQUALITY);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());
		
		nodestack.push(node);
	}

	public void evaluateAndExpr() {
		String operator = String.valueOf(pop());
		
		XPathNode node = new XPathNode(XPathNode.TYPE_ANDEXPR);
		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());

		nodestack.push(node);
	}

	public void evaluateOrExpr() {
		String operator = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_OREXPR);

		node.addArg(operator);
		node.insertChild((XPathNode)nodestack.pop());
		node.insertChild((XPathNode)nodestack.pop());
		
		nodestack.push(node);
	}

	public void evaluateRelativePathPattern() {}

	public void evaluateRelativeLocationPath() {
	}

	public void evaluateStepPredicate() {}

	public void evaluateQName() {}

	public void evaluateQNameWONodeType() {}

	public void evaluatePattern() {}

	public void evaluateSingleIDKeyPattern() {}

	public void evaluateDoubleIDKeyPattern() {}

	public void evaluateVariableReference() {
		String tovariable = ""+pop();
		String var = ""+pop();
//		String qtype = (String)variable_to_qtype.get(tovariable);

//		if (qtype == null) error("XPath: could not find variable reference "+tovariable);

		XPathNode node = new XPathNode(XPathNode.TYPE_VARIABLEREF);
		
		node.addArg(tovariable);
//		node.addArg(qtype);

		nodestack.push(node);
	}

	public void evaluateWildCardQName() {}

	public void evaluateWildCardStar() {}

	public void evaluateWildCardNCName() {}

	public void evaluatePrimaryFilterExpr() {}
	
//	public void evaluatePredicateFilterExpr() {}

	public void evaluatePathExpr() {}
	
	public void evaluateUnionExpr() {}

	public void evaluateAbsoluteLocationPath() {}

	public void evaluateProcessingInstruction() {}

	public void evaluateAbbreviatedAbsoluteLocationPath() {}

	public void evaluateAbbreviatedRelativeLocationPath() {}

	public void evaluateLocationIDPathPattern() {}

	public void evaluateLocationRelPathPattern() {}

	public void evaluateLocationPathPattern() {}

	public void evaluateFunctionCall() {
		String closeB = String.valueOf(pop());

		int args = Integer.parseInt(""+pop());

		String openB = String.valueOf(pop());
		
		//TODO are we doing this because of a problem in the parser?
		String function;
		if (openB.equals("(")) {
			function = String.valueOf(pop());
		} else {
			function = openB;
		}

		XPathNode node = new XPathNode(XPathNode.TYPE_FUNCTION_CALL);

		node.addArg(function);
		for (int i = 0; i < args; i++) {
			node.insertChild((XPathNode)nodestack.pop());
		}

		if (function.equals("getVariableData")) {
			
			error("function getVariableData(...) has been removed from the BPEL specification, instead use $variablename");
			
			/*
			//we convert this into a non-variable expression and parse it
			
			StringBuffer subexpression = new StringBuffer();
			
			if (node.getChildCount() == 0) error("getVariableData function must have at least 1 argument");
			if (node.getChildCount() > 3) error("getVariableData function can have at most 3 arguments");
			
			for (int k = 0; k < node.getChildCount(); k++) {
				if (node.getChild(k).getType() != XPathNode.TYPE_LITERAL) {
					error("all arguments to bpel getVariableData function must be string literals");
				}
			}
			
			//variable reference
			subexpression.append("$");
			subexpression.append(removeQuotes((String)node.getChild(0).getArg(0)));
			if (node.getChildCount() > 1) {
				//part
				subexpression.append("/");
				subexpression.append(removeQuotes((String)node.getChild(1).getArg(0)));
				if (node.getChildCount() > 2) {
					//part query
					subexpression.append("/");
					subexpression.append(removeQuotes((String)node.getChild(2).getArg(0)));
				}
			}
			
			XPathParser parser = new XPathParser(new StringReader(subexpression.toString()));
			try {
				parser.XPath(this);
			} catch (Exception e) {
				error(getStackTrace(e));
			}
			*/

		} else if (function.equals("getVariableProperty")) {

			if (node.getChildCount() < 2) error("getVariableProperty function must have exactly 2 arguments");
			
			for (int k = 0; k < node.getChildCount(); k++) {
				if (node.getChild(k).getType() != XPathNode.TYPE_LITERAL) {
					error("all arguments to bpel getVariableProperty function must be string literals");
				}
			}
			
			String varname = String.valueOf(node.getChild(0).getArg(0));
			String propname = String.valueOf(node.getChild(1).getArg(0));
			
			varname = removeQuotes(varname);
			propname = removeQuotes(propname);
			
			try {
				propname = nt.qualify(propname,false);
			} catch (NamespaceException e) {
				error("Failed to translate namespace of property "+propname);
			}

			//
			// parse out the variable reference
			//
			XPathParser parser = new XPathParser(new StringReader("$"+varname));
			try {
				parser.XPath(this);
			} catch (Exception e) {
				error(getStackTrace(e));
			}
			
			//then add a property
			node.removeChild(node.getChild(0));
			node.removeChild(node.getChild(0));

			node.addArg(varname);
			node.addArg(propname);
			
			nodestack.push(node);
			
		} else if (function.equals("getLinkStatus")) {
			error("function getLinkStatus(...) has been removed from the BPEL specification, instead use $linkname");

			/*
			//we convert this into a non-variable expression and parse it
			
			if (node.getChildCount() < 1) {
				error("function getLinkStatus must have exactly one argument");
			}
			
			XPathNode argnode = node.getChild(0);
			if (argnode.getType() != XPathNode.TYPE_LITERAL) {
				error("function getLinkStatus must take a constant literal string as it's argument");
			}
			
			String linkName = String.valueOf(argnode.getArg(0));
			
			if (linkName.length() > 1) {
				if ((linkName.startsWith("\"") && linkName.endsWith("\""))
					|| (linkName.startsWith("'") && linkName.endsWith("'"))) {
					
					linkName = linkName.substring(1);
					linkName = linkName.substring(0,linkName.length()-1);
				}
			}
			
			node.addArg(linkName);
			nodestack.push(node);
			*/
			
		} else {
			nodestack.push(node);
		}
	}

	public void evaluateNodeTypeTest() {
		
		String closeB = String.valueOf(pop());
		String arg = String.valueOf(pop());
		if (arg.equals("(")) {
			arg = "";
		} else {
			pop(); //pop the (
		}
		
		String function = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_NODE_TYPE_TEST);

		node.addArg(function);

		nodestack.push(node);
	}
	
	public void evaluateNameTest() {
		String name = String.valueOf(pop());

		XPathNode node = new XPathNode(XPathNode.TYPE_NAME_TEST);

		node.addArg(name);

		nodestack.push(node);
	}
	
	public void evaluateAxisSpecifier() {
		String axis = String.valueOf(peek());
		
		if (axis.equals("/")) {
			axis = "child::";
		}
		
		if (!axis.equals("null") && axis.endsWith("::")) {
			pop();//pop the axis
			pop();//pop the /

			axis = axis.substring(0,axis.length()-2);

			XPathNode node = new XPathNode(XPathNode.TYPE_AXIS_SPECIFIER);
	
			node.addArg(axis);
	
			nodestack.push(node);
		}
	}
	
	private String removeQuotes(String s) {
		if (	s.length() > 1
				&& 
				(
					(s.startsWith("\"") && s.endsWith("\""))
					|| (s.startsWith("'") && s.endsWith("'"))
				)
			) {
			return s.substring(1,s.length()-1);
		}
		return s;
	}
	public String getStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method

}