/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.core.bpel;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.core.RunnerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.internal.XMLAccessible;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.Correlatable;
import org.eclipse.stp.b2j.core.jengine.internal.utils.XPathAccessible;

/**
 * 
 * @author amiguel
 * 
 * Part of the generic runtime engine API, used to help implement BPEL correlation sets.
 */
public class BPELCorrelationSet implements Comparator {
	
//	static Object LOCK = new Object();
//	static HashMap sets = new HashMap();
	private static final String MAPNAME = "BPEL_CORRELATION_SETS";


	public static BPELCorrelationSet getOrCreate(RunnerInterface runner, String name, XMLAccessible builder, BPELCorrelationToken[] tokens) throws Exception {
		HashMap sets = runner.getEngineLocalStorageMap(MAPNAME);
		synchronized(sets) {
			BPELCorrelationSet set = (BPELCorrelationSet)sets.get(name);
			if (set == null) {
				set = new BPELCorrelationSet(builder,tokens);
				sets.put(name,set);
			}
			return set;
		}
	}
	public static BPELCorrelationSet getOrError(RunnerInterface runner, String name) throws Exception {
		HashMap sets = runner.getEngineLocalStorageMap(MAPNAME);
		synchronized(sets) {
			BPELCorrelationSet set = (BPELCorrelationSet)sets.get(name);
			if (set == null) {
				throw new Exception("failed to get (and not create) correlation set");
			}
			return set;
		}
	}
	public static BPELCorrelationSet createOrError(RunnerInterface runner, String name, XMLAccessible builder, BPELCorrelationToken[] tokens) throws Exception {
		HashMap sets = runner.getEngineLocalStorageMap(MAPNAME);
		synchronized(sets) {
			BPELCorrelationSet set = (BPELCorrelationSet)sets.get(name);
			if (set != null) {
				throw new Exception("failed to create (and not get) correlation set");
			}

			set = new BPELCorrelationSet(builder,tokens);
			sets.put(name,set);
			
			return set;
		}
	}
	public static void setUninitialised(RunnerInterface runner, String name) throws Exception {
		HashMap sets = runner.getEngineLocalStorageMap(MAPNAME);
		synchronized(sets) {
			sets.remove(name);
		}
	}
	
	XMLAccessible builder;
	BPELCorrelationToken[] tokens;
	private BPELCorrelationSet(XMLAccessible builder, BPELCorrelationToken[] tokens) {
		this.builder = builder;
		this.tokens = tokens;
		
		//sort, so that all correlation sets that are the same produce the same string
		Arrays.sort(tokens,this);
	}
	
	public XMLAccessible getObjectBuilder() {
		return builder;
	}
	
	public void merge(BPELCorrelationSet set) {
		BPELCorrelationToken[] ntokens = new BPELCorrelationToken[tokens.length + set.tokens.length];
		System.arraycopy(tokens,0,ntokens,0,tokens.length);
		System.arraycopy(set.tokens,0,ntokens,tokens.length,set.tokens.length);
		tokens = ntokens;
		Arrays.sort(tokens,this);
	}
	
	public String toString() {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < tokens.length; i++) {
			tokens[i].toString(sb);
		}
		return sb.toString();
	}
	
	public int compare(Object o1, Object o2) {
		BPELCorrelationToken t1 = (BPELCorrelationToken)o1;
		BPELCorrelationToken t2 = (BPELCorrelationToken)o2;
		
		return t1.getName().compareTo(t2.getName());
	}
	
	public boolean matches(XPathAccessible match) throws Exception {
		for (int i = 0; i < tokens.length; i++) {
			if (!tokens[i].matches(match)) {
				return false;
			}
		}
		return true;
	}
}