/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketException;
import java.net.URL;

import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPFactory;
import org.eclipse.stp.b2j.core.jengine.internal.utils.CharStack;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;

/**
 * 
 * @author amiguel
 *
 * An HTTP client implementation used in the SOAP binding
 */
public class HTTPClient {

//	public static final int TIMEOUT = 60000;	//time out after 60 seconds
	public static final int TIMEOUT = 120000;	//time out after 2 minutes
//	public static final int TIMEOUT = 1800000;	//time out after 30 minutes
	
	private static final int RETRIES = 1;
	
	URL address;
	Socket sock;
	OutputStream sout;
	InputStream sin;
	
	String host;
	int port;
	String hostandport;
	String protocol;
	String resource;
	String fullurl;
	
	boolean secureprotocol = false;
	
	String version = HTTPUtils.VERSION_1_1;

	String proxy_host;
	int proxy_port;
	
	public HTTPClient(URL address) throws IOException {
		System.out.println("************* NEW CLIENT");		
		if (SOAPFactory.DEBUG) System.out.println("HTTP CLIENT ADDRESS "+address);		
		this.address = address;
		
		host = address.getHost();
		port = address.getPort();
		if (port == -1) {
			try {
				port = address.getDefaultPort();
			} catch (NoSuchMethodError t) {
				if (address.getProtocol().equalsIgnoreCase("http")) {
					port = 80;
				} else if (address.getProtocol().equalsIgnoreCase("https")) {
					port = 443;
				} else {
					port = 80;
				}
			}
		}
		
		hostandport = host+":"+port;
		protocol = address.getProtocol();
		resource = address.getFile();
		fullurl = address.toString();
		
		if (resource == null) resource = "/";
		if (resource.length() == 0) resource = "/";
		
//		connect();
	}
	
	private void setResource(String res) {
	    resource = res;

		if (resource == null) resource = "/";
		if (resource.length() == 0) resource = "/";
		
		if (SOAPFactory.DEBUG) System.out.println("************* SET RESOUCE "+resource);
	}
	public void setResource(URL url) {
	    setResource(url.getFile());
	    fullurl = url.toString();
	}

	private void connect() throws IOException {
		System.out.println("************* HTTP CLIENT CONNECT");		
		String connect_host = null;
		int connect_port;
		if (proxy_host != null) {
			connect_host = proxy_host;
			connect_port = proxy_port;
		} else {
			connect_host = address.getHost();
			connect_port = port;
		}
		
		if (protocol.equalsIgnoreCase("http")) {
			secureprotocol = false;
			
			sock = new Socket(connect_host,connect_port);
		} else if (protocol.equalsIgnoreCase("https")) {
			secureprotocol = true;

			SSLSocket sslsock = (SSLSocket)SSLSocketFactory.getDefault().createSocket(connect_host,connect_port);
			
			String[] suites = sslsock.getSupportedCipherSuites();
			sslsock.setEnabledCipherSuites(suites);
			
			sslsock.setNeedClientAuth(false);
			sslsock.setWantClientAuth(false);
			
			sock = sslsock;
		}
		
		sock.setSoTimeout(TIMEOUT);
		
		sout = new BufferedOutputStream(sock.getOutputStream());
		sin = new BufferedInputStream(sock.getInputStream());
	}

	public void setProxy(String host, int port) {
		boolean reconnect = false;
		
		if (host == null) {
			if (proxy_host != null) {
				reconnect = true;
				proxy_host = null;
			}
		} else {
			if (proxy_host == null) {
				reconnect = true;
				proxy_host = host;
			} else if (!proxy_host.equals(host)) {
				reconnect = true;
				proxy_host = host;
			}
		}
		
		if (proxy_port != port) {
			reconnect = true;
			proxy_port = port;
		}
		
		if (reconnect) {
			try {
				close();
			} catch (IOException e) {}
		}
	}
	
	public void close() throws IOException {
		System.out.println("************* CLIENT CLOSED");		
		if (sock != null) {
			sock.close();
			sock = null;
		}
	}
	
	private void writeMandatoryHeaders(String mimetype) throws IOException {
		if (version == HTTPUtils.VERSION_1_0) {
			
		} else { //if (version == HTTPUtils.VERSION_1_1) {
			HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_HOST,hostandport);
			if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_HOST+"="+hostandport);
		}
		if (mimetype != null) {
			HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_CONTENT_TYPE,mimetype);
			if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_CONTENT_TYPE+"="+mimetype);
		}
	}
	
	private void writeHeaders(String[][] headers) throws IOException {
		for (int i = 0; i < headers.length; i++) {
			HTTPUtils.writeHeader(sout,headers[i][0],headers[i][1]);
			if (SOAPFactory.DEBUG) System.out.println("Write Header:"+headers[i][0]+"="+headers[i][1]);
		}
	}
	
	private void writeCookieHeaders(CookieCache cookies) throws IOException {
		if (cookies != null) {
			String cookieValue = cookies.getCookies(host,resource,secureprotocol);
			if (cookieValue.length() > 0) {
				HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_COOKIE,cookieValue);
				if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_COOKIE+"="+cookieValue);
			}
		}
	}
	
	private void writeContent(byte[] dat) throws IOException {
		if (dat != null) {
			if (dat.length != 0) {
				HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_CONTENT_LENGTH,Integer.toString(dat.length));
				if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_CONTENT_LENGTH+"="+Integer.toString(dat.length));
			} else {
				dat = null;
				HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_CONTENT_LENGTH,"0");
				if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_CONTENT_LENGTH+"="+0);
			}
		} else {
			HTTPUtils.writeHeader(sout,HTTPUtils.HEADER_CONTENT_LENGTH,"0");
			if (SOAPFactory.DEBUG) System.out.println("Write Header:"+HTTPUtils.HEADER_CONTENT_LENGTH+"="+0);
		}

		HTTPUtils.writeHeadersEnd(sout);
		
		if (dat != null) {
			HTTPUtils.writeContent(sout,dat);
			if (SOAPFactory.DEBUG) System.out.println("Write Content:"+new String(dat));
		}
	}
	
	private String readResponse(CookieCache cookies) throws IOException {
		
		boolean finished = false;
		String res_code = null;
		
		while (!finished) {
			
			//read an initial response line
			CharStack stack = new CharStack(new String(StreamUtils.readLine(sin)));
			stack.popWhitespace();
			String res_version = stack.popText(true);
			stack.popWhitespace();
			res_code = stack.popText(true);

			if (res_code.equals("100")) {
				if (SOAPFactory.DEBUG) System.out.println("Response: 100 Continue");					
				//100 Continue
				//ignore it and skipt to the next response
				readContent(cookies);
				
			} else {
				if (SOAPFactory.DEBUG) System.out.println("Response: "+res_code);					
				finished = true;
			}
		}
		
		return res_code;
	}
	
	private byte[] readContent(CookieCache cookies) throws IOException {
		String res_content_type = null;
		int res_content_len = 0;
		
		boolean closeConnection = false;
		
		boolean chunked = false;
		
		String header = new String(StreamUtils.readLine(sin));
		while (header.trim().length() > 0) {
			int index = header.indexOf(':');
			if (index >= 0) {
			
				String key = header.substring(0,index).trim().toLowerCase();
				String value = header.substring(index+1);
				
				if (SOAPFactory.DEBUG) System.out.println("Header: "+key+"="+value);					

				if (key.equals(HTTPUtils.HEADER_CONTENT_LENGTH)) {
					res_content_len = Integer.parseInt(value.trim());

				} else if (key.equals(HTTPUtils.HEADER_CONTENT_TYPE)) {
					res_content_type = value;
				
				} else if (key.equals(HTTPUtils.HEADER_CONNECTION)) {
					if (value.trim().toLowerCase().equals("close")) {
						closeConnection = true;
					}
				
				} else if (key.equals(HTTPUtils.HEADER_TRANSFER_ENCODING)) {
					if (value.trim().toLowerCase().equals("chunked")) {
						chunked = true;
					}
				
				} else if (key.equals(HTTPUtils.HEADER_SET_COOKIE)) {
					if (cookies != null) {
						cookies.addCookie(value,host);
					}
					
				} else if (key.equals(HTTPUtils.HEADER_LOCATION)) {
					//TODO maybe have an option to automatically follow location?
					//or at least get the redirect
				}
			} else {
				if (SOAPFactory.DEBUG) System.out.println("Invalid Header: "+header);
			}
			header = new String(StreamUtils.readLine(sin));
		}

		byte[] content = null;
		
		if (!chunked) {
			if (res_content_len > 0) {
				//content length specified - OK
				content = StreamUtils.readBytes(sin,res_content_len);
			} else if (res_content_type != null && closeConnection) {
				//content type specified but not length
				content = StreamUtils.readAll(sin);
			} else if (res_content_type != null && !closeConnection) {
				//content type specified but not length
				closeConnection = true;
				content = StreamUtils.readAll(sin);
			} else {
				//no content type or length specified
				content = new byte[0];
			}
		} else {
			
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			
			while (true) {
				String chunk_header = new String(StreamUtils.readLine(sin));
				
				int index = chunk_header.indexOf(';');
				if (index != -1) {
					chunk_header = chunk_header.substring(0,index);
				}
				
				chunk_header = chunk_header.trim();
				
				int chunk_len = Integer.parseInt(chunk_header,16);

				if (chunk_len == 0) {
					break;
				} else {
					bout.write(StreamUtils.readBytes(sin,chunk_len));
					StreamUtils.readLine(sin);
				}
			}
			
			//ignored footers
			String footer = new String(StreamUtils.readLine(sin));
			while (footer.trim().length() > 0) {
				footer = new String(StreamUtils.readLine(sin));
			}
			
			content = bout.toByteArray();
		}

//		if (SOAPFactory.DEBUG) System.out.println("Content: ("+content.length+" bytes) "+new String(content));		
		
		if (closeConnection) {
			close();
		}
		
		return content;
	}
	
	public byte[] doGet(String[][] headers) throws IOException {
		return doGet(headers,null);
	}

	public byte[] doGet(String[][] headers, CookieCache cookies) throws IOException {

		if (sock == null) {
			connect();
		}
		
		IOException last_error = null;
		
		for (int i = 0; i < (RETRIES+1); i++) {
			try {
				
				if (proxy_host == null) {
					HTTPUtils.writeInitialHeader(sout,HTTPUtils.REQUEST_GET,resource,version);
					if (SOAPFactory.DEBUG) System.out.println(HTTPUtils.REQUEST_GET+" "+resource+" "+version);
				} else {
					HTTPUtils.writeInitialHeader(sout,HTTPUtils.REQUEST_GET,fullurl,version);
					if (SOAPFactory.DEBUG) System.out.println(HTTPUtils.REQUEST_GET+" "+fullurl+" "+version);
				}
		
				writeMandatoryHeaders(null);
				writeHeaders(headers);
				writeCookieHeaders(cookies);
				writeContent(null);
				
				sout.flush();
				
				String resp_code = readResponse(cookies);
				byte[] resp_dat = readContent(cookies);
				
				return resp_dat;
				
//			} catch (SocketException x) {
			} catch (IOException x) {
				last_error = x;
				if (x instanceof SocketException || x.getMessage().equals("End of stream")) {
					if (i < (RETRIES+1)) {
						//re-establish socket and try again
						connect();
					} else {
						//still got an error - so rethrow
						throw x;
					}
				}
			}
		}
		
		if (last_error != null) throw last_error;
		
		throw new IOException("Unknown error");
	}
	
	public byte[] doPost(byte[] dat, String mimetype, String[][] headers) throws IOException {
		return doPost(dat,mimetype,headers,null);
	}
	public byte[] doPost(byte[] dat, String mimetype, String[][] headers, CookieCache cookies) throws IOException {

		if (sock == null) {
			connect();
		}
		
		IOException last_error = null;
		
		for (int i = 0; i < (RETRIES+1); i++) {
			try {
				
				if (proxy_host == null) {
					HTTPUtils.writeInitialHeader(sout,HTTPUtils.REQUEST_POST,resource,version);
					if (SOAPFactory.DEBUG) System.out.println(HTTPUtils.REQUEST_POST+" "+resource+" "+version);
				} else {
					HTTPUtils.writeInitialHeader(sout,HTTPUtils.REQUEST_POST,fullurl,version);
					if (SOAPFactory.DEBUG) System.out.println(HTTPUtils.REQUEST_POST+" "+fullurl+" "+version);
				}

				writeMandatoryHeaders(mimetype);
				writeHeaders(headers);
				writeCookieHeaders(cookies);
				writeContent(dat);
				
				sout.flush();
		
				String resp_code = readResponse(cookies);
				byte[] resp_dat = readContent(cookies);
				
				return resp_dat;
				
//				} catch (SocketException x) {
			} catch (IOException x) {
				last_error = x;
				if (x instanceof SocketException || x.getMessage().equals("End of stream")) {
					if (i < (RETRIES+1)) {
						//re-establish socket and try again
						connect();
					} else {
						//still got an error - so rethrow
						throw x;
					}
				}
			}
		}
		
		if (last_error != null) throw last_error;
		
		throw new IOException("Unknown error");
	}
	
	/*
	public void write(byte[] dat, String[][] headers) throws IOException {
		HttpURLConnection httpConn = newHttpConnection("POST");
		
		//set any request headers necessary
		for (int i = 0; i < headers.length; i++) {
			httpConn.setRequestProperty(headers[i][0],headers[i][1]);
		}
        
		writeHTTP(dat,httpConn);
		
		httpConn.disconnect();
	}*/
	/*
	public byte[] read() throws IOException {
		HttpURLConnection httpConn = newHttpConnection("GET");
        
		byte[] dat;
		try {
			dat = readHTTP(httpConn);
		} catch (IOException e) {
			httpConn.disconnect();
			dat = readErrors(httpConn);
		}

		httpConn.disconnect();
		
		return dat;
	}*/
	/*
	public byte[] writeAndRead(byte[] dat, String[][] headers) throws IOException {
		HttpURLConnection httpConn = newHttpConnection("POST");

		//set any request headers necessary
		for (int i = 0; i < headers.length; i++) {
			httpConn.setRequestProperty(headers[i][0],headers[i][1]);
		}
       
		writeHTTP(dat,httpConn);
		
		byte[] rdat;
		try {
			rdat = readHTTP(httpConn);
		} catch (IOException e) {
			try {
				rdat = readErrors(httpConn);
			} catch (IOException x) {
				httpConn.disconnect();
				throw x;
			}
		}
		
		httpConn.disconnect();

		return rdat;
	}*/
	
	/*
	private HttpURLConnection newHttpConnection(String method) throws IOException {
		HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
		httpConn.setRequestProperty("Content-Type","text/xml; charset=utf-8");
//replaced by the array of headers
//		httpConn.setRequestProperty("SOAPAction",SOAPAction);
        httpConn.setRequestMethod(method);
        httpConn.setDoOutput(true);
        httpConn.setDoInput(true);
        return httpConn;
	}*/
	
	/*
	private byte[] readErrors(HttpURLConnection httpConn) throws IOException {
		BufferedInputStream bin = new BufferedInputStream(httpConn.getErrorStream());
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		
		byte[] buf = new byte[1024];
		int n = 0;
		
		while (n != -1) {
			n = bin.read(buf,0,1024);
			if (n > 0) {
				bout.write(buf,0,n);
			}
		}
		
		throw new IOException("HTTP server returned error code "+httpConn.getResponseCode()+" ("+httpConn.getResponseMessage()+"): "+new String(bout.toByteArray()));
//		return bout.toByteArray();
	}
	*/
	/*
	private byte[] readHTTP(HttpURLConnection httpConn) throws IOException {
		BufferedInputStream bin = new BufferedInputStream(httpConn.getInputStream());
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		byte[] buf = new byte[1024];
		int n = 0;
		
		while (n != -1) {
			n = bin.read(buf,0,1024);
			if (n > 0) {
				bout.write(buf,0,n);
			}
		}
		
		return bout.toByteArray();
	}
	
	private void writeHTTP(byte[] dat, HttpURLConnection httpConn) throws IOException {
		OutputStream out = httpConn.getOutputStream();
		out.write(dat);
		out.flush();
	}*/
}