/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.mutex.QueuedBlockingMap;

/**
 * An HTTP client that can be used instead of the HTTPClient class.
 * 
 * This class will manage a pool of connections using keepalive.
 * 
 * @author amiguel
 *
 */
public class HTTPClientPool {
	
	/** The maximum number of concurrent connections to any one 'protocol/host/port' tuple */
	private static int MAX_CONCURRENT_CONNECTIONS = 15000;
	/** If we're still waiting for a connection from the pool after this time then timeout, 0 = forever, N = N milliseconds */
	private static int MAX_POOL_WAIT = 0;	
	
	private static QueuedBlockingMap clients = new QueuedBlockingMap();
	private static HashMap available_clients = new HashMap();
	
//	private static Object pools_LOCK = new Object();
//	private static HashMap pools = new HashMap();
	
	private static String getKey(URL url, String correlation) throws IOException {
		if (correlation == null) correlation = "";
		
		StringBuffer sb = new StringBuffer();
		sb.append("CRLN:");
		sb.append(correlation);
		sb.append("URL:");
		sb.append(url.getProtocol());
		sb.append("://");
		sb.append(url.getHost());
		sb.append(":");
		int port = url.getPort();
		if (port == -1) {
			try {
				port = url.getDefaultPort();
			} catch (NoSuchMethodError t) {
				if (url.getProtocol().equalsIgnoreCase("http")) {
					port = 80;
				} else if (url.getProtocol().equalsIgnoreCase("https")) {
					port = 443;
				} else {
					port = 80;
				}
			}
		}
		sb.append(port);
		return sb.toString();
	}
	
	private static HTTPClient getConnectionFromPool(String key, URL address) throws IOException {
		
		try {
			
			//try to get a previously used client from the keep alive pool
			HTTPClient client = (HTTPClient)clients.get(key,1);
			
			client.setResource(address);
			
			//return this previously used client
			return client;
			
		} catch(InterruptedException e) {
			
			//no client found, maybe we need to increase the pool
			synchronized(available_clients) {
				
				//find out what the current concurrent connection count is
				Integer total = (Integer)available_clients.get(key);
				if (total == null) {
					total = new Integer(0);
				}
				
				if (total.intValue() < MAX_CONCURRENT_CONNECTIONS) {
					//we can add more connections
					
					//create a new connection
					HTTPClient client = new HTTPClient(address);
					
					//keep track of the concurrent connections we have
					total = new Integer(total.intValue() + 1);
					available_clients.put(key,total);
					
					//return this newly created client
					return client;
					
				}
			}//end sync
			
			//we are at the limit of allowed concurrent connections

			long timeout = MAX_POOL_WAIT;
			if (timeout <= 0) {
				timeout = 0;
			}

			try {
				//we need to wait until an existing one becomes free
				return (HTTPClient)clients.get(key,timeout);
			} catch (InterruptedException x) {
				throw new IOException("Timed out waiting for a client from the pool ("+timeout+"ms) - "+x);
			}

		}
	}
	private static void returnConnectionToPool(String key, HTTPClient client) {
		//put this back into the pool
		try {
			clients.put(key,client);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}
	
	public static byte[] doGet(URL address, String[][] headers) throws IOException {
		return doGet(address,headers,null,-1,null);
	}

	public static byte[] doGet(URL address, String[][] headers, CookieCache cookies) throws IOException {
		return doGet(address,headers,null,-1,null,cookies);
	}
	
	public static byte[] doGet(URL address, String[][] headers, String correlationKey) throws IOException {
		return doGet(address,headers,null,-1,correlationKey,null);
	}

	public static byte[] doGet(URL address, String[][] headers, String proxy_host, int proxy_port) throws IOException {
		return doGet(address,headers,proxy_host,proxy_port,null);
	}

	public static byte[] doGet(URL address, String[][] headers, String proxy_host, int proxy_port, CookieCache cookies) throws IOException {
		return doGet(address,headers,proxy_host,proxy_port,null,cookies);
	}
	
	public static byte[] doGet(URL address, String[][] headers, String proxy_host, int proxy_port, String correlationKey, CookieCache cookies) throws IOException {
		String key = getKey(address,correlationKey);
		
		//get an HTTP client from the pool
		HTTPClient client = getConnectionFromPool(key,address);

		byte[] ret = null;
		Throwable err = null;
		
		try {
			client.setProxy(proxy_host,proxy_port);
			ret = client.doGet(headers,cookies);
		} catch (Throwable t) {
			err = t;
		}
		
		returnConnectionToPool(key,client);
		
		if (err != null) {
			if (err instanceof IOException) {
				throw (IOException)err; 
			} else {
				throw new IOException(""+err);
			}
		} else {
			return ret;
		}
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers) throws IOException {
		return doPost(address,dat,mimetype,headers,null,-1,null);
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers, CookieCache cookies) throws IOException {
		return doPost(address,dat,mimetype,headers,null,-1,null,cookies);
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers, String correlationKey) throws IOException {
		return doPost(address,dat,mimetype,headers,null,-1,correlationKey,null);
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers, String proxy_host, int proxy_port) throws IOException {
		return doPost(address,dat,mimetype,headers,proxy_host,proxy_port,null,null);
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers, String proxy_host, int proxy_port, CookieCache cookies) throws IOException {
		return doPost(address,dat,mimetype,headers,proxy_host,proxy_port,null,cookies);
	}
	public static byte[] doPost(URL address, byte[] dat, String mimetype, String[][] headers, String proxy_host, int proxy_port, String correlationKey, CookieCache cookies) throws IOException {
		String key = getKey(address,correlationKey);
		
		//get an HTTP client from the pool
		HTTPClient client = getConnectionFromPool(key,address);

		byte[] ret = null;
		Throwable err = null;
		
		try {
			client.setProxy(proxy_host,proxy_port);
			ret = client.doPost(dat,mimetype,headers,cookies);
		} catch (Throwable t) {
			err = t;
		}
		
		returnConnectionToPool(key,client);
		
		if (err != null) {
			if (err instanceof IOException) {
				throw (IOException)err; 
			} else {
				throw new IOException(""+err);
			}
		} else {
			return ret;
		}
	}
}