/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http;


import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.TimeZone;

import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPFactory;
import org.eclipse.stp.b2j.core.jengine.internal.utils.CharStack;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.SyncCounter;

//import javax.net.ssl.SSLServerSocket;
//import javax.net.ssl.SSLServerSocketFactory;

/**
 * 
 * @author amiguel
 *
 * Part of the HTTP server implementation used by the SOAP binding
 */
public class PortServer {

private static final String RECEIVE_PREFIX = "RCV:";
private static final String REPLY_PREFIX = "RPLY:";
	
String protocol;
int port;

SimpleDateFormat sdf;

	public PortServer(String protocol, int port) throws Exception {
		this.protocol = protocol;
		this.port = port;
		
		sdf = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z");
		sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
		
		new AcceptThread(protocol,port).start();
	}

	class AcceptThread extends Thread {
		ServerSocket sock;
		public AcceptThread(String protocol, int port) throws Exception {
			setDaemon(true);
			
			if (protocol.equalsIgnoreCase("http")) {
				sock = new ServerSocket(port,500);
			} else if (protocol.equalsIgnoreCase("https")) {
				javax.net.ssl.SSLServerSocket sslsock = (javax.net.ssl.SSLServerSocket)javax.net.ssl.SSLServerSocketFactory.getDefault().createServerSocket(port,500);
				
				//enable all cipher suites
				String[] suites = sslsock.getSupportedCipherSuites();
				sslsock.setEnabledCipherSuites(suites);
				
				//disable authentication
				sslsock.setNeedClientAuth(false);
				sslsock.setWantClientAuth(false);
				
				sock = sslsock;
//				throw new Exception("HTTPS currently unsupported");
			}
			
			if (SOAPFactory.DEBUG) System.out.println("Port server created on "+port);
		}
		public void run() {
			try {
				if (SOAPFactory.DEBUG) System.out.println("Port server on "+port+" waiting for connections");
				while (true) {
					Socket s = sock.accept();
					new SocketThread(s).start();
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	private String getHtmlStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		
		StringBuffer buf = new StringBuffer(os.toString());
		
		int index = buf.indexOf("'\n");
		while (index != -1) {
			buf.insert(index+1,"<br>");
			index = buf.indexOf("\n",index+1);
		}
		index = buf.indexOf("\t");
		while (index != -1) {
			buf.replace(index,index+1,"&nbsp;");
			index = buf.indexOf("\t",index+1);
		}
		index = buf.indexOf(" ");
		while (index != -1) {
			buf.replace(index,index+1,"&nbsp;");
			index = buf.indexOf(" ",index+1);
		}
		
		return buf.toString();
	}//end method
	

	SyncCounter open_connections = new SyncCounter();
	
	class SocketThread extends Thread {
		Socket sock;
		public SocketThread(Socket s) {
			this.sock = s;
		}
		public void run() {

			open_connections.add(1);
			System.out.println("PortServer - NEW CONNECTION "+open_connections.getValue());

			if (SOAPFactory.DEBUG) System.out.println("Port server on "+port+" handling incoming connection");
			try {
				
				BufferedInputStream bin = new BufferedInputStream(sock.getInputStream());
				HashMap headers = new HashMap();
				CharStack stack = null;
				
				boolean loop = true;
				
				//loop=true means KEEPALIVE
				while (loop) {

					String reqtype = null;
					String version = null;
					String resource = null;
					
					String payload = null;
					
					String response_code = null;
					String response_errdetail = "";
					String response_payload = "";
					
					boolean addKeepAliveHeader = false;
					
					try {
						
						//initial request
						stack = new CharStack(new String(StreamUtils.readLine(bin)));
						stack.popWhitespace();
						reqtype = stack.popText(true);
						stack.popWhitespace();
						resource = stack.popText(true);
						stack.popWhitespace();
						version = stack.popText(true);
	
						if (SOAPFactory.DEBUG) System.out.println("Request: "+reqtype);					
						if (SOAPFactory.DEBUG) System.out.println("Resource: "+resource);					
						if (SOAPFactory.DEBUG) System.out.println("Version: "+version);					
						
						if (version.endsWith("1.1")) {
							//HTTP 1.1
							StringBuffer response_cont = new StringBuffer();
							response_cont.append("HTTP/1.1 100 Continue\r\n\r\n");
							sock.getOutputStream().write(response_cont.toString().getBytes());
						}
						
						//optional headers
						String header = new String(StreamUtils.readLine(bin));
						while (header.trim().length() > 0) {
							
							int index = header.indexOf(':');
							
							if (index >= 0) {
							
								String key = header.substring(0,index).trim().toLowerCase();
								String value = header.substring(index+1);
								
								//store all headers as lower case keys
								headers.put(key,value);
		
								if (SOAPFactory.DEBUG) System.out.println("Header: "+key+"="+value);					
							} else {
								if (SOAPFactory.DEBUG) System.out.println("Invalid Header: "+header);
							}
							header = new String(StreamUtils.readLine(bin));
						}
						
						//payload
						String content_length = (String)headers.get("content-length");
						if (content_length != null) {
							int len = (Integer.parseInt(content_length.trim()));
	
							if (SOAPFactory.DEBUG) System.out.println("(reading content, "+len+" bytes)");
							
							//TODO this payload we shouldn't really convert to a string, should pass around as bytes
							payload = new String(StreamUtils.readBytes(bin,len));
							
							if (SOAPFactory.DEBUG) System.out.println("Payload: ("+content_length+" bytes)");					
						} else {
							if (SOAPFactory.DEBUG) System.out.println("Payload: (none)");					
						}
					
					} catch (Exception e) {
						response_code = "500 Internal Server Error";
						response_errdetail = getHtmlStackTrace(e);
					}
					
					if (version.endsWith("1.1")) {
						//HTTP 1.1
						//keep alive unless we get a connection: close header
						loop = true;

						String tmp = (String) headers.get(HTTPUtils.HEADER_CONNECTION);
						if (tmp != null) {
							tmp = tmp.trim();
							if (tmp.equalsIgnoreCase("close")) {
								//Connection: close
								loop = false;
							}
						}
						
					} else if (version.endsWith("1.0")) {
						//HTTP 1.0
						//keep alive only if keepalive header is specified
						loop = false;
						
						String tmp = (String) headers.get(HTTPUtils.HEADER_CONNECTION);
						if (tmp != null) {
							tmp = tmp.trim().toLowerCase();
							if (tmp.startsWith("keep") && tmp.endsWith("alive")) {
								//Connection: Keep-Alive
								loop = true;
								addKeepAliveHeader = true;
							}
						}
						
					} else {
						loop = false;
					}
					
					try {
						if (headers.get(HTTPUtils.HEADER_TRANSFER_ENCODING) != null) {
							response_code = "501 Not Implemented";
							response_errdetail = "This HTTP server does not currently support the HTTP Transfer-Encoding header";
						} else if (!version.endsWith("1.0") && !version.endsWith("1.1")) {
							response_code = "505 HTTP Version Not Supported";
							response_errdetail = "This HTTP server supports only HTTP 1.0 and HTTP 1.1 (requested version was "+version+")";
						} else if (version.endsWith("1.1") && headers.get(HTTPUtils.HEADER_HOST) == null) {
							response_code = "400 Bad Request";
							response_errdetail = "No 'Host: ' header received.  All HTTP 1.1 requests MUST contain a 'Host: ' header";
						}
						
						if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_GET)) {
							//OK
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_POST)) {
							//OK
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_HEAD)) {
							//OK
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_PUT)) {
							//OK
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_TRACE)) {
							response_code = "501 Not Implemented";
							response_errdetail = "This HTTP server does not support the TRACE method";
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_OPTIONS)) {
							response_code = "501 Not Implemented";
							response_errdetail = "This HTTP server does not support the OPTIONS method";
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_DELETE)) {
							response_code = "501 Not Implemented";
							response_errdetail = "This HTTP server does not support the DELETE method";
						} else if (reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_CONNECT)) {
							response_code = "501 Not Implemented";
							response_errdetail = "This HTTP server does not support the CONNECT method";
						} else {
							response_code = "501 Not Implemented";
							response_errdetail = "HTTP method "+reqtype+" not recognised";
						}

						if (response_code == null) {
							//no errors so far
							response_payload = HTTPServer.doRequest(port, resource, payload, headers);
							if (response_payload == null) {
								response_payload = "";
							}
							response_code = "200 OK";
						}
					} catch (HTTPNotFoundException e) {
						//404 not found
						response_code = "404 Not Found";
					} catch (HTTPException e) {
						//500 internal server error
						response_code = "500 Internal Server Error";
						response_errdetail = getHtmlStackTrace(e);
					}
					if (SOAPFactory.DEBUG) System.out.println("Response: "+response_code);
					
					if (!response_code.startsWith("20") && response_payload.length() == 0) {
						response_payload = HTTPUtils.getHtmlError(response_code,response_errdetail);
					}
				
					StringBuffer response = new StringBuffer();
					response.append(version);
					response.append(" ");
					response.append(response_code);
					response.append("\r\n");
					
					if (version.endsWith("1.1")) {
						//in HTTP 1.1 we must return a current Date: header
						response.append(sdf.format(new Date()));
						response.append("\r\n");
					}
					
					if (addKeepAliveHeader) {
						//in HTTP 1.0 we must echo a [Connection: Keep-Alive] header
						response.append("Connection: Keep-Alive");
						response.append("\r\n");
					}

					if (response_payload.length() > 0) {
						response.append("Content-Type: text/html");
						response.append("\r\n");
						response.append("Content-Length: "+response_payload.length());
						response.append("\r\n");

						if (SOAPFactory.DEBUG) System.out.println("Content-Length: "+response_payload.length());
					} else {
						response.append("Content-Length: 0");
						response.append("\r\n");

						if (SOAPFactory.DEBUG) System.out.println("Content-Length: 0");
					}
					
					//end headers
					response.append("\r\n");
					
					//begin payload
					if (!reqtype.equalsIgnoreCase(HTTPUtils.REQUEST_HEAD)) {
						//if its a HEAD request then dont send any payload
						if (response_payload.length() > 0) {
							response.append(response_payload);
						}
					}
					//end payload
					
					if (SOAPFactory.DEBUG) System.out.println("**** RESPONSE ****");
					if (SOAPFactory.DEBUG) System.out.print(response);
					if (SOAPFactory.DEBUG) System.out.println("**** (END) ****");

					byte[] responsedat = response.toString().getBytes();
					sock.getOutputStream().write(responsedat);
					sock.getOutputStream().flush();
				}
			
			} catch (IOException x) {
			} catch (Exception x) {
				//Invalid HTTP request
//				x.printStackTrace();
			}
			
			try {
				sock.close();
			} catch (Exception e) {
			}
			
			open_connections.subtract(1);
			System.out.println("PortServer - END CONNECTION "+open_connections.getValue());
			
		}
	}
	
}