/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.wsif;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDComplexType;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDComplexTypeElement;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDRootElement;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDType;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslatorException;
import org.w3c.dom.Element;

/**
 * 
 * @author amiguel
 *
 * Compiles in conversion functions between the engine-native representation
 * of XSD types and specified Java types.
 * 
 * http://ws.apache.org/wsif/providers/wsdl_extensions/formattypemapping_extension.html
 * 
 */
public class WSIFFormatTypeCodec implements XSDTypeTranslator {

public static final String NAMESPACE_FORMATNS = "http://schemas.xmlsoap.org/wsdl/formatbinding/";

ClassLoader dependancy_loader;

StringBuffer methods = new StringBuffer();

XSDMap xsdmap;
WSDLMap wsdlmap;

HashMap qxsdType_to_formatType = new HashMap();

Util compiler_util;

class TypeMapping {
	boolean isArray;
	String qtype;
	ArrayList qportTypes = new ArrayList();
	ArrayList formatTypes = new ArrayList();
}

	public String getToJavaFunction(String qtype, String qportType) throws WSDLBindingTranslatorException {
		TypeMapping mapping = (TypeMapping)qxsdType_to_formatType.get(qtype);

		if (mapping == null) {
			throw new WSDLBindingTranslatorException("formatType mapping not found for "+qtype);
		}
		
		int index = mapping.qportTypes.indexOf(qportType);
		
		if (index == -1) {
			throw new WSDLBindingTranslatorException("no formatType mapping in type "+qtype+" found for portType "+qportType);
		}

		return getToJavaFunctionName((String)mapping.formatTypes.get(index));
	}
	public String getFromJavaFunction(String qtype, String qportType) throws WSDLBindingTranslatorException {
		TypeMapping mapping = (TypeMapping)qxsdType_to_formatType.get(qtype);
		
		if (mapping == null) {
			throw new WSDLBindingTranslatorException("formatType mapping not found for "+qtype);
		}

		int index = mapping.qportTypes.indexOf(qportType);
		
		if (index == -1) {
			throw new WSDLBindingTranslatorException("no formatType mapping in type "+qtype+" found for portType "+qportType);
		}

		return getFromJavaFunctionName((String)mapping.formatTypes.get(index));
	}
	public String getJavaType(String qtype, String qportType) throws WSDLBindingTranslatorException {
		TypeMapping mapping = (TypeMapping)qxsdType_to_formatType.get(qtype);
		
		if (mapping == null) {
			throw new WSDLBindingTranslatorException("formatType mapping not found for "+qtype);
		}

		int index = mapping.qportTypes.indexOf(qportType);
		
		if (index == -1) {
			throw new WSDLBindingTranslatorException("no formatType mapping in type "+qtype+" found for portType "+qportType);
		}

		return (String)mapping.formatTypes.get(index);
	}

	private static String classnameToID(String classname) {
		return classname.replace('.','_');
	}
	
	private static String getSetterFunction(String elemName) {
		StringBuffer sb = new StringBuffer();
		sb.append("set");
		sb.append(Character.toUpperCase(elemName.charAt(0)));
		sb.append(elemName.substring(1));
		return sb.toString();
	}

	private static String getGetterFunction(String elemName) {
		StringBuffer sb = new StringBuffer();
		sb.append("get");
		sb.append(Character.toUpperCase(elemName.charAt(0)));
		sb.append(elemName.substring(1));
		return sb.toString();
	}
	
	private static String getToJavaFunctionName(String classname) {
		return "wsifFormatType_to_"+classnameToID(classname)+"";
	}
	private static String getFromJavaFunctionName(String classname) {
		return "wsifFormatType_from_"+classnameToID(classname)+"";
	}
	
	public void init(Util compiler_util, XSDMap xsdmap, WSDLMap wsdlmap, ClassLoader dependancy_loader) {
		this.xsdmap = xsdmap;
		this.wsdlmap = wsdlmap;
		this.dependancy_loader = dependancy_loader;
		this.compiler_util = compiler_util;
	}

	/**
	 * Add a format type mapping for an XSD type
	 * @param qtype the XSD type to map
	 * @param formatType the formatType to map to
	 * @param qportType the portType associated with this mapping
	 * @throws XSDTypeTranslatorException
	 */
	private void addTypeMapping(String qtype, String formatType, String qportType) throws XSDTypeTranslatorException {

		//convert 'int' to 'java.lang.Integer' etc - we deal only with objects and autobox as we need to later on
		formatType = getNonPrimitiveClass(formatType);
		
		//get the type mappings for this XSD type
		TypeMapping mapping = (TypeMapping)qxsdType_to_formatType.get(qtype);
		if (mapping == null) {
			mapping = new TypeMapping();
			mapping.qtype = qtype;
			qxsdType_to_formatType.put(qtype,mapping);
//			DBG.info("added type mapping for type ["+qtype+"]");
		}
	
		//find the (single) mapping of this XSD type for a particular portType
		int index = mapping.qportTypes.indexOf(qportType);
		if (index != -1) {
			String fType = (String)mapping.formatTypes.get(index);
			if (fType.equals(formatType)) {
				//all is well - this is just a simple duplicate
			} else {
				//multiple format type definitions for the same portType
				throw new XSDTypeTranslatorException("multiple format type definitions found for type "+qtype+" under port type "+qportType+" (def 1:"+fType+") (def 2:"+formatType+")");
			}
			
		} else {
			//this is a new mapping of the XSD type for a particular portType
			mapping.formatTypes.add(formatType);
			mapping.qportTypes.add(qportType);

//			DBG.info("added formatType mapping for "+qtype+" -> "+formatType+" under portType "+qportType+"");
			
			//need to recursively generate mappings on this class now for each element of the type qtype
			XSDType abstractType = (XSDType)xsdmap.getType(qtype);
			if (abstractType == null) {
				throw new XSDTypeTranslatorException("no xsd type found in XSD Map for xsd type "+qtype);
			}
			if (abstractType instanceof XSDComplexType) {
				//this type that we are mapping is a complex type
				XSDComplexType complexType = (XSDComplexType)abstractType;
	
				//check all its fields and map them
				for (int k = 0; k < complexType.getElementCount(); k++) {
					XSDComplexTypeElement cxelem = complexType.getElement(k);
					String elemFormatType = getClassFieldType(formatType,cxelem.getName());
					
					elemFormatType = getNonPrimitiveClass(elemFormatType);
					
					//create a new mapping from each element of this complex type to its corresponding
					//java type
					addTypeMapping(cxelem.getQualifiedType(),elemFormatType,qportType);
				}
			}
		}		
	}
	
	String[][] primitive_mappings = null;
	
	
	private void initPrimitiveMappings() {
		//
		// NOTE - this must ONLY CONTAIN PRIMITIVES as we use it for 
		// checking to see if a type is primitive as well as mapping classes
		//
		if (primitive_mappings == null) {
			primitive_mappings = new String[][] {
				{Integer.TYPE.getName(),Integer.class.getName()},
				{Character.TYPE.getName(),Character.class.getName()},
				{Byte.TYPE.getName(),Byte.class.getName()},
				{Short.TYPE.getName(),Short.class.getName()},
				{Integer.TYPE.getName(),Integer.class.getName()},
				{Long.TYPE.getName(),Long.class.getName()},
				{Float.TYPE.getName(),Float.class.getName()},
				{Double.TYPE.getName(),Double.class.getName()},
               {Boolean.TYPE.getName(),Boolean.class.getName()},
			};
		}
	}

	private boolean isPrimitiveClass(String clazz) {
		initPrimitiveMappings();
		for (int i = 0; i < primitive_mappings.length; i++) {
			if (clazz.equals(primitive_mappings[i][0])) {
				return true;
			}
		}
		return false;
	}
	
	private String getNonPrimitiveClass(String clazz) {
		initPrimitiveMappings();
		//we are using reflection to invoke methods, so we make sure that we always
		//map primitive classes onto their non-primitive versions (eg int.class to Integer.class)
		for (int i = 0; i < primitive_mappings.length; i++) {
			if (clazz.equals(primitive_mappings[i][0])) {
				return primitive_mappings[i][1];
			}
		}
		return clazz;
	}

	private String getPrimitiveClass(String clazz) {
		initPrimitiveMappings();
		//we are using reflection to invoke methods, so we make sure that we always
		//map primitive classes onto their non-primitive versions (eg int.class to Integer.class)
		for (int i = 0; i < primitive_mappings.length; i++) {
			if (clazz.equals(primitive_mappings[i][1])) {
				return primitive_mappings[i][0];
			}
		}
		return clazz;
	}
	
	public void readWSDL(Element definitions) throws XSDTypeTranslatorException, NamespaceException {
		NamespaceTranslator defnt = compiler_util.createNamespaceTranslator();
		defnt.addNamespaces(definitions);
		
		ArrayList elems = Util.getAllElements(definitions);
		
		for (int z = 0; z < elems.size(); z++) {
			Element elem = (Element)elems.get(z);
		
			//this codec doesn't need any extra annotation in the WSDL
			defnt.addNamespaces(elem);
	
			String name = NamespaceTranslator.getName(elem);
			String ns = defnt.getNamespace(elem,true);
			
			String bindingName = "";
			
			if (name.equals("binding")) {
				bindingName = elem.getAttribute("name");
				String qportType = defnt.qualify(elem.getAttribute("type"),false);
				
				ArrayList typemappings = Util.getAllElements(elem);
				for (int i = 0; i < typemappings.size(); i++) {
					Element typemapping = (Element)typemappings.get(i);
					defnt.addNamespaces(typemapping);
					
					String typemappingname = NamespaceTranslator.getName(typemapping);
					String typemappingns = defnt.getNamespace(typemapping,false);
			
					if (defnt.checkQName(typemapping,NAMESPACE_FORMATNS,"typeMapping")) {
	
						String style = typemapping.getAttribute("style");
						String encoding = typemapping.getAttribute("encoding");

						if (!style.toLowerCase().equals("java")) {
							throw new XSDTypeTranslatorException("unsupported formatType style "+style);
						}
						if (!encoding.toLowerCase().equals("java")) {
							throw new XSDTypeTranslatorException("unsupported formatType encoding "+style);
						}
						
						ArrayList typemaps = Util.getAllElements(typemapping);
						for (int k = 0; k < typemaps.size(); k++) {
							Element typemap = (Element)typemaps.get(k);
							defnt.addNamespaces(typemap);
	
							String typemapname = NamespaceTranslator.getName(typemap);
							String typemapns = defnt.getNamespace(typemap,false);
							
	//						if (typemapns.equals(NAMESPACE_FORMATNS) && typemapname.equals("typeMap")) {
							if (defnt.checkQName(typemap,NAMESPACE_FORMATNS,"typeMap")) {
								String typeName = typemap.getAttribute("typeName");
								String elementName = typemap.getAttribute("elementName");
								String formatType = typemap.getAttribute("formatType");
								
								if (formatType.length() == 0) {
									throw new XSDTypeTranslatorException("expected 'formatType' attribute");
								} else if (typeName.length() > 0) {
									
									//map type <typeName> to <formatType>
									String qtype = defnt.qualify(typeName,false);
									addTypeMapping(qtype,formatType,qportType);
									
								} else if (elementName.length() > 0) {
									
									//map element <elementName> to <formatType>
									XSDRootElement rootElem = xsdmap.getRootElementTypeByQName(defnt.qualify(elementName,false));
									String qtype = rootElem.getQualifiedType();
									addTypeMapping(qtype,formatType,qportType);
									
								} else {
									throw new XSDTypeTranslatorException("expected 'typeName' attribute or 'elementName' attribute");
								}
							}
	
							defnt.removeNamespaces(typemap);
						}
					}
					
					defnt.removeNamespaces(typemapping);
				}
			}
			defnt.removeNamespaces(elem);
		}
		defnt.removeNamespaces(definitions);
	}

	public void translateComplex(NamespaceTranslator nt, String name, String[] atypes, String[] anames, String[] types, String[] names, String[] elem_names, Element complex) throws XSDTypeTranslatorException, NamespaceException {
		methods.setLength(0);
		methods.append("\n");

		String originalName = name;
		String classname = NamespaceTranslator.getJavaClassName(name);
		
		String qname = nt.qualify(name,true);

//		DBG.info("WSIF Format Type codec translating complex type "+qname+"");
		
		TypeMapping typemap = (TypeMapping)qxsdType_to_formatType.get(qname);
		if (typemap == null) {
//			DBG.info("no format type mapping found for "+qname);
		} else {
			
			HashMap alreadyMapped = new HashMap();
			ArrayList formatTypes = typemap.formatTypes;
			for (int i = 0; i < formatTypes.size(); i++) {
				String formatType = (String)formatTypes.get(i);
				
				if (alreadyMapped.get(formatType) == null) {
					//dont add the same mapping more than once
					alreadyMapped.put(formatType,formatType);
				
//					DBG.info("adding complex type "+qname+" format type mapping implementation for "+formatType+"");
				
					String toFunctionName = getToJavaFunctionName(formatType);
					String fromFunctionName = getFromJavaFunctionName(formatType);
		
					XSDType abstractType = (XSDType)xsdmap.getType(qname);
					if (abstractType == null) {
						throw new XSDTypeTranslatorException("no xsd type found in XSD Map for xsd type "+qname+" ("+name+")");
					}
					if (!(abstractType instanceof XSDComplexType)) {
						throw new XSDTypeTranslatorException("expected to find complex type for xsd type "+qname+" ("+name+")");
					}
					XSDComplexType complexType = (XSDComplexType)abstractType;
					
					//
					// XSD TO JAVA method
					//
					methods.append("  public static "+formatType+" "+toFunctionName+"("+classname+" o) throws Exception {\n");
					methods.append("    //WSIF Mapping, "+classname+"->"+formatType+"\n");
					methods.append("    if (o == null) return null;\n");
					methods.append("    "+formatType+" tmp = new "+formatType+"();\n");
					for (int k = 0; k < complexType.getElementCount(); k++) {
						XSDComplexTypeElement cxelem = complexType.getElement(k);
						String setterFunction = getSetterFunction(cxelem.getName());
						String fieldType = getClassFieldType(formatType,cxelem.getName());
						boolean isPrimitive = isPrimitiveClass(fieldType);
						String nonPrimitiveFieldType = getNonPrimitiveClass(fieldType);
						String fieldTypeToJava = getToJavaFunctionName(nonPrimitiveFieldType);
						String xsdFieldType = cxelem.getQualifiedType();
						boolean isArray = isClassFieldTypeArray(formatType,cxelem.getName());

						if (isArray) {
							methods.append("	"+fieldType+"[] TMPSET"+k+" = new "+fieldType+"[o."+NamespaceTranslator.getElement(cxelem.getName())+".length];\n");
							methods.append("    for (int TMPI"+k+" = 0; TMPI"+k+" < TMPSET"+k+".length; TMPI"+k+"++) {\n");
							if (isPrimitive) {
								methods.append("    	TMPSET"+k+"[TMPI"+k+"] = "+xsdFieldType+"."+fieldTypeToJava+"(o."+NamespaceTranslator.getElement(cxelem.getName())+"[TMPI"+k+"])."+fieldType+"Value();\n");
							} else {
								methods.append("    	TMPSET"+k+"[TMPI"+k+"] = "+xsdFieldType+"."+fieldTypeToJava+"(o."+NamespaceTranslator.getElement(cxelem.getName())+"[TMPI"+k+"]);\n");
							}
							methods.append("	}\n");
							methods.append("	tmp."+setterFunction+"(TMPSET"+k+");\n");
						} else {
							if (isPrimitive) {
								//setter function will be expecting a primtive so we need to do our own autoboxing
								methods.append("    if (o."+NamespaceTranslator.getElement(cxelem.getName())+".length > 0) {\n");
								methods.append("    	tmp."+setterFunction+"("+xsdFieldType+"."+fieldTypeToJava+"(o."+NamespaceTranslator.getElement(cxelem.getName())+"[0])."+fieldType+"Value());\n");
								methods.append("    }\n");
							} else {
								methods.append("    if (o."+NamespaceTranslator.getElement(cxelem.getName())+".length > 0) {\n");
								methods.append(" 	    tmp."+setterFunction+"("+xsdFieldType+"."+fieldTypeToJava+"(o."+NamespaceTranslator.getElement(cxelem.getName())+"[0]));\n");
								methods.append("    }\n");
							}
						}
					}
					methods.append("     return tmp;\n");
					methods.append("  }\n");
					
					//
					// XSD FROM JAVA method
					//
					methods.append("  public static Object "+fromFunctionName+"("+formatType+" o) throws Exception {\n");
					methods.append("    //WSIF Mapping, "+formatType+"->"+classname+"\n");
					methods.append("    "+classname+" tmp = new "+classname+"();\n");
					
					methods.append("    if (o == null) {");
//					methods.append("       //return valid class but with null INTERNAL VALUE\n");
//					methods.append("       tmp.INTERNAL_VALUE = null;\n");
					methods.append("       return tmp;\n");
					methods.append("    }\n");
					
					methods.append("    tmp."+NamespaceTranslator.getElementTagName()+" = \""+name+"\";\n");
					for (int k = 0; k < complexType.getElementCount(); k++) {
						XSDComplexTypeElement cxelem = complexType.getElement(k);
						String getterFunction = getGetterFunction(cxelem.getName());
						String fieldType = getClassFieldType(formatType,cxelem.getName());
						boolean isPrimitive = isPrimitiveClass(fieldType);
						String nonPrimitiveFieldType = getNonPrimitiveClass(fieldType);
						String fieldTypeFromJava = getFromJavaFunctionName(nonPrimitiveFieldType);
						String xsdFieldType = cxelem.getQualifiedType();
						boolean isArray = isClassFieldTypeArray(formatType,cxelem.getName());

						if (isArray) {
							methods.append("    "+fieldType+" TMPGET"+k+"[] = o."+getterFunction+"();\n");
							methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+" = new "+cxelem.getQualifiedType()+"[TMPGET"+k+".length];\n");
							methods.append("	for (int TMPI"+k+" = 0; TMPI"+k+" < TMPGET"+k+".length; TMPI"+k+"++) {\n");
							if (isPrimitive) {
								//getter function will return a primtive so we need to do our own autoboxing
								methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+"[TMPI"+k+"] = ("+xsdFieldType+") "+xsdFieldType+"."+fieldTypeFromJava+"(new "+nonPrimitiveFieldType+"(TMPGET"+k+"[TMPI"+k+"]));\n");
							} else {
								methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+"[TMPI"+k+"] = ("+xsdFieldType+") "+xsdFieldType+"."+fieldTypeFromJava+"(TMPGET"+k+"[TMPI"+k+"]);\n");
							}
							methods.append("	}\n");
						} else {
							methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+" = new "+cxelem.getQualifiedType()+"[1];\n");
							if (isPrimitive) {
								//getter function will return a primtive so we need to do our own autoboxing
								methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+"[0] = ("+xsdFieldType+") "+xsdFieldType+"."+fieldTypeFromJava+"(new "+nonPrimitiveFieldType+"(o."+getterFunction+"()));\n");
							} else {
								methods.append("    tmp."+NamespaceTranslator.getElement(cxelem.getName())+"[0] = ("+xsdFieldType+") "+xsdFieldType+"."+fieldTypeFromJava+"(o."+getterFunction+"());\n");
							}
						}
					}
					methods.append("     return tmp;\n");
					methods.append("   }\n");
				}
			}
		}
	}
	
	public void translateSimple(NamespaceTranslator nt, String name, String base, Element simple) throws XSDTypeTranslatorException, NamespaceException {
		translateBase(nt,base,name,XSDTypeTranslator.XSD_TYPE_ANYTYPE);
	}

	public void translateBase(NamespaceTranslator nt, String name) throws XSDTypeTranslatorException, NamespaceException {
		translateBase(nt,name,name,XSDTypeTranslator.XSD_TYPE_ANYTYPE);	
	}
	
	private boolean isNumberClass(String clazz) {
		if (clazz.equals("java.lang.Integer")
			|| clazz.equals("java.lang.Double")
			|| clazz.equals("java.lang.Float")
			|| clazz.equals("java.lang.Short")
			|| clazz.equals("java.lang.Byte")
			|| clazz.equals("java.lang.Long")) {
			return true;
		}
		return false;
	}
	
	public void translateBase(NamespaceTranslator nt, String name, String realname, String listType) throws XSDTypeTranslatorException, NamespaceException {
		methods.setLength(0);
		methods.append("\n");

		String qname = nt.qualify(name,false);

		String mapping = compiler_util.getInternalBaseMapping(nt,qname);

		String classname = NamespaceTranslator.getJavaClassName(realname);
		
//		DBG.info("WSIF Format Type codec translating base type "+qname+"");
		
		TypeMapping typemap = (TypeMapping)qxsdType_to_formatType.get(qname);
		
		if (typemap == null) {
//			DBG.info("no format type mapping found for "+qname);
			
		} else {
			//we found some mappings for this type

			HashMap alreadyMapped = new HashMap();
			ArrayList formatTypes = typemap.formatTypes;
			for (int i = 0; i < formatTypes.size(); i++) {
				String formatType = (String)formatTypes.get(i);
	
				if (alreadyMapped.get(formatType) == null) {
					//dont add the same mapping more than once
					alreadyMapped.put(formatType,formatType);
					if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_ANYTYPE,false))) {
						//we should never be asked to translate anything for a list type
						throw new XSDTypeTranslatorException("formatType type mapping codec cannot deal with mapping xsd:anyType type");
						
					} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_LIST,false))) {
						//we should never be asked to translate anything for a list type
						throw new XSDTypeTranslatorException("formatType type mapping codec cannot deal with mapping xsd:list type");
						
					} else {
						//we need to add methods to this base type to convert it to and from <formatType>
						
						String toFunctionName = getToJavaFunctionName(formatType);
						String fromFunctionName = getFromJavaFunctionName(formatType);
						
						//TODO we dont deal with casting to and from the primitive type CHAR here
						//if we're mapping anything then it must be a Number instance

						//get the internal class that we use to represent this XSD base type 
						String internal_class = compiler_util.getInternalJavaType(NamespaceTranslator.getName(name));
						//no mapping for this base type?
						if (internal_class == null) throw new XSDTypeTranslatorException("no internal mapping found for XSD base type ["+name+"]");
						//get the non-primitive version of this class (in case it is primitive)
						internal_class = getNonPrimitiveClass(internal_class);
						
						//if the formatType does not match the internal type then we need to cast it
						boolean isCasted = !internal_class.equals(formatType);

						//we only support casting of primitive numeric types 
						//we dont support non-internal mappings of boolean, date, string etc
						if (isCasted && !isNumberClass(internal_class)) {
							throw new XSDTypeTranslatorException("can only return a type of ["+internal_class+"] for xsd base type ["+name+"], unable to return requested formatType ["+formatType+"]");
						}
						
						//
						// TO method
						//
						methods.append("  public static "+formatType+" "+toFunctionName+"("+classname+" tmp) throws Exception {\n");
						methods.append("    //WSIF Mapping, "+classname+"->"+formatType+"\n");
						methods.append("    if (tmp == null) return null;\n");
						if (!isCasted) {
							//no casting requred - just return the value
							methods.append("    return ("+formatType+")tmp.INTERNAL_VALUE;\n");
						} else {
							//casting requred - use the Number class hierarchy to get the appropriate primitive value and then
							//create an object round it, e.g.   new Integer(blah.intValue())
							String primitive = getPrimitiveClass(formatType);
							methods.append("    return new "+formatType+"(tmp.INTERNAL_VALUE."+primitive+"Value());\n");
						}
						methods.append("  }\n");
						
						//
						// FROM method
						//
						methods.append("  public static "+classname+" "+fromFunctionName+"(Object o) throws Exception {\n");
						methods.append("    //WSIF Mapping, "+formatType+"->"+classname+"\n");
						methods.append("    "+classname+" tmp = new "+classname+"();\n");

						methods.append("    if (o == null) {");
						methods.append("       //return valid class but with null INTERNAL VALUE\n");
						methods.append("       tmp.INTERNAL_VALUE = null;\n");
						methods.append("       return tmp;\n");
						methods.append("    }\n");
						
						if (!isCasted) {
							//no casting requred - just return the value
							methods.append("    tmp.INTERNAL_VALUE = ("+formatType+")o;\n");
						} else {
							//casting requred - use the Number class hierarchy to get the appropriate primitive value and then
							//create an object round it, e.g.   new Integer(blah.intValue())
							String primitive = getPrimitiveClass(internal_class);
							methods.append("    tmp.INTERNAL_VALUE = new "+internal_class+"((("+formatType+")o)."+primitive+"Value());\n");
						}
						methods.append("    return tmp;\n");
						methods.append("   }\n");
						
					}
				}
			}
		}
	}
	
	/**
	 * Returns a java class body with any method implementations for this particular complex type
	 * @return
	 */
	public String getMethods() {
		if (methods.length() == 0) return "";
		return methods.toString();
	}
	
	/**
	 * Returns an array of interfaces that this complex type will be said to implement
	 * @return
	 */
	public String[] getImplementations() {
		return new String[] {
		};
	}
	
	public String[] getImports() {
		return new String[0];
	}
	
	public boolean isClassFieldTypeArray(String classname, String fieldName) throws XSDTypeTranslatorException {
		Field field = getClassField(classname,fieldName);
		return field.getType().isArray();
	}

	public String getClassFieldType(String classname, String fieldName) throws XSDTypeTranslatorException {
		Field field = getClassField(classname,fieldName);

		Class fieldType = field.getType();
		
		if (!fieldType.isPrimitive()) {
			classCache.put(fieldType.getName(),fieldType);
//			DBG.info("WSIF Format Type codec got + cached field "+fieldName+"'s class ("+fieldType.getName()+")");
		} else {
//			DBG.info("WSIF Format Type codec got field "+fieldName+"'s primitive type ("+fieldType.getName()+")");
		}
		
		String clazz = fieldType.getName();
		
		//deal with arrays - strip out the class
		if (clazz.startsWith("[L")) {
			clazz = clazz.substring(2,clazz.length()-1);
		}
		
		return clazz;
	}
	
	HashMap classCache = new HashMap();
	
	public Field getClassField(String classname, String fieldName) throws XSDTypeTranslatorException {
		String error = "";
		try {
			Class c = (Class)classCache.get(classname);
			if (c == null) {
//				DBG.info("WSIF Format Type codec trying to load class "+classname+" via Class.forName to get field types");
				try {
					c = Class.forName(classname);
//					DBG.info("WSIF Format Type codec loaded class "+classname+" OK");
					classCache.put(classname,c);
				} catch (ExceptionInInitializerError e) {
//					DBG.warning("initialization error loading class "+classname+" "+e);
					error = e.toString();
				} catch (LinkageError e) {
//					DBG.warning("linkage error loading class "+classname+" "+e);
					error = e.toString();
				} catch (ClassNotFoundException e) {
//					DBG.warning("class "+classname+" not found "+e);
					error = e.toString();
				}
			}
			if (c == null) {
//				DBG.info("WSIF Format Type codec trying to load class "+classname+" via own classloader to get field types");
				try {
					c = getClass().getClassLoader().loadClass(classname);
//					DBG.info("WSIF Format Type codec loaded class "+classname+" OK");
					classCache.put(classname,c);
				} catch (ExceptionInInitializerError e) {
//					DBG.warning("initialization error loading class "+classname+" "+e);
					error = e.toString();
				} catch (LinkageError e) {
//					DBG.warning("linkage error loading class "+classname+" "+e);
					error = e.toString();
				} catch (ClassNotFoundException e) {
//					DBG.warning("class "+classname+" not found "+e);
					error = e.toString();
				}
			}
			if (c == null) {
//				DBG.info("WSIF Format Type codec trying to load class "+classname+" via JAR dependancy ClassLoader to get field types");
				try {
					c = dependancy_loader.loadClass(classname);
//					DBG.info("WSIF Format Type codec loaded class "+classname+" OK");
					classCache.put(classname,c);
				} catch (ExceptionInInitializerError e) {
//					DBG.warning("initialization error loading class "+classname+" "+e);
					error = e.toString();
				} catch (LinkageError e) {
//					DBG.warning("linkage error loading class "+classname+" "+e);
					error = e.toString();
				} catch (ClassNotFoundException e) {
//					DBG.warning("class "+classname+" not found "+e);
					error = e.toString();
				}
			}
			
			if (c != null) {
//				DBG.info("WSIF Format Type codec trying to get field "+fieldName+"'s type");
				
				Field field = c.getDeclaredField(fieldName);
				if (field == null) {
//					DBG.error("WSIF Format Type codec field "+fieldName+" was null");
				}
				
				return field;
			} else {
				error = "failed to load class "+classname+" to search field types";
			}
//		} catch (NullPointerException e) {
//			DBG.error("WSIF Format Type codec failed to get type of class field - no field named "+fieldName+"?",e);
//			error = e.toString();
		} catch (NoSuchFieldException e) {
//			DBG.error("WSIF Format Type codec failed to get type of class field - no field named "+fieldName+"?",e);
			error = e.toString();
		} catch (SecurityException e) {
//			DBG.error("WSIF Format Type codec failed to access class "+classname+" field "+fieldName+" because of security",e);
			error = e.toString();
		} catch (Throwable e) {
//			DBG.error("WSIF Format Type codec unknown error while trying to get type of class field",e);
			error = e.toString();
		}
		throw new XSDTypeTranslatorException("problem searching class for field types to determine mapping - "+error);
	}
	
}