/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.message;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * @author amiguel
 * 
 * Utility class for the message api
 * users should use MessageReader/MessageWriter instead of this class
 */
public class MessageUtils implements MessageConstants {

//////////////////////////////////////////////////////
// Methods to convert messages to and from bytes
//////////////////////////////////////////////////////

/**
 * Convert a message to bytes
 * @param m the message to convert
 * @return the byte array representation of this message
 */
public static byte[] messageToBytes(Message m) {
	ByteArrayOutputStream bout = new ByteArrayOutputStream();
	
	try {
		writeMessage(bout,m);
	} catch (IOException e) {
		e.printStackTrace();
//		System.exit(0);
	}//end try catch
	
	return bout.toByteArray();
}//end method

/**
 * Convert a byte array to a message
 * @param b the byte array to convert
 * @return the message converted from the byte array
 */
public static Message bytesToMessage(byte[] b) {
	ByteArrayInputStream bin = new ByteArrayInputStream(b);
	Message m = null;
	
	try {
		m = readMessage(bin);
	} catch (IOException e) {
		e.printStackTrace();
//		System.exit(0);
	}//end try catch
		
	return m;
}//end method

//////////////////////////////////////////////////////
// Methods to read messages from and write messages to I/O streams
//////////////////////////////////////////////////////

/**
 * Write a message to an output stream
 * @param bout the output stream to write to
 * @param m the message to write
 */
public static void writeMessage(OutputStream bout, Message m) throws IOException {
				
	int len = m.length();
		
	//
	// Write Header
	//
	writeInt(bout,MESSAGE_START);
	writeInt(bout,m.getType());
	
	//
	// Write Message Data
	//
	if (m.callstack != null) {
		writeInt(bout,len+1);//extra length - for the callstack
	} else {
		writeInt(bout,len);
	}

	
	Object o;
	int typ;	
	for (int i = 0; i < len; i++) {
		o = m.get(i);
		typ = m.getType(i);

		writeInt(bout, typ);
				
		switch (typ) {
		
		case TYPE_STRING:
			String s = (String)o;
			if (s == null) {
				writeInt(bout, NULL_STRING);
			} else {
				byte[] dat = s.getBytes();
				writeInt(bout, dat.length);
				bout.write(dat);
			}
			
			break;
			
		case TYPE_INTEGER:
			Integer xi = (Integer)o;
			writeInt(bout, xi.intValue());
			
			break;
			
		case TYPE_LONG:
			Long xl = (Long)o;
			writeLong(bout, xl.longValue());
			
			break;
			
		case TYPE_DOUBLE:
			Double xd = (Double)o;
			writeDouble(bout, xd.doubleValue());
		
			break;

		case TYPE_SUBMESSAGE:
			Message xm = (Message)o;
			writeMessage(bout,xm);

			break;
					
		case TYPE_BYTE_ARRAY:
			byte[] b = (byte[])o;
			writeInt(bout, b.length);
			bout.write(b);
	
			break;
			
		}//end switch
		
	}//end for
	
	if (m.callstack != null) {
		writeInt(bout, TYPE_CALLSTACK);
		byte[] dat = m.callstack.toString().getBytes();
		writeInt(bout, dat.length);
		bout.write(dat);
	}
	
	//
	// Write Footer
	//
	writeInt(bout,MESSAGE_END);

	bout.flush();
		
}//end method

/**
 * Read a message from an input stream
 * @param bin the input stream to read from
 * @return the message to read
 */
public static Message readMessage(InputStream bin) throws IOException {

	Message m = new Message();
	int tmpi;
	long tmpl;
	double tmpd;
	byte[] tmpb;
	Message tmpmsg;

	//
	// Read Header
	//
	tmpi = readInt(bin);
	if (tmpi != MESSAGE_START) {
		throw new IOException("Invalid Message Start "+tmpi);
	}//end if

	//
	// Read type
	//
	tmpi = readInt(bin);
	m.setType(tmpi);
	
	//
	// Read Message Data
	//
	int datalen = readInt(bin);
	
	for (int i = 0; i < datalen; i++) {
		int dtype = readInt(bin);
		
		switch (dtype) {
		
		case TYPE_STRING:
			tmpi = readInt(bin);
			if (tmpi == NULL_STRING) {
				m.append( (String)null );
			} else {
				tmpb = readBytes(bin,tmpi);
				m.append( new String(tmpb,0,tmpi) );
			}
				
			break;
		
		case TYPE_INTEGER:
			tmpi = readInt(bin);
			m.append( new Integer(tmpi) );
			
			break;
				
		case TYPE_LONG:
			tmpl = readLong(bin);
			m.append( new Long(tmpl) );
			
			break;

		case TYPE_DOUBLE:
			tmpd = readDouble(bin);
			m.append( new Double(tmpd) );
		
			break;

		case TYPE_SUBMESSAGE:
			tmpmsg = readMessage(bin);
			m.append( tmpmsg );
		
			break;
		
		case TYPE_CALLSTACK:
			tmpi = readInt(bin);
			tmpb = readBytes(bin,tmpi);
			m.callstack = new StringBuffer(new String(tmpb,0,tmpi));
				
			break;

		default:
			tmpi = readInt(bin);
			tmpb = readBytes(bin,tmpi);
			m.append( tmpb, 0, tmpi );
					
		}//end switch
		
	}//end for
	
	//
	// Read Footer
	//
	tmpi = readInt(bin);
	if (tmpi != MESSAGE_END) {
		throw new IOException("Invalid Message End "+tmpi);
	}//end if
		
	return m;
}//end method



///////////////////////////////////////////////////
// Private utility methods
///////////////////////////////////////////////////

private static void writeInt(OutputStream o, int i) throws IOException {
	byte[] b = new byte[4];
	b[0] = (byte) (i>>24);
	b[1] = (byte) (i>>16);
	b[2] = (byte) (i>>8);
	b[3] = (byte) (i);
	o.write(b);
}//end method
private static void writeLong(OutputStream o, long i) throws IOException {
	byte[] b = new byte[8];
	b[0] = (byte) (i>>56);
	b[1] = (byte) (i>>48);
	b[2] = (byte) (i>>40);
	b[3] = (byte) (i>>32);
	b[4] = (byte) (i>>24);
	b[5] = (byte) (i>>16);
	b[6] = (byte) (i>>8);
	b[7] = (byte) (i);
	o.write(b);
}//end method
private static void writeDouble(OutputStream o, double i) throws IOException {
	long il = Double.doubleToRawLongBits(i);
	writeLong(o,il);
}//end method

private static int readInt(InputStream i) throws IOException {
	byte[] b = readBytes(i,4);
	
	int x = (((int)(b[0] & 0xff) << 24) |
			 ((int)(b[1] & 0xff) << 16) |
			 ((int)(b[2] & 0xff) << 8) |
			 ((int)(b[3] & 0xff)));
	
	return x;
}//end method
private static long readLong(InputStream i) throws IOException {
	byte[] b = readBytes(i,8);
	
	long x = (((long)(b[0] & 0xff) << 56) |
			 ((long)(b[1] & 0xff) << 48) |
			 ((long)(b[2] & 0xff) << 40) |
			 ((long)(b[3] & 0xff) << 32) |
			 ((long)(b[4] & 0xff) << 24) |
			 ((long)(b[5] & 0xff) << 16) |
			 ((long)(b[6] & 0xff) << 8) |
			 ((long)(b[7] & 0xff)));
	
	return x;
}//end method
private static double readDouble(InputStream i) throws IOException {
	long x = readLong(i);
	return Double.longBitsToDouble(x);
}//end method
private static byte[] readBytes(InputStream i, int len) throws IOException {
	byte[] b = new byte[len];
	int t = 0;
	int n = 0;
	
	while (t < len) {
		n = i.read(b,t,len-t);
		if (n == -1) {
			throw new IOException("End of Stream");
		} else {
			t+=n;
		}//end if
	}//end while

	return b;
}//end method

}
