/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.mutex;

import java.util.HashMap;
import java.util.LinkedList;

/**
 * @author amiguel
 * 
 * A message passing class used in the local engine implementation
 */
public class QueuedBlockingMap {

///////////////////////////////////////////////////////////
//
// internal single mutex implementation
//
	private Object mutex_LOCK = new Object();
	private int mutex_val = 1;
	
	private void mutex_lock() throws InterruptedException {
		synchronized(mutex_LOCK) {
			mutex_val--;
			if (mutex_val < 0) {
				mutex_LOCK.wait();
			}	
		}
	}
	private void mutex_release() {
		synchronized(mutex_LOCK) {
			mutex_val++;
			if (mutex_val > 1) mutex_val = 1; 
			if (mutex_val <= 0) mutex_LOCK.notify();
		}
	}
//
///////////////////////////////////////////////////////////

	
HashMap pool = new HashMap();

	private void removeLock(Object key) {
		pool.remove(key);
	}

	private Lock getLock(Object key) {
		Lock notifier;
		
		notifier = (Lock)pool.get(key);
		if (notifier == null) {
			notifier = new Lock();
			pool.put(key,notifier);
		}
		
		return notifier;
	}
	
	private void removeRecipient(Recipient recipient) {
		Object key = recipient.key;
		Lock tmp = getLock(key);

		//remove the recipient from conversation [i]
		tmp.recipients.remove(recipient);
		
		if (tmp.isEmpty()) {
			removeLock(key);
		}
	}
	
	//non blocking put
	public void put(String key, Object value) throws InterruptedException {

		//XXX LOCK WORLD
		mutex_lock();
		
		Lock notifier = getLock(key);

		if (notifier.recipients.size() > 0) {
			//something is waiting for a message already
			
			Recipient recipient = (Recipient)notifier.recipients.getFirst();
			Object keys = recipient.key;
			
			synchronized(recipient) {

				//set the recipient's value field
				recipient.value = value;
			
				//We need to remove it now otherwise another PUT might come along and override us
				removeRecipient(recipient);
				
				//XXX UNLOCK WORLD
				mutex_release();

				recipient.notify();
			}
			
		} else {
			//nothing is waiting for a message, queue it
			notifier.messages.addLast(value);

			//XXX UNLOCK WORLD
			mutex_release();
		}
		
	}
	
	public Object get(String keylist) throws InterruptedException {
		return get(keylist,0);
	}
	
	public Object get(String keylist, long timeout) throws InterruptedException {
		
		//TODO could get this from a pool rather than suffering object creation/deletion all the time
		Recipient recipient = new Recipient();
		recipient.key = keylist;
		recipient.value = null;

		boolean timed_out = false;

		//XXX LOCK WORLD
		mutex_lock();

		Lock lock = getLock(keylist);
		
		if (lock.messages.size() > 0) {
			//get (and remove) the first message in this queue
			recipient.value = lock.messages.removeFirst();

			//if the lock is empty remove it from the map
			if (lock.isEmpty()) {
				removeLock(keylist);
			}
		}

		if (recipient.value == null) {
			//we found no existing message, we'll have to wait for one
			
			//wait on our notification object (Recipient)
			synchronized(recipient) {
				
				//add ourselves as a listener on each lock
				lock = getLock(keylist);
				lock.recipients.add(recipient);
			
				//XXX UNLOCK WORLD
				mutex_release();
				recipient.wait(timeout);
			}
			
			//We do this rather than just locking mutex_lock inside the sync(recipient) block
			//because we MUST always lock them in the same order otherwise we could get deadlock

			//XXX LOCK WORLD
			mutex_lock();
				synchronized(recipient) {
					
					//If we timed out then we must remove ourselves from all conversations
					if (recipient.value == null) {
						timed_out = true;
						removeRecipient(recipient);
					}
				}
			//XXX UNLOCK WORLD
			mutex_release();
			
		} else {
			//XXX UNLOCK WORLD
			mutex_release();
		}
		
		if (timed_out) {
			throw new InterruptedException("Timed out ("+timeout+"ms)");
		}

		return recipient.value;		
	}
	
	class Lock {
		LinkedList messages = new LinkedList();
		LinkedList recipients = new LinkedList();
		
		public boolean isEmpty() {
			return messages.size() == 0 && recipients.size() == 0;
		}
	}	

	class Recipient {
		Object key;
		Object value;
	}
}
