/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.transport.session;

/**
 * 
 * @author amiguel
 *
 * A statically allocated circular buffer for primitive shorts
 */
public class CircularShortBuffer {

int ptr_start = 0;
int ptr_end = 0;
int size = 0;
short[] buf;
	
	public CircularShortBuffer(int size) {
		buf = new short[size];
	}
	
	public int indexOf(short s) {
		for (int i = 0; i < size; i++) {
			if (get(i) == s) return i;
		}
		return -1;
	}
	
	public short get(int index) {
		if (index > size) throw new ArrayIndexOutOfBoundsException("index too high ("+index+" > "+size+")");
		if (ptr_start == ptr_end && size == 0) throw new ArrayIndexOutOfBoundsException("No elements in buffer");
		index = ptr_start + index;
		if (index >= buf.length) index -= buf.length;
		return buf[index];
	}
	
	public void add(short s) {
		if (ptr_start == ptr_end && size == buf.length) {
			throw new ArrayIndexOutOfBoundsException("No room left in circular buffer");
		}
		
		buf[ptr_end++] = s;
		if (ptr_end == buf.length) {
			ptr_end = 0;
		}
		
		size++;
	}
	
	public int size() {
		return size;
	}
	
	public void discard(int count) {
		if (count > size) {
			throw new ArrayIndexOutOfBoundsException("Cannot discard more than entirity of buffer ("+size+") elements");
		}
		
		ptr_start += count;
		if (ptr_start >= buf.length) {
			ptr_start -= buf.length;
		}
		
		size -= count;
	}
	
	public static void main(String[] args) {
		CircularShortBuffer cb = new CircularShortBuffer(100);
		
		for (int i = 0; i < 100; i++) {
			cb.add((short)i);
		}
		System.out.println("Added 100 elements OK");
		
		if (cb.size() != 100) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}

		System.out.println("Size OK");
		
		for (int i = 0; i < 100; i++) {
			short s = cb.get(i);
			if (i != s) {
				new Exception("MISMATCH").printStackTrace();
				System.exit(0);
			}
		}

		System.out.println("Checked 100 elements OK");
		
		if (cb.size() != 100) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}

		System.out.println("Size OK");
		
		try {
			cb.add((short)101);
			new Exception("Upper limit failed").printStackTrace();
			System.exit(0);
		} catch (Exception e) {
		}

		System.out.println("Upper limit OK");
		
		cb.discard(50);

		System.out.println("Discarded 50 elements");
		
		if (cb.size() != 50) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}
		
		System.out.println("Size OK");

		for (int i = 0; i < 50; i++) {
			short s = cb.get(i);
			if ((50 + i) != s) {
				new Exception("MISMATCH").printStackTrace();
				System.exit(0);
			}
		}
		
		System.out.println("Checked 50 elements OK");

		cb.discard(50);

		System.out.println("Discarded 50 elements");
		
		if (cb.size() != 0) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}
		
		System.out.println("Size OK");

		try {
			cb.get(0);
			new Exception("Should be empty but returned value").printStackTrace();
			System.exit(0);
		} catch (Exception e) {
		}

		System.out.println("Emptiness OK");
		
		for (int i = 0; i < 40; i++) {
			cb.add((short)i);
		}

		System.out.println("Added 40 elements OK");
		
		if (cb.size() != 40) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}
		
		System.out.println("Size OK");

		for (int i = 0; i < 40; i++) {
			short s = cb.get(i);
			if (s != i) {
				new Exception("MISMATCH").printStackTrace();
				System.exit(0);
			}
		}

		System.out.println("Checked 40 elements OK");
		
		cb.discard(40);
		
		System.out.println("Discarded 40 elements");

		if (cb.size() != 0) {
			new Exception("Incorrect size reported").printStackTrace();
			System.exit(0);
		}
		
		System.out.println("Size OK");
	}
	
}