/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;

/**
 * @author amiguel
 * 
 * Used to serialise common Java types
 */
public class DataTransfer {

	private static final byte INTEGER = 1;
	private static final byte SHORT = 2;
	private static final byte BYTE = 3;
	private static final byte DOUBLE = 4;
	private static final byte FLOAT = 5;
	private static final byte BYTES = 6;
	private static final byte STRING = 7;
	private static final byte LONG = 8;

	private DataTransfer() {
	}

	private static void writeObject(Object o, OutputStream bout) throws IllegalArgumentException, IOException {
		if (o instanceof Number) {
			Number n = (Number)o;
			if (o instanceof Integer) {
				bout.write(INTEGER);
				StreamUtils.writeInt(bout,n.intValue());
			} else if (o instanceof Long) {
				bout.write(LONG);
				StreamUtils.writeLong(bout,n.longValue());
			} else if (o instanceof Double) {
				bout.write(DOUBLE);
				StreamUtils.writeDouble(bout,n.doubleValue());
			} else if (o instanceof Short) {
				bout.write(SHORT);
				StreamUtils.writeShort(bout,n.shortValue());
			} else if (o instanceof Float) {
				bout.write(FLOAT);
				StreamUtils.writeFloat(bout,n.floatValue());
			} else if (o instanceof Byte) {
				bout.write(BYTE);
				bout.write(n.byteValue());
			} else {
				throw new IllegalArgumentException("unserializable argument "+o);
				//unrecognised	
			}
		} else if (o instanceof byte[]) {
				bout.write(BYTES);
				StreamUtils.writeBytes(bout,(byte[])o);
		} else if (o instanceof String) {
				bout.write(STRING);
				StreamUtils.writeString(bout,(String)o);
		} else {
			throw new IllegalArgumentException("unserializable argument "+o);
			//unrecognised
		} 
	}

	public static byte[] serialise(ArrayList args) throws IllegalArgumentException, IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		bout.reset();
		StreamUtils.writeInt(bout,args.size());
		for (int i = 0; i < args.size(); i++) {
			writeObject( args.get(i), bout );
		}		
		return bout.toByteArray();
	}
	
	public static byte[] serialise(Object[] args) throws IllegalArgumentException, IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		StreamUtils.writeInt(bout,args.length);
		for (int i = 0; i < args.length; i++) {
			writeObject( args[i], bout );
		}		
		return bout.toByteArray();
	}
	
	public static ArrayList deserialise(byte[] dat) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(dat);
		return deserialise(bin);
	}
	
	public static ArrayList deserialise(InputStream bin) throws IOException {
		ArrayList args = new ArrayList();

		int count = StreamUtils.readInt(bin);
		
		for (int i = 0; i < count; i++) {
			int t = bin.read();
			byte b = (byte)t;
Logger.info(t+" "+b);			

			switch(b) {
				case INTEGER:
					args.add(new Integer(StreamUtils.readInt(bin)));
				break;
				case STRING:
					args.add(StreamUtils.readNString(bin,100000000));
				break;
				case BYTES:
					args.add(StreamUtils.readNBytes(bin,100000000));
				break;
				case LONG:
					args.add(new Long(StreamUtils.readLong(bin)));
				break;
				case DOUBLE:
					args.add(new Double(StreamUtils.readDouble(bin)));
				break;
				case SHORT:
					args.add(new Short(StreamUtils.readShort(bin)));
				break;
				case BYTE:
					args.add(new Byte((byte)bin.read()));
				break;
				case FLOAT:
					args.add(new Float(StreamUtils.readFloat(bin)));
				break;
				default:
					throw new IOException("unrecognised deserialisation type "+b);
			}

			t = bin.read();						
		}
		
		return args;
	}
}