/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.utils;

import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.HashMap;

/**
 * @author amiguel
 * 
 * Provides logging for the engine
 * 
 * Provides debugging output, verbosity levels info, warning and error
 *
 * NB: to get the most out of the debugger you should set the name of
 *     each thread:
 *           Thread.currentThread().setName("Server 1");
 *
 */
public class Logger {

///////////////////////////////////////////////////////////////////////
// Settings
//
	// Print information messages
	// Print warning messages
	// Print error messages
	public static boolean PRINT_INFO = false;
	public static boolean PRINT_WARNING = true;
	public static boolean PRINT_ERROR = true;

	// Log everything to the cli
	// Create a log file per thread
	public static boolean LOG_TO_CLI = true;
	public static boolean LOG_TO_FILE = true;

	// Print the stack trace if it is given
	// Generate a stack trace for all warnings and errors
	public static boolean PRINT_STACKTRACE = true;
	public static boolean GENERATE_STACKTRACE = false;

//
///////////////////////////////////////////////////////////////////////

	private static HashMap log_map = new HashMap();
	private static HashMap mark_map = new HashMap();

	private static Object chelog_lock = new Object();
	private static FileOutputStream chelog = null;
	
	public static void log (String message) {

		if (LOG_TO_CLI) {
			if (message.charAt(message.length()-1) != '\n') {
				System.err.println(message);
			} else {
				System.err.print(message);	
			}
		}//end if
		if (LOG_TO_FILE) {
			//note that this will be first created by the Daemon and then the others won't be able to create it
			synchronized(chelog_lock) {
				try {
					if (chelog == null) {
							chelog = new FileOutputStream("b2j.log");
					}
					chelog.write(message.getBytes());
					chelog.write((byte)'\r');
					chelog.write((byte)'\n');
					chelog.flush();
				} catch (Exception e) {
					LOG_TO_FILE = false;
					System.err.println("Unable to open b2j.log (this is probably ok, only Daemon should open it)");
				}
			}
		}
		
	}//end method

	/**
	 * Print some text - this method is used for timing
	 * @param textToPrint the text to print
	 */
	public static void mark( String textToPrint ) {
		if (PRINT_INFO) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			
			long tmp = 0;
			try {
				long now = System.currentTimeMillis();
				Long mark = (Long) mark_map.get(thread);
				if (mark == null) {
					mark = new Long(System.currentTimeMillis());
				}//end if
				tmp = now - mark.longValue();
				mark_map.put( thread , new Long(now) );
				
			} catch (Exception e) {}
			
			String message = "M:("+thread+"):("+tmp+"ms):"+caller+":"+textToPrint+"\n";
			log(message);
		}//end if
	}//end method

	/**
	 * Print a message that should always appear
	 * @param textToPrint the text to print
	 */
	public static void direct( String textToPrint ) {
		//direct message - always print
		String caller = getCallerStackTrace();
		String thread = Thread.currentThread().getName();
		String message = "D:("+thread+"):"+caller+":"+textToPrint+"\n";
		log(message);
	}//end method	
	
	/**
	 * Print something at the information level
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) to print the stacktrace of
	 */
	public static void info( String textToPrint, Throwable t ) {
		if (PRINT_INFO) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "I:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			log(message);
		}//end if
	}//end method

	/**
	 * Print something at the information level
	 * @param textToPrint the text to print
	 */
	public static void info( String textToPrint ) {
		if (PRINT_INFO) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "I:("+thread+"):"+caller+":"+textToPrint+"\n";
			log(message);
		}//end if
	}//end method

	/**
	 * Print some text as a warning
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) which pertains to this warning
	 */
	public static void warning( String textToPrint, Throwable t ) {
		if (PRINT_WARNING) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "W:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			log(message);
		}//end if
	}//end method
	
	/**
	 * Print some text as a warning
	 * @param textToPrint the text to print
	 */
	public static void warning( String textToPrint ) {
		if (PRINT_WARNING) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "W:("+thread+"):"+caller+":"+textToPrint+"\n";
			if (GENERATE_STACKTRACE) {
				message = message + getStackTrace();
			}//end if
			log(message);
		}//end if
	}//end method

	/**
	 * Print some text as an error
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) which pertains to this error
	 */
	public static void error( String textToPrint, Throwable t ) {
		if (PRINT_ERROR) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "E:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			log(message);
		}//end if
	}//end method

	/**
	 * Print some text as an error
	 * @param textToPrint the text to print
	 */
	public static void error( String textToPrint ) {
		if (PRINT_ERROR) {
			String caller = getCallerStackTrace();
			String thread = Thread.currentThread().getName();
			String message = "E:("+thread+"):"+caller+":"+textToPrint+"\n";
			if (GENERATE_STACKTRACE) {
				message = message + getStackTrace();
			}//end if
			log(message);
		}//end if
	}//end method

	/**
  	 * Get the line of the stack trace pertaining to the object calling this Debug method
	 */
	private static String getCallerStackTrace() {
		String ret = StackTracer.getCallerTrace(3);
		if (ret == null) {
			Throwable t = new Throwable();
			ret = getLineOfStackTrace(t, getStackTrace(t), 4);
		}
		return ret;
	}//end method

	/**
  	 * Get a specific line of a stack trace without the "   at " and the newline after it
	 */
	private static String getLineOfStackTrace( Throwable t, String stackTrace, int lineNumber ) {
		int beginIndex = 0;
		int endIndex = 0;

		String cname = t.getClass().getName();

		//jump one line down from the last occurrence of the exception class
		int lastindex = stackTrace.lastIndexOf(cname);
		if (lastindex == -1) {
			lastindex = 0;	
		} else {
			lastindex += cname.length();	
		}

		//find string based on newline chars
		int index = lastindex;
		for (int i = 0; i < lineNumber; i++) {
			beginIndex = index+5; // get rid of the "    at "
			index = stackTrace.indexOf('\n',index+1);
			if (stackTrace.charAt(index-1) == '\n' || stackTrace.charAt(index-1) == '\r') {
				endIndex = index-1;  // get rid of the newline
			} else {
				endIndex = index;
			}
		}//end for

		try {
			return stackTrace.substring( beginIndex, endIndex );
		} catch (Exception e) {
			return "unknown-error";
		}
	}//end method
/*	private static String getLineOfStackTrace( String stackTrace, int lineNumber ) {
		int beginIndex = 0;
		int endIndex = 0;

		//find string based on newline chars
		int index = 0;
		for (int i = 0; i < lineNumber; i++) {
			beginIndex = index+5; // get rid of the "    at "
			index = stackTrace.indexOf("\n",index+1);
			endIndex = index-1;  // get rid of the newline
		}//end for

		return stackTrace.substring( beginIndex, endIndex );
	}//end method
*/
	/**
	 * Returns a stack trace for the Throwable (Exception)
	 * @param t the Throwable (Exception) to get the stack trace for
	 * @return a string representation of a stack trace for the Throwable
	 */
	public static String getStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method

	/**
	 * Returns a stack trace for the Throwable (Exception)
	 * @param t the Throwable (Exception) to get the stack trace for
	 * @return a string representation of a stack trace for the Throwable
	 */
	public static String getStackTrace() {
		Throwable t= new Throwable("DEBUGGER GENERATED STACKTRACE");
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method

	public static void main(String[] args) {	
		testStackTrace();
	}
	
	public static void testStackTrace() {
		System.out.println(getStackTrace());
		direct("CORRECT STACK TRACE?");
	}

}//end class
