/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.utils;

import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Iterator;

/**
 * @author amiguel
 * 
 * A class to read and write common data to any type of OutputStream or 
 * InputStream.
 *
 * All static methods.
 *
 * No buffering, everything writes/reads directly to/from stream.
 */
public class StreamUtils {

private static final byte CLASS_INT = 0;
private static final byte CLASS_LONG = 1;
private static final byte CLASS_BOOLEAN = 3;
private static final byte CLASS_DOUBLE = 4;
private static final byte CLASS_FLOAT = 5;
private static final byte CLASS_COLLECTION = 6;
private static final byte CLASS_STRING = 7;
private static final byte CLASS_CHAR = 8;

	/**
	 * A simple method for reading objects from a stream written by writeObject()
	 * @param in the InputStream to read the object from
	 * @return the Object read from the stream
	 */
	public static Object readObject(InputStream in) throws Exception {
		Object o;
		String claz = readString(in);
		
		Class clazz = Class.forName(claz);
		o = clazz.newInstance();		//recurse right up to Object() for this? may not work

		int fieldcount = readInt(in);
		
		Field[] fields = clazz.getDeclaredFields();
		
		for (int i = 0; i < fieldcount; i++) {

			String fname = readString(in);
			Field field = null;
			
			for (int f = 0; f < fields.length; f++) {
				if (fields[f].getName().equals(fname)) {
					field = fields[f];	
					break;
				}
			}
			
			if (field == null) throw new Exception("Field "+fname+" not found");
			
			field.setAccessible(true);
			
			int tmp = in.read();
			if (tmp == -1) throw new EOFException("End of stream");
			byte fclaz = (byte)tmp;
			
			if (fclaz == CLASS_INT) {
				field.setInt(o,readInt(in));

			} else if (fclaz == CLASS_LONG) {
				field.setLong(o,readLong(in));

			} else if (fclaz == CLASS_BOOLEAN) {
				field.setBoolean(o,readBoolean(in));

			} else if (fclaz == CLASS_DOUBLE) {
				field.setDouble(o,readDouble(in));

			} else if (fclaz == CLASS_FLOAT) {
				field.setFloat(o,readFloat(in));

			} else if (fclaz == CLASS_CHAR) {
				field.setChar(o,readChar(in));

			} else if (fclaz == CLASS_STRING) {
				field.set(o,readString(in));

			} else if (fclaz == CLASS_COLLECTION) {
				String colname = readString(in);
				Class colclazz = Class.forName(colname);
				Collection col = (Collection)colclazz.newInstance();

				int colsiz = readInt(in);

				for (int z = 0; z < colsiz; z++) {
					col.add(readString(in));
				}
				
				field.set(o,col);

			} else {
				throw new IllegalArgumentException("Field class "+fclaz+" not supported");
			}	
			
		}
			
		return o;
	}


	/**
	 * A simple method for writing objects to a stream
	 * The objects fields will be written to a file
	 * The objects fields must be primitive types or java.lang.String or java.Collection
	 * The contents of any java.lang.Collections must be java.lang.String
	 * Only fields in the objects immediate class will be written.  Fields of superclasses are ignored.
	 * @param out
	 * @param o
	 * @throws IOException
	 */
	public static void writeObject(OutputStream out, Object o) throws Exception {
		String claz = o.getClass().getName();
		writeString(out,claz);

		Class clazz = o.getClass();
		Field[] fields = clazz.getDeclaredFields();
		
		writeInt(out,fields.length);

		for (int i = 0; i < fields.length; i++) {
			fields[i].setAccessible(true);

			String fname = fields[i].getName();
			writeString(out,fname);
			
			Class fclazz = fields[i].getType();
			if (fclazz == int.class) {
				out.write(CLASS_INT);
				writeInt(out,fields[i].getInt(o));
				
			} else if (fclazz == long.class) {
				out.write(CLASS_LONG);
				writeLong(out,fields[i].getLong(o));
				
			} else if (fclazz == boolean.class) {
				out.write(CLASS_BOOLEAN);
				writeBoolean(out,fields[i].getBoolean(o));

			} else if (fclazz == double.class) {
				out.write(CLASS_DOUBLE);
				writeDouble(out,fields[i].getDouble(o));

			} else if (fclazz == float.class) {
				out.write(CLASS_FLOAT);
				writeFloat(out,fields[i].getFloat(o));

			} else if (fclazz == char.class) {
				out.write(CLASS_CHAR);
				writeChar(out,fields[i].getChar(o));

			} else if (fclazz == String.class) {
				out.write(CLASS_STRING);
				String s = (String)fields[i].get(o);
				writeString(out,s);

			} else if (Collection.class.isAssignableFrom(fclazz)) {
				out.write(CLASS_COLLECTION);
				writeString(out,fclazz.getName());
				Collection col = (Collection)fields[i].get(o);

				writeInt(out,col.size());
				
				Iterator it = col.iterator();
				
				while (it.hasNext()) {
					String s = (String)it.next();
					writeString(out,s);
				}
			} else {
				throw new IllegalArgumentException("Field class "+fclazz+" not supported");	
			}

		}
	
	}

	/**
	 * Read all the data from an input stream
	 * @param in the input stream to read from
	 * @return a String containing all the read data
	 */
	public static String readAllAsString(InputStream in) throws IOException {
		byte[] dat = readAll(in);
		return new String(dat,0,dat.length,"ISO-8859-1");
	}

	/**
	 * Read all the data from an input stream
	 * @param in the input stream to read from
	 * @return a byte array containing all the read data
	 */
	public static byte[] readAll(InputStream in) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		byte[] buf = new byte[65536];
		int n = 0;
		
		while (n != -1) {
			n = in.read(buf,0,65536);
			if (n > 0) {
				bout.write(buf,0,n);
			}
		}
		
		return bout.toByteArray();
	}

	/**
	 * Read all the data from an input stream and tolerate EOFExceptions
	 * @param in the input stream to read from
	 * @return a byte array containing all the read data
	 */
	public static byte[] readAllPossible(InputStream in) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		byte[] buf = new byte[65536];
		int n = 0;

		try {		
			while (n != -1) {
				n = in.read(buf,0,65536);
				if (n > 0) {
					bout.write(buf,0,n);
				}
			}
		} catch (EOFException e) {
			//ignore this
		}
		
		return bout.toByteArray();
	}

	/**
	 * Write some bytes to the output stream (NOTE: Not the same as OutputStream.write()).
	 * Note that this writes the length of the bytes too so the user can
	 * call readBytes() to read the entire byte array at the other end without
	 * having to worry about the length of the byte array
	 * @param out the OutputStream to write to
	 * @param b the byte array to write
	 */
	public static void writeBytes(OutputStream out, byte[] b) throws IOException {
		writeInt(out,b.length);
		out.write(b,0,b.length);
	}//end method

	/**
	 * Write some bytes to the output stream (NOTE: Not the same as OutputStream.write()).
	 * Note that this writes the length of the bytes too so the user can
	 * call readBytes() to read the entire byte array at the other end without
	 * having to worry about the length of the byte array
	 * @param out the OutputStream to write to
	 * @param b the byte array to write
	 */
	public static void writeBytes(OutputStream out, byte[] b, int off, int len) throws IOException {
		writeInt(out,len);
		out.write(b,off,len);
	}//end method

	/**
	 * Write a string to the output stream
	 * @param out the OutputStream to write to
	 * @param n the string to write
	 */
	public static void writeString(OutputStream out, String n) throws IOException {
		writeBytes(out,n.getBytes());
	}//end method

	/**
	 * Write a long to the output stream
	 * @param out the OutputStream to write to
	 * @param n the long to write
	 */
	public static void writeLong(OutputStream out, long n) throws IOException {
		
		out.write( (byte)(n >>> 56) );
		out.write( (byte)(n >>> 48) );
		out.write( (byte)(n >>> 40) );
		out.write( (byte)(n >>> 32) );
		out.write( (byte)(n >>> 24) );
		out.write( (byte)(n >>> 16) );
		out.write( (byte)(n >>> 8) );
		out.write( (byte)(n) );
		
	}//end method

	/**
	 * Write a double to the output stream
	 * @param out the OutputStream to write to
	 * @param n the double to write
	 */
	public static void writeDouble(OutputStream out, double n) throws IOException {
		
		writeLong(out,Double.doubleToLongBits(n));
		
	}//end method
		
	/**
	 * Write an int to the output stream
	 * @param out the OutputStream to write to
	 * @param n the int to write
	 */
	public static void writeInt(OutputStream out, int n) throws IOException {

		out.write( (byte)(n >>> 24) );
		out.write( (byte)(n >>> 16) );
		out.write( (byte)(n >>> 8) );
		out.write( (byte)(n) );
		
	}//end method

	/**
	 * Write a float to the output stream
	 * @param out the OutputStream to write to
	 * @param n the float to write
	 */
	public static void writeFloat(OutputStream out, float n) throws IOException {

		writeInt(out,Float.floatToIntBits(n));		
		
	}//end method

	/**
	 * Write a short to the output stream
	 * @param out the OutputStream to write to
	 * @param n the short to write
	 */
	public static void writeShort(OutputStream out, short n) throws IOException {
		
		out.write( (byte)(n >>> 8) );
		out.write( (byte)(n) );
		
	}//end method

	/**
	 * Write a char to the output stream
	 * @param out the OutputStream to write to
	 * @param v the char to write
	 */
	public static void writeChar(OutputStream out, char v) throws IOException {

		out.write((byte)(0xff & (v >> 8)));
		out.write((byte)(0xff & v));

	}

	/**
	 * Write a boolean to the output stream
	 * @param out the OutputStream to write to
	 * @param n the short to write
	 */
	public static void writeBoolean(OutputStream out, boolean n) throws IOException {

		if (n) {
			out.write(0xFF);	
		} else {
			out.write(0xEE);	
		}
		
	}//end method

	/**
	 * Read some bytes from an input stream.
	 * @param len the number of bytes to read
	 * @param in the input stream to read from
	 * @return a byte[]	containing the data read
	 */
	public static byte[] readBytes(InputStream in, int len) throws IOException {
		byte[] b = new byte[len];
		
		int red = 0;
		int tot = 0;
		while (tot < len) {
			red = in.read(b,tot,len-tot);
			if (red == -1) {
				throw new EOFException("End of stream");
			} else {
				tot+=red;
			}
		}
		
		return b;
	}//end method
	
	public static byte[] readLine(InputStream in) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		
		int last = -1;
		int c = in.read();
		
		if (c == -1) throw new IOException("End of stream");
		
		while (c != '\n' && c != -1) {
			
			if (last != -1) bout.write(last);
			last = c;
			
			c = in.read();
		}
		
		if (last != -1 && last != '\r') {
			bout.write(last);
		}
		
		return bout.toByteArray();
	}
	
	/**
	 * Read some bytes from an input stream (NOTE: Not the same as InputStream
	 * .read() or any normal byte reading)
	 * This expects the bytes to be in the format written by the writeBytes()
	 * method.
	 * @param in the input stream to read from
	 * @return a byte[]	containing the data read 
	 * @deprecated use the more secure readNBytes(InputStream in, int maxlen) method
	 */
	public static byte[] readBytes(InputStream in) throws IOException {
		int len = readInt(in);
		byte[] b = new byte[len];

		int red = 0;
		int tot = 0;
		while (tot < len) {
			red = in.read(b,tot,len-tot);
			if (red == -1) {
				throw new EOFException("End of stream");
			} else {
				tot+=red;
			}
		}
		
		return b;
	}//end method

	/**
	 * Read a string from an input stream
	 * @param in the input stream to read from
	 * @return the string read from the input stream
	 * @deprecated use the more secure readNString(InputStream in, int maxlen) method
	 */
	public static String readString(InputStream in) throws IOException {
		return new String(readBytes(in));
	}//end method

	/**
	 * Read some bytes from an input stream (NOTE: Not the same as InputStream
	 * .read() or any normal byte reading)
	 * This expects the bytes to be in the format written by the writeBytes()
	 * method.
	 * @param in the input stream to read from
	 * @param maxlen the maximum number of bytes to read
	 * @return a byte[]	containing the data read 
	 */
	public static byte[] readNBytes(InputStream in, int maxlen) throws IOException {
		int len = readInt(in);
		
		if (len < 0) {
			throw new IOException("Invalid data length specified ("+len+")");
		}
		if (len > maxlen) {
			throw new IOException("Unsafe data length specified ("+len+")");
		}
		
		byte[] b = new byte[len];
		
		int red = 0;
		int tot = 0;
		while (tot < len) {
			red = in.read(b,tot,len-tot);
			if (red == -1) {
				throw new EOFException("End of stream");
			} else {
				tot+=red;
			}
		}
		
		return b;
	}//end method

	/**
	 * Read a string from an input stream
	 * @param in the input stream to read from
	 * @param maxlen the maximum length of string to read
	 * @return the string read from the input stream
	 */
	public static String readNString(InputStream in, int maxlen) throws IOException {
		return new String(readNBytes(in,maxlen));
	}//end method


	/**
	 * Read a long from an input stream
	 * @param in the input stream to read from
	 * @return the long read from the input stream
	 */
	public static long readLong(InputStream in) throws IOException {
		long n = 0;
		int r = 0;
		
		for (int i = 0; i < 8; i++) {
			r = in.read();
			if (r == -1) throw new EOFException("End of stream");
			n = (n << 8) | r;
		}//end for
	
		return n;
	}//end method

	/**
	 * Read a double from an input stream
	 * @param in the input stream to read from
	 * @return the double read from the input stream
	 */	
	public static double readDouble(InputStream in) throws IOException {
		long l = readLong(in);
		return Double.longBitsToDouble(l);
	}//end method
	
	/**
	 * Read an int from an input stream
	 * @param in the input stream to read from
	 * @return the int read from the stream
	 */
	public static int readInt(InputStream in) throws IOException {
		int n = 0;
		int r = 0;
		
		for (int i = 0; i < 4; i++) {
			r = in.read();
			if (r == -1) throw new EOFException("End of stream");
			n = (n << 8) | r;
		}//end for
	
		return n;
	}//end method

	/**
	 * Read a float from an input stream
	 * @param in the input stream to read from
	 * @return the float read from the stream
	 */
	public static float readFloat(InputStream in) throws IOException {
		int i = readInt(in);
		return Float.intBitsToFloat(i);
	}//end method

	/**
	 * Read a short from an input stream
	 * @param in the input stream to read from
	 * @return the short read from the stream
	 */
	public static short readShort(InputStream in) throws IOException {
		short n = 0;
		int r = 0;
		
		for (int i = 0; i < 2; i++) {
			r = in.read();
			if (r == -1) throw new EOFException("End of stream");
			n = (short)((n << 8) | r);
		}//end for
	
		return n;
	}//end method

	/**
	 * Read a char from an input stream
	 * @param in the input stream to read from
	 * @return the char read from the stream
	 */
	public static char readChar(InputStream in) throws IOException {
		int a = in.read();
		int b = in.read();
		if (a == -1 || b == -1) throw new EOFException("End of stream");
		return (char)((a << 8) | (b & 0xff));
	}

	/**
	 * Read a boolean from an input stream
	 * @param in the input stream to read from
	 * @return the boolean read from the stream
	 */
	public static boolean readBoolean(InputStream in) throws IOException {
		
		int n = in.read();
		if (n == -1) throw new EOFException("End of stream");
		
		if (n == 0xFF) {
			return true;	
		} else {
			return false;
		}
	}//end method

}//end class
