/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.xpath.internal;
import java.util.ArrayList;

/////////////////////////////////////////////////////////////////
//	XPathVisitor Abstract Class
//	
//	The XPath Visitor class provides core method implementations
//	and abstract method definitions to be passed into the
//	XPath generated parser. Implement the abstract 
//  methods to interpret the parse tree as required.
//
//	Scapa Technologies
//	17/03/2004
/////////////////////////////////////////////////////////////////

public abstract class XPathVisitor
{
	// The elements stack is used to store lexical tokens
	// Elements are the stack are popped, processed and 
	// the result is pushed. Each token in an expression
	// equates to a single object on the stack.
	
	protected ArrayList elementStack;

	// Print out debug information
	
	protected boolean debug = false;

	// Constructor - assumed debug default
	
	public XPathVisitor()
	{
		if (debug==true) 
			System.out.println("Starting XPath Visitor");
			
		elementStack = new ArrayList();
	}

	// Debug Constructor

	public XPathVisitor(boolean debug)
	{
		if (debug==true) 
			System.out.println("Starting XPath Visitor");
			
		elementStack = new ArrayList();
		this.debug = debug;
	}

	// Stack Operations - push, pop and peek.

	public void push(Object o)
	{
		if (debug==true) 
			System.out.println("Pushed '"+o+"' " + o.getClass().toString());
			
		elementStack.add(o);
	}

	public Object pop()
	{
		if (elementStack.size() <= 0)
			return null;
			
		if (debug==true) 
			System.out.println("Popped '"+elementStack.get(elementStack.size()-1)+"'");

		return elementStack.remove(elementStack.size()-1);
	}

	public Object peek()
	{
		if (debug==true) 
			System.out.print("Peeking (size "+elementStack.size()+")\n");

		if (elementStack.size() <= 0)
			return null;

		return elementStack.get(elementStack.size()-1);
	}

	public void visit(String s)
	{
		if (debug==true) 
			System.out.println(s);
	}

	public void dumpStackContents()
	{
		for (int i = 0 ; i < elementStack.size();i++)
		{
			if (debug==true) System.out.print(""+elementStack.get(i));
		}
		if (debug==true) System.out.println("");
	}

	public abstract void prePredicateEval();

	public abstract void postPredicateEval();

	public abstract void processNumber();

	public abstract void processLiteral();

	public abstract void processSlash();

	public abstract void processAbsoluteSlash();

	public abstract void processAbbreviatedAbsoluteSlash();

	public abstract void processAbbreviatedRelativeSlash();

	public abstract void setContextNode();

	public abstract void evaluateAdditiveExpr();

	public abstract void evaluateMultiplicativeExpr();

	public abstract void evaluateUnaryExpr();

	public abstract void evaluateRelationalExpr();

	public abstract void evaluateEqualityExpr();

	public abstract void evaluateAndExpr();

	public abstract void evaluateOrExpr();

	public abstract void evaluateRelativePathPattern();

	public abstract void evaluateRelativeLocationPath();

	public abstract void evaluateStepPredicate();

	public abstract void evaluateQName();

	public abstract void evaluateQNameWONodeType();

	public abstract void evaluatePattern();

	public abstract void evaluateSingleIDKeyPattern();

	public abstract void evaluateDoubleIDKeyPattern();

	public abstract void evaluateVariableReference();

	public abstract void evaluateWildCardQName();

	public abstract void evaluateWildCardStar();

	public abstract void evaluateWildCardNCName();
	
	public abstract void evaluatePrimaryFilterExpr();
	
//	public abstract void evaluatePredicateFilterExpr();

	public abstract void evaluatePathExpr();
	
	public abstract void evaluateUnionExpr();

	public abstract void evaluateAbsoluteLocationPath();

	public abstract void evaluateProcessingInstruction();

	public abstract void evaluateAbbreviatedAbsoluteLocationPath();

	public abstract void evaluateAbbreviatedRelativeLocationPath();

	public abstract void evaluateLocationIDPathPattern();

	public abstract void evaluateLocationRelPathPattern();

	public abstract void evaluateLocationPathPattern();

	public abstract void evaluateFunctionCall();

	public abstract void evaluateNodeTypeTest();
	
	public abstract void evaluateNameTest();
	
	public abstract void evaluateAxisSpecifier();
	
	public void evaluateAbbreviatedAxisSpecifier()
	{
		Object axis = peek();
		if (axis == null) {
			push("child::");
		} else if (axis.equals("@")) {
			pop();
			push("attribute::");
		} else {
			push("child::");
		}
 		
		evaluateAxisSpecifier();
	}
	
}

