/******************************************************************************
 * Copyright (c) 2006, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************/

/** 
 * Date         	    Author              Changes 
 * 11  2006   	MPeleshchyshyn  	Created 
 **/

package org.eclipse.stp.bpmn.tools;

import java.lang.reflect.Field;

import org.eclipse.draw2d.PositionConstants;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionDimension;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.PrecisionRectangle;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.SnapToHelper;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.tools.ResizeTracker;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.stp.bpmn.Activity;
import org.eclipse.stp.bpmn.ActivityType;
import org.eclipse.swt.SWT;

/**
 * A Tracker for dragging a resize handle. The ResizeTracker will resize all of
 * the selected editparts in the viewer which understand a RESIZE request. A
 * {@link ChangeBoundsRequest} is sent to each member of the operation set. The
 * tracker allows for the resize direction to be specified in the constructor.
 * <p>
 * This can also be used for other shapes. It will use the custom handle figures
 * and allow resizing in any direction.
 * It is used by the bpmn text annotation.
 * </p>
 * 
 * @author mpeleshchyshyn
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class ActivityResizeTracker extends ResizeTracker {

    private static Field ownerField;

    private static Field sourceRectField;

    private static Field snapToHelperField;

    private static void init() {
        if (ownerField != null) {
            return;
        }
        try {
            ownerField = ResizeTracker.class.getDeclaredField("owner");
            ownerField.setAccessible(true);
            sourceRectField = ResizeTracker.class
                    .getDeclaredField("sourceRect");
            sourceRectField.setAccessible(true);
            snapToHelperField = ResizeTracker.class
                    .getDeclaredField("snapToHelper");
            snapToHelperField.setAccessible(true);
        } catch (SecurityException e) {
            e.printStackTrace();
        } catch (NoSuchFieldException e) {
            e.printStackTrace();
        }
    }

    /**
     * Constructs a resize tracker that resizes in the specified direction. The
     * direction is specified using {@link PositionConstants#NORTH},
     * {@link PositionConstants#NORTH_EAST}, etc.
     * 
     * @param owner
     *            of the resize handle which returned this tracker
     * @param direction
     *            the direction
     */
    public ActivityResizeTracker(GraphicalEditPart owner, int direction) {
        super(owner, direction);
        init();
    }

    protected GraphicalEditPart getOwner() {
        try {
            return (GraphicalEditPart) ownerField.get(this);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    protected SnapToHelper getSnapToHelper() {
        try {
            return (SnapToHelper) snapToHelperField.get(this);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    protected PrecisionRectangle getSourceRectangle() {
        try {
            return (PrecisionRectangle) sourceRectField.get(this);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * @see org.eclipse.gef.tools.SimpleDragTracker#updateSourceRequest()
     */
    protected void updateSourceRequest() {
        
        GraphicalEditPart owner = getOwner();
        if (owner != null && owner.getModel() instanceof View) {
                EObject semanticElement = ((View)owner.getModel()).getElement();
                if (!(semanticElement instanceof Activity) ||
                        ((Activity)semanticElement).getActivityType().getValue()
                                == ActivityType.TASK) {
                            //no resize constraint for a task.
                            //all the other ones must resize equally in height and width
                            super.updateSourceRequest();
                            return;                    
                }
        }
        
        
        ChangeBoundsRequest request = (ChangeBoundsRequest) getSourceRequest();
        Dimension d = getDragMoveDelta();

        Point location = new Point(getLocation());
        Point moveDelta = new Point(0, 0);
        Dimension resizeDelta = new Dimension(0, 0);
        int direction = getResizeDirection();

        PrecisionRectangle sourceRect = getSourceRectangle();
        SnapToHelper snapToHelper = getSnapToHelper();

        if (owner != null) {
            request.setConstrainedResize(true);

            final float ratio = 1;

            if (direction == PositionConstants.SOUTH_EAST) {
                if (d.height > (d.width * ratio))
                    d.width = (int) (d.height / ratio);
                else
                    d.height = (int) (d.width * ratio);
            } else if (direction == PositionConstants.NORTH_WEST) {
                if (d.height < (d.width * ratio))
                    d.width = (int) (d.height / ratio);
                else
                    d.height = (int) (d.width * ratio);
            } else if (direction == PositionConstants.NORTH_EAST) {
                if (-(d.height) > (d.width * ratio))
                    d.width = -(int) (d.height / ratio);
                else
                    d.height = -(int) (d.width * ratio);
            } else if (direction == PositionConstants.SOUTH_WEST) {
                if (-(d.height) < (d.width * ratio))
                    d.width = -(int) (d.height / ratio);
                else
                    d.height = -(int) (d.width * ratio);
            } else if (direction == PositionConstants.NORTH) {
                direction = direction | PositionConstants.EAST;
                d.width = -(int) (d.height / ratio);
            } else if (direction == PositionConstants.SOUTH) {
                direction = direction | PositionConstants.EAST;
                d.width = (int) (d.height / ratio);
            } else if (direction == PositionConstants.EAST) {
                direction = direction | PositionConstants.SOUTH;
                d.height = (int) (d.width * ratio);
            } else if (direction == PositionConstants.WEST) {
                direction = direction | PositionConstants.SOUTH;
                d.height = -(int) (d.width * ratio);
            }
        } else
            request.setConstrainedResize(false);

        request.setCenteredResize(getCurrentInput().isModKeyDown(SWT.MOD1));

        if ((direction & PositionConstants.NORTH) != 0) {
            if (getCurrentInput().isControlKeyDown()) {
                resizeDelta.height -= d.height;
            }
            moveDelta.y += d.height;
            resizeDelta.height -= d.height;
        }
        if ((direction & PositionConstants.SOUTH) != 0) {
            if (getCurrentInput().isControlKeyDown()) {
                moveDelta.y -= d.height;
                resizeDelta.height += d.height;
            }
            resizeDelta.height += d.height;
        }
        if ((direction & PositionConstants.WEST) != 0) {
            if (getCurrentInput().isControlKeyDown()) {
                resizeDelta.width -= d.width;
            }
            moveDelta.x += d.width;
            resizeDelta.width -= d.width;
        }
        if ((direction & PositionConstants.EAST) != 0) {
            if (getCurrentInput().isControlKeyDown()) {
                moveDelta.x -= d.width;
                resizeDelta.width += d.width;
            }
            resizeDelta.width += d.width;
        }

        request.setMoveDelta(moveDelta);
        request.setSizeDelta(resizeDelta);
        request.setLocation(location);
        request.setEditParts(getOperationSet());

        request.getExtendedData().clear();

        if (!getCurrentInput().isAltKeyDown() && snapToHelper != null) {
            PrecisionRectangle rect = sourceRect.getPreciseCopy();
            rect.translate(moveDelta);
            rect.resize(resizeDelta);
            PrecisionRectangle result = new PrecisionRectangle();

            snapToHelper.snapRectangle(request, request.getResizeDirection(),
                    rect, result);
            if (request.isCenteredResize()) {
                if (result.preciseX != 0.0)
                    result.preciseWidth += -result.preciseX;
                else if (result.preciseWidth != 0.0) {
                    result.preciseX = -result.preciseWidth;
                    result.preciseWidth *= 2.0;
                }

                if (result.preciseY != 0.0)
                    result.preciseHeight += -result.preciseY;
                else if (result.preciseHeight != 0.0) {
                    result.preciseY = -result.preciseHeight;
                    result.preciseHeight *= 2.0;
                }
                result.updateInts();
            }

            PrecisionPoint preciseMove = new PrecisionPoint(result.x
                    + moveDelta.x, result.y + moveDelta.y);

            PrecisionDimension preciseResize = new PrecisionDimension(
                    result.width + resizeDelta.width, result.height
                            + resizeDelta.height);

            if (preciseResize.width != preciseResize.height) {
                // make resize figure square
                preciseResize.preciseHeight = preciseResize.preciseWidth;
                preciseResize.updateInts();
            }
            request.setMoveDelta(preciseMove);
            request.setSizeDelta(preciseResize);
        }
    }
}
