/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.infrastructure.emf;

import java.util.Set;

import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * The IEditModelScribbler provides a mechanism to modify, save or revert a set
 * of EMF resources as a single atomic unit.
 * 
 * <p>
 * IEditModelScribblers describe the resources they are interested by one or
 * more IEdtingDomains. The set of IScribblerDomains are specified when the
 * IEditngModelScribbler is created from an
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModel#createScribbler(IScribblerDomain[], boolean)}.
 * The IEditModelScribbler may be used to create resources, acquire resources,
 * or delete resources.
 * </p>
 * <p>
 * The IEditModelScribbler also serves as the IUndoContext for any operations
 * that are executed against the
 * {@link org.eclipse.core.commands.operations.IOperationHistory}. The
 * IUndoContext is determined by the set of IScribblerDomains used to create the
 * IEditModelScribbler.
 * </p>
 * <p>
 * What this means is that if two IEditModelScribblers are created on the same
 * set of IScribblerDomains, the IEditModel will have a reference count for each
 * current IEditModelScribbler (remember there are two instances in this
 * example), and operations executed against one IEditModelScribbler (serving as
 * the IUndoContext) will be undoable in the "context" of the other
 * IEditModelScribbler.
 * </p>
 * <p>
 * This behavior is important for cases where an editor spawns a wizard or an
 * operation that creates an IEditModelScribbler on the same IScribblerDomain(s)
 * as the editor. The operation's IEditModelScribbler will be an equivalent
 * IUndoContext, so if the user requests an undo of the last undoable operation
 * just after the operation completes, then the IUndoContext of the editor will
 * match that of the operation, and the operation will be undone.
 * </p>
 */
public interface IEditModelScribbler extends IUndoContext, IAdaptable {

   int OPTION_NONE              = 0;

   int OPTION_LOAD_RESOURCE     = 2;

   int OPTION_LOAD_AS_READ_ONLY = 4;

   /**
    * Return an unmodifiable set of IScribblerDomains
    * 
    * <b>Implemented.</b>
    * 
    * @return An unmodifiable set of {@link IScribblerDomain}
    * 
    */
   Set getScribblerDomains();

   /**
    * Return a Resource based on an IPath. The Resource should already exist.
    * The path should be a fully-qualified workspace relative path to a file.
    * Using this method will always result in a Resource object being created,
    * even if no physical file exists for it on disk.
    * 
    * <b>Implemented.</b>
    * 
    * @param aUri
    * @return
    */
   Resource getResource(IPath aWorkspaceRelativePath);

   /**
    * Return a Resource based on an IPath. The Resource should already exist. If
    * {@link #OPTION_LOAD_RESOURCE} is used, then the a Resource object will
    * always be created, even if no physical file exists for it on disk.
    * 
    * <p>
    * The options could specify {@link #OPTION_LOAD_RESOURCE},
    * {@link #OPTION_NONE}, {@link #OPTION_LOAD_AS_READ_ONLY}. This pattern of
    * using an integer bitmask for extensibility and evolution of an API is
    * consistent with the Eclipse Platform.
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @param aDescriptor
    * @param options
    *           One of {@link #OPTION_LOAD_RESOURCE} or {@link #OPTION_NONE}
    * @return
    */
   Resource getResource(IPath aWorkspaceRelativePath, int options);

   /**
    * Return a Resource based on a URI. The Resource should already exist. Using
    * this method will always result in a Resource object being created, even if
    * no physical file exists for it on disk.
    * 
    * <b>Implemented.</b>
    * 
    * @param aUri
    * @return
    */
   Resource getResource(URI aUri);

   /**
    * Return a Resource based on a URI. The Resource should already exist. If
    * {@link #OPTION_LOAD_RESOURCE} is used, then the a Resource object will
    * always be created, even if no physical file exists for it on disk.
    * 
    * <p>
    * The options could specify {@link #OPTION_LOAD_RESOURCE},
    * {@link #OPTION_NONE}, {@link #OPTION_LOAD_AS_READ_ONLY}. This pattern of
    * using an integer bitmask for extensibility and evolution of an API is
    * consistent with the Eclipse Platform.
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @param aDescriptor
    * @param options
    *           One of {@link #OPTION_LOAD_RESOURCE} or {@link #OPTION_NONE}
    * @return
    */
   Resource getResource(URI aUri, int options);

   /**
    * Return a Resource based on a URI. The Resource should already exist. Using
    * this method will always result in a Resource object being created, even if
    * no physical file exists for it on disk.
    * 
    * <p>
    * Return a Resource described by the ResourceDescriptor. A standard
    * algorithm will be defined and documented in the javadoc of
    * ResourceDescriptor for how the search for resources should proceed.
    * </p>
    * 
    * <p>
    * By default, this method will attempt to load the resource (if exists). If
    * you do not want the resource loaded, use
    * {@link #getResource(IResourceDescriptor, int)} with
    * {@link #OPTION_LOAD_RESOURCE}. Using this method will always result in a
    * Resource object being created, even if no physical file exists for it on
    * disk.
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @param aDescriptor
    * @return
    * @see IResourceDescriptor For more information on how to acquire a
    *      ResourceDescriptor.
    */
   Resource getResource(IResourceDescriptor aDescriptor);

   /**
    * Return a Resource described by the ResourceDescriptor. The Resource should
    * already exist. If {@link #OPTION_LOAD_RESOURCE} is used, then the a
    * Resource object will always be created, even if no physical file exists
    * for it on disk.
    * 
    * <p>
    * The options could specify {@link #OPTION_LOAD_RESOURCE},
    * {@link #OPTION_NONE}, , {@link #OPTION_LOAD_AS_READ_ONLY}. This pattern
    * of using an integer bitmask for extensibility and evolution of an API is
    * consistent with the Eclipse Platform.
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @param aDescriptor
    * @param options
    *           One of {@link #OPTION_LOAD_RESOURCE} or {@link #OPTION_NONE}
    * @return
    */
   Resource getResource(IResourceDescriptor aDescriptor, int options);

   /**
    * Resources returned sorted by XMI first, then XML.
    * 
    * <b>NOT Implemented.</b>
    * 
    */
   Resource[] getSortedResources();

   /**
    * Creates a new Resource based on the ResourceDescriptor, adds it to the
    * underlying ResourceSet, indicates that it should be tracked by the
    * underlying IEditModel, and returns it for use.
    * 
    * <b>NOT Implemented.</b>
    * 
    * @param aResourceDescriptor
    * @return
    */
   Resource createResource(IResourceDescriptor aResourceDescriptor,
         IProgressMonitor monitor);

   /**
    * A deleted resource will only be removed from disk once save() is invoked.
    * If a call to deleteResource() is not followed at some point by a call to
    * save(), then the resource will not be deleted. Calling revert() will
    * forget the call to deleteResource().
    * 
    * 
    * <b>Implemented.</b>
    * 
    * @param aResourceDescriptor
    * @param monitor
    * @return
    */
   boolean deleteResource(IResourceDescriptor aResourceDescriptor,
         IProgressMonitor monitor) throws EditModelException;

   /**
    * A deleted resource will only be removed from disk once save() is invoked.
    * If a call to deleteResource() is not followed at some point by a call to
    * save(), then the resource will not be deleted. Calling revert() will
    * forget the call to deleteResource().
    * 
    * 
    * <b>Implemented.</b>
    * 
    * @param aResource
    * @param monitor
    * @return
    */
   boolean deleteResource(Resource aResource, IProgressMonitor monitor)
         throws EditModelException;

   /**
    * Close the resources described by this IEditModelScribbler's
    * IScribblerDomains. If there are unsaved changes in any of the resources,
    * the user will be prompted to save or discard those changes.
    * 
    * 
    * <b>PARTIALLY Implemented.</b>
    * 
    * @throws EditModelException
    */
   void close(IProgressMonitor monitor) throws EditModelException;

   /**
    * Discard changes in the resources described by this IEditModelScribbler's
    * IScribblerDomains. No changes will be persisted and the underlying
    * resources that have been modified will be reverted. This
    * IEditModelScribbler will no longer be usable after a call to discard().
    * 
    * <p>
    * This method will force requests to deleteResource() to be forgotten.
    * </p>
    * 
    * <b>NOT Implemented.</b>
    * 
    * @throws EditModelException
    */
   void discard(IProgressMonitor monitor) throws EditModelException;

   /**
    * Persist all changes from the set of resources described by the
    * IScribblerDomains. Changes in similar or related resources from the
    * underlying IEditModel not described by this IEditModelScribbler's
    * IScribblerDomains will not be persisted.
    * 
    * <p>
    * This method will also persist any requests to delete resource (from
    * deleteResource()).
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @param force
    *           Specifies whether the save should be forced if other
    *           IEditModelScribblers existing on the set of IScribblerDomains in
    *           use by this IEditModelScribbler.
    * @throws EditModelException
    */
   void save(boolean force, IProgressMonitor monitor) throws EditModelException;

   /**
    * Revert all of the resources described by the set of IScribblerDomains used
    * by this IEditModelScribbler.
    * 
    * <p>
    * This method will force requests to deleteResource() to be forgotten.
    * </p>
    * 
    * <b>Implemented.</b>
    * 
    * @throws EditModelException
    */
   void revert(IProgressMonitor monitor) throws EditModelException;

   /**
    * 
    * <b>Implemented.</b>
    * 
    * @return True if any of the interesting resources have been modified but
    *         not saved.
    */
   boolean isDirty();

   /**
    * 
    * <b>Implemented.</b>
    * 
    * @return True if the scribbler is loaded as read-only
    */
   boolean isReadOnly();

   /**
    * {@inheritDoc}
    * 
    * This IUndoContext is equivalent to all other IEditModelScribblers who
    * contain an equivalent set of IScribblerDomains.
    */
   boolean matches(IUndoContext context);

   /**
    * A UserInterfaceController should be provided that allows the
    * IEditModelScribbler to prompt the user for advice when necessary. The
    * IUserInterfaceController will be invoked to prompt the user when a
    * Resource inconsistency (between disk and memory) is detected, or when no
    * explicit save() is invoked and the IEditModelScribbler is close()ed.
    * 
    * 
    * <b>Implemented.</b>
    * 
    * 
    * @param userInterfaceController
    */
   void setUserInterfaceController(
         IHumanInteractionController userInterfaceController);

   /**
    * Return the IUserInterfaceController currently set. Only one
    * IUserInterfaceController may be set.
    * 
    * <b>Implemented.</b>
    * 
    * @return
    */
   IHumanInteractionController getUserInterfaceController();

   /**
    * Provides a convenience method for supplying and operation with the correct
    * context, executing an operation, and adding it to the IOperationHistory
    * managed by the IEditModel for this IEditModelScribbler.
    * 
    * <b>Implemented.</b>
    * 
    * @see org.eclipse.core.commands.operations.IOperationHistory#execute(org.eclipse.core.commands.operations.IUndoableOperation,
    *      org.eclipse.core.runtime.IProgressMonitor,
    *      org.eclipse.core.runtime.IAdaptable)
    */
   IStatus execute(IUndoableOperation operation, IProgressMonitor monitor,
         IAdaptable args);

   /**
    * Provides a convenience method for supplying and operation with the correct
    * context, executing an operation, and adding it to the IOperationHistory
    * managed by the IEditModel for this IEditModelScribbler.
    * 
    * <specific-request-for-input>
    * 
    * Should this method return false if it cannot succeed or throw an
    * Exception?
    * 
    * </specific-request-for-input>
    * 
    * <b>Implemented.</b>
    * 
    * @see org.eclipse.core.commands.operations.IOperationHistory#undoOperation(org.eclipse.core.commands.operations.IUndoableOperation,
    *      org.eclipse.core.runtime.IProgressMonitor,
    *      org.eclipse.core.runtime.IAdaptable)
    */
   IStatus undo(IUndoableOperation operation, IProgressMonitor monitor,
         IAdaptable args);

   /**
    * Provides a convenience method for supplying and operation with the correct
    * context, executing an operation, and adding it to the IOperationHistory
    * managed by the IEditModel for this IEditModelScribbler.
    * 
    * <specific-request-for-input>
    * 
    * Should this method return false if it cannot succeed or throw an
    * Exception?
    * 
    * </specific-request-for-input>
    * 
    * 
    * <b>Implemented.</b>
    * 
    * 
    * @see org.eclipse.core.commands.operations.IOperationHistory#redoOperation(org.eclipse.core.commands.operations.IUndoableOperation,
    *      org.eclipse.core.runtime.IProgressMonitor,
    *      org.eclipse.core.runtime.IAdaptable)
    */
   IStatus redo(IUndoableOperation operation, IProgressMonitor monitor,
         IAdaptable args);

   /**
    * Requests the last operation that can be undone for the IUndoContext
    * supplied by this IEditModelScribbler. If an operation is available, a
    * request is made to undo that operation. The operation may not have been
    * executed by this specific IEditModelScribbler; it could be the operation
    * associated with another IEditModelScribbler with a matching set of
    * IScribblerDomains.
    * 
    * 
    * <b>Implemented.</b>
    * 
    * 
    * @return
    * @see org.eclipse.core.commands.operations.IOperationHistory#undo(org.eclipse.core.commands.operations.IUndoContext,
    *      org.eclipse.core.runtime.IProgressMonitor,
    *      org.eclipse.core.runtime.IAdaptable)
    */
   IStatus undoLast(IProgressMonitor monitor, IAdaptable args);

   /**
    * Requests the last operation that can be undone for the IUndoContext
    * supplied by this IEditModelScribbler. If an operation is available, a
    * request is made to redo that operation. The operation may not have been
    * executed by this specific IEditModelScribbler; it could be the operation
    * associated with another IEditModelScribbler with a matching set of
    * IScribblerDomains.
    * 
    * <b>Implemented.</b>
    * 
    * @return
    * @see org.eclipse.core.commands.operations.IOperationHistory#redo(org.eclipse.core.commands.operations.IUndoContext,
    *      org.eclipse.core.runtime.IProgressMonitor,
    *      org.eclipse.core.runtime.IAdaptable)
    */
   IStatus redoLast(IProgressMonitor monitor, IAdaptable args);

   /**
    * Flushes the operation history for this IEditModelScribbler.
    * 
    * <b>Implemented.</b>
    * 
    * @param monitor
    * @return
    */
   void flushOperationHistory(IProgressMonitor monitor);

   /**
    * Return a TechnologyFlavor that can be used to manage model elements
    * associated with a particular technology not modeled as EMF. The flavor
    * must be cast to the appropriate implementation.
    * 
    * @param technologyFlavorLifecycleId
    * @return
    */
   ITechnologyFlavor createTechnologyFlavor(String technologyFlavorLifecycleId,
         IAdaptable input);

   /**
    * Returns the IUndoContext for this IEditModelScribbler which can be used to
    * associate all operations with this scribbler which are executed on data
    * managed by the scribbler.
    * 
    * @return
    */
   IUndoContext getUndoContext();

}
