/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.handler;

import org.eclipse.core.resources.IContainer;
import org.eclipse.stp.core.saf.IUIContext;
import org.eclipse.stp.core.saf.exception.InvalidOperationException;
import org.eclipse.stp.core.saf.exception.SAFException;
import org.eclipse.stp.core.sca.ExternalService;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

/**
 * This interface must be implemented by all external service handlers. Most
 * handlers should extend the abstract class
 * {@link org.eclipse.stp.core.saf.handler.AbstractExternalServiceHandler AbstractExternalServiceHandler}
 * which provides certain convenient methods.
 */
public interface IExternalServiceHandler extends IHandler {

	/**
	 * Queries if an external service of the kind supported by this handler can
	 * be created for the specified model object. If this method returns
	 * <code>true</code>, the handler's
	 * {@link #createExternalServiceFor(Object,IContainer,IUIContext)} method
	 * may be called to create an external service for the model object.
	 * <p>
	 * This method should be quick to execute, because it may be called to populate a
	 * context menu or a user dialog.
	 * 
	 * @param service
	 *            an object representing a service that may be imported
	 * @return <code>true</code> if the handler can create an
	 *         <code>ExternalService</code> object from the specified model
	 *         object; <code>false</code> otherwise
	 */
	public boolean canCreateExternalServiceFor(Object service);
	
	/**
	 * Creates an external service with a binding of the kind supported by this
	 * handler for the specified model object. All interfaces of the service
	 * file are imported. To import specific interfaces, call
	 * {@link #createExternalServiceBindingFor(ExternalService, IUIContext)}
	 * instead. Optionally, this method may create other objects required to
	 * support the new external service (such as a component). This method is
	 * not expected to associate a resource with the returned
	 * {@link ExternalService} or supporting object.
	 * 
	 * @param service
	 *            an object representing a service that may be imported
	 * @param container
	 *            the container that holds the module, subsystem, composite, etc.
	 *            that will use the external service object returned by this
	 *            method. The handler could use its location as a base to place
	 *            any binding-related files that need to be generated.
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the external service. 
	 *            Callers may pass in a null value.
	 * @return an array of {@link SCAObject} containing the new external service
	 *         and any other SCA objects generated by the handler to
	 *         support the new external service
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                An external service cannot be created from the given
	 *                service object.
	 * @exception InterruptedException
	 *                The caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public SCAObject[] createExternalServiceFor(Object service,
			IContainer container, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;
	
	/**
	 * Queries if the external service handler provides support for creating an
	 * external service binding using the method
	 * {@link #createExternalServiceBindingFor(ExternalService, IUIContext) createExternalServiceBindingFor()}.
	 * 
	 * @return <code>true</code> if support is provided for creating a
	 *         binding; <code>false</code> otherwise
	 */
	public boolean canCreateExternalServiceBinding();
	
	/**
	 * Queries if the external service handler provides support for creating more
	 * than one binding instance for an external service.
	 * 
	 * @return <code>true</code> if support is provided for creating more than
	 *         one binding of the supported kind; <code>false</code> otherwise
	 */
	public boolean canCreateManyExternalServiceBindings();

	/**
	 * Adds a binding of the kind supported by this handler to the specified
	 * {@link ExternalService ExternalService} object, based on the specified
	 * service file and the interfaces in the external service definition. The
	 * handler is expected to update the external service definition, if
	 * necessary. If the external service has an existing binding, the handler
	 * can add to or replace the binding. The handler can use the callback to
	 * query for more information from the user. If the binding requires
	 * associated files, this method should generate those files. In this case,
	 * this method should be aware that the files it generates may replace some
	 * already being managed in a team environment. The ExternalService object
	 * should have a resource associated with it. The handler can use this
	 * location to generate relative file references if needed or use the
	 * location for storing binding-related files that need to be generated. The
	 * resource can be retrieved by casting the
	 * {@link ExternalService ExternalService} object to an
	 * {@link org.eclipse.emf.ecore.EObject EObject} and calling
	 * {@link org.eclipse.emf.ecore.EObject#eResource() EObject.eResource()}.
	 * 
	 * @param theExternalService
	 *            an existing external service definition, which may contain
	 *            interfaces and a binding
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the binding. Callers
	 *            may pass in a null value.
	 * @return an IDataModelOperation instance that, once executed by the
	 *         caller, creates the new external service binding
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                A binding cannot be created for the given external service.
	 * @exception InterruptedException
	 *                The caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public IDataModelOperation createExternalServiceBindingFor(
			ExternalService theExternalService, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;

	// ===========
	// Auto-wiring
	// ===========

	/**
	 * Queries if an external service of the kind supported by this handler
	 * supports interfaces of the kind specified by the given interface handler
	 * in its interface set.
	 * 
	 * @param interfaceHandler
	 *            the interface handler that will be used to manipulate the
	 *            interface. The implementation of this method can obtain the
	 *            interface kind by invoking {@link IInterfaceHandler#getKind()}
	 *            when performing the evaluation.
	 * @return <code>true</code> if the external service supports interfaces
	 *         of the specified interface kind; <code>false</code> otherwise
	 * @throws SAFException
	 *             An unexpected exception occurred in the handler.
	 */
	public boolean isInterfaceKindSupported(IInterfaceHandler interfaceHandler)
			throws SAFException;
	
	/**
	 * Removes a given binding kind from the external service. Generally,
	 * existing interfaces will be preserved. No clean-up is done on existing
	 * binding-related files. If the external service does not contain the given
	 * binding type, this method performs no work.
	 * 
	 * @param theExternalService
	 *            the external service being modified
	 * @return an {@link IDataModelOperation} instance that, once executed by the
	 *         caller, resets the extenal service point kind
	 * @exception SAFException
	 *                No handler for the specified kind is found, or the
	 *                handler throws an exception.
	 * @exception InvalidOperationException
	 *                The external service kind cannot be removed.
	 */
	public IDataModelOperation removeKind(ExternalService theExternalService)
			throws SAFException, InvalidOperationException;

}
