/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.editors;

import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.part.MultiPageSelectionProvider;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;

/**
 * @author rcernich
 * 
 * Created on Mar 5, 2004
 */
public abstract class EnhancedFormEditor extends FormEditor implements
		ISelectionProvider {

	// This should be moved somewhere else.
	public static final String DEFAULT_EDITOR_PAGE_KEY = "defaultPageKey"; //$NON-NLS-1$

	public static final int SECTION_SPACING_HORIZ = 15;
	public static final int SECTION_SPACING_VERT = 15;
	public static final int SECTION_MARGIN_WIDTH = 10;
	public static final int SECTION_MARGIN_HEIGHT = 10;

	public static final int CONTROL_SPACING_HORIZ = 5;
	public static final int CONTROL_SPACING_VERT = 5;
	public static final int CONTROL_MARGIN_WIDTH = 2;
	public static final int CONTROL_MARGIN_HEIGHT = 2;

	public static final GridLayout newPageGridLayout() {
		GridLayout layout = new GridLayout();
		layout.horizontalSpacing = SECTION_SPACING_HORIZ;
		layout.verticalSpacing = SECTION_SPACING_VERT;
		layout.marginWidth = SECTION_MARGIN_WIDTH;
		layout.marginHeight = SECTION_MARGIN_HEIGHT;
		return layout;
	}

	public static final GridLayout newSectionGridLayout() {
		GridLayout layout = new GridLayout();
		layout.horizontalSpacing = CONTROL_SPACING_HORIZ;
		layout.verticalSpacing = CONTROL_SPACING_VERT;
		layout.marginWidth = CONTROL_MARGIN_WIDTH;
		layout.marginHeight = CONTROL_MARGIN_HEIGHT;
		return layout;
	}

	public static final FormLayout newPageFormLayout() {
		FormLayout layout = new FormLayout();
		layout.spacing = SECTION_SPACING_HORIZ;
		layout.marginWidth = SECTION_MARGIN_WIDTH;
		layout.marginHeight = SECTION_MARGIN_HEIGHT;
		return layout;
	}

	private ListenerList mSelectionListeners = new ListenerList();
	private ISelection mSelection = StructuredSelection.EMPTY;

	public EnhancedFormEditor() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IEditorPart#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		super.init(site, input);
		site.setSelectionProvider(new MultiPageSelectionProvider(this) {

			public void addSelectionChangedListener(
					ISelectionChangedListener listener) {
				EnhancedFormEditor.this.addSelectionChangedListener(listener);
			}

			public ISelection getSelection() {
				return EnhancedFormEditor.this.getSelection();
			}

			public void removeSelectionChangedListener(
					ISelectionChangedListener listener) {
				EnhancedFormEditor.this.removeSelectionChangedListener(listener);
			}

			public void setSelection(ISelection selection) {
				EnhancedFormEditor.this.setSelection(selection);
			}

			public void fireSelectionChanged(final SelectionChangedEvent event) {
				EnhancedFormEditor.this.fireSelectionChanged(event);
			}

		});
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.MultiPageEditorPart#createPages()
	 */
	protected void createPages() {
		super.createPages();
		String pageId = loadDefaultPage();
		if (pageId != null) {
			setActivePage(pageId);
		}
		// Remove null page entry from pages list.
		pages.remove(null);
	}

	protected abstract IPropertySheetPage getPropertySheetPage();

	public IFormPage[] getPages() {
		return (IFormPage[]) pages.toArray(new IFormPage[pages.size()]);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.MultiPageEditorPart#getEditor(int)
	 */
	protected IEditorPart getEditor(int pageIndex) {
		IEditorPart editor = super.getEditor(pageIndex);
		if (editor == null) {
			// Workaround bug where FormPage's do not get associated with the
			// tab items.
			for (Iterator it = pages.iterator(); it.hasNext(); ) {
				IFormPage page = (IFormPage)it.next();
				if (page.getIndex() == pageIndex) {
					editor = page;
					break;
				}
			}
		}
		return editor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#isDirty()
	 */
	public boolean isDirty() {
		boolean dirty = false;
		IFormPage[] pages = getPages();
		for (int index = 0, count = pages.length; !dirty && index < count; ++index) {
			dirty = pages[index] == null ? false : pages[index].isDirty();
		}
		return dirty || super.isDirty();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		IFormPage[] pages = getPages();
		for (int index = 0, count = pages.length; index < count; ++index) {
			// Make sure any local changes get committed.
			pages[index].doSave(monitor);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		if (adapter.isAssignableFrom(IContentOutlinePage.class)) {
			return getContentOutline();
		}
		else if (adapter.isAssignableFrom(IPropertySheetPage.class)) {
			return getPropertySheetPage();
		}
		else if (adapter.isAssignableFrom(getClass())) {
			return this;
		}
		return super.getAdapter(adapter);
	}

	protected abstract IContentOutlinePage getContentOutline();

	protected void setActivePage(int pageIndex) {
		super.setActivePage(pageIndex);
		updateContentOutline(getActivePageInstance());
	}

	protected void updateContentOutline(IFormPage part) {
		getContentOutline().setSelection(new StructuredSelection(part));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionProvider#addSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		mSelectionListeners.add(listener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
	 */
	public ISelection getSelection() {
		return mSelection;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionProvider#removeSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void removeSelectionChangedListener(
			ISelectionChangedListener listener) {
		mSelectionListeners.remove(listener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection) {
		mSelection = selection;
		SelectionChangedEvent event = new SelectionChangedEvent(this, selection);
		fireSelectionChanged(event);
	}

	protected void fireSelectionChanged(final SelectionChangedEvent event) {
		Object[] listeners = mSelectionListeners.getListeners();
		for (int index = 0, count = listeners.length; index < count; ++index) {
			ISelectionChangedListener listener = (ISelectionChangedListener) listeners[index];
			listener.selectionChanged(event);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose() {
		storeDefaultPage();
		super.dispose();
	}

	protected abstract String getPropertyStorageQualifier();

	private void storeDefaultPage() {
		IEditorInput input = getEditorInput();
		int pageIndex = getCurrentPage();
		IFormPage[] pages = getPages();
		String pageId = null;
		if (pageIndex > 0 && pageIndex < pages.length) {
			pageId = pages[pageIndex].getId();
		}
		if (input instanceof IFileEditorInput) {
			// load the setting from the resource
			IFile file = ((IFileEditorInput) input).getFile();
			if (file != null) {
				//set the settings on the resouce
				try {
					file.setPersistentProperty(new QualifiedName(
							getPropertyStorageQualifier(),
							DEFAULT_EDITOR_PAGE_KEY), pageId);
				}
				catch (CoreException e) {
// TODO: Error handling here?
				}
			}
		}
	}

	private String loadDefaultPage() {
		String pageId = null;
		IEditorInput input = getEditorInput();
		if (input instanceof IFileEditorInput) {
			// load the setting from the resource
			IFile file = ((IFileEditorInput) input).getFile();
			try {
				pageId = file
						.getPersistentProperty(new QualifiedName(
								getPropertyStorageQualifier(),
								DEFAULT_EDITOR_PAGE_KEY));
			}
			catch (CoreException e) {
// TODO: Error handling here?
			}
		}
		return pageId;
	}

}
