/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.core;

import java.util.Comparator;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IServerDefinition;
import org.eclipse.stp.soas.deploy.core.ISupportedTechnologyType;
import org.eclipse.stp.soas.deploy.core.ITechnologyDefinition;
import org.eclipse.stp.soas.deploy.core.ITechnologyMap;
import org.eclipse.stp.soas.deploy.core.IVersion;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;


/**
 * @author rcernich
 * 
 * Created on Apr 8, 2004
 */
public class ServerDefinition implements IServerDefinition, Comparable {

	public static final String ATTR_ID = "id"; //$NON-NLS-1$

	private IConfigurationElement mElement;
	private String mID;
	private SortedMap mSupportingServerVersionssByTechnologyDef;
	private SortedMap mTechnologyMapsByServerVersion;

	public ServerDefinition(IConfigurationElement element)
			throws PackageExtensionException {
		super();
		init(element);
		mSupportingServerVersionssByTechnologyDef = new TreeMap();
		mTechnologyMapsByServerVersion = new TreeMap(
				new DecendingVersionOrder());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IServerDefinition#getID()
	 */
	public String getID() {
		return mID;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IServerDefinition#getTechnologyMaps()
	 */
	public ITechnologyMap[] getTechnologyMaps() {
		return (ITechnologyMap[]) mTechnologyMapsByServerVersion.values()
				.toArray(new ITechnologyMap[0]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IServerDefinition#getTechnologyMap(org.eclipse.stp.soas.deploy.core.IVersion)
	 */
	public ITechnologyMap getTechnologyMap(IVersion serverVersion) {
		ITechnologyMap techMap;
		Iterator it = mTechnologyMapsByServerVersion.tailMap(serverVersion)
				.values().iterator();
		if (it.hasNext()) {
			techMap = (ITechnologyMap) it.next();
			if (!techMap.getServerVersion().isCompatibleWithVersion(
					serverVersion, techMap.getVersionMatchRule())) {
				techMap = null;
			}
		}
		else {
			techMap = null;
		}
		return techMap;
	}

	public IVersion[] getSupportingServerVersions(ITechnologyDefinition techDef) {
		return (IVersion[]) getSupportingServerVersionsSet(techDef).toArray(
				new IVersion[0]);
	}

	/* package */void addTechnologyMap(ITechnologyMap itm)
			throws PackageExtensionException {
		Assert.isTrue(itm.getServerDefinition().equals(this));
		if (mTechnologyMapsByServerVersion.containsKey(itm.getServerVersion())) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"ServerDefinition.trace.error.duplicateTechnologyMap", //$NON-NLS-1$
										new Object[] { mID,
												itm.getServerVersion()}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"ServerDefinition.exception.duplicateTechnologyMap")); //$NON-NLS-1$
		}
		mTechnologyMapsByServerVersion.put(itm.getServerVersion(), itm);
		addTechnologiesToList(itm);
	}

	private void addTechnologiesToList(ITechnologyMap itm) {
		ISupportedTechnologyType[] techTypes = itm
				.getSupportedTechnologyTypes();
		for (int index = 0, count = techTypes.length; index < count; ++index) {
			ITechnologyDefinition techDef = techTypes[index]
					.getTechnologyDefinition();
			SortedSet serverVersions = getSupportingServerVersionsSet(techDef);
			serverVersions.add(itm.getServerVersion());
		}
	}

	private SortedSet getSupportingServerVersionsSet(
			ITechnologyDefinition techDef) {
		SortedSet serverVersions = (SortedSet) mSupportingServerVersionssByTechnologyDef
				.get(techDef);
		if (serverVersions == null) {
			serverVersions = new TreeSet(new DecendingVersionOrder());
			mSupportingServerVersionssByTechnologyDef.put(techDef,
					serverVersions);
		}
		return serverVersions;
	}

	private void init(IConfigurationElement element)
			throws PackageExtensionException {
		Assert.isTrue(DeploymentExtensionManager.EXT_ELEM_SERVER_DEFINITION
				.equals(element.getName()));

		mElement = element;

		mID = element.getAttribute(ATTR_ID);

		if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
			System.out
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"ServerDefinition.trace.error.serverDefinitionParseError", //$NON-NLS-1$
									new Object[] { getID()}));
			System.out.flush();
		}

		if (mID == null) {
			throw new PackageExtensionException(DeployCorePlugin
					.getDefault().getResourceString(
							"ServerDefinition.exception.idNotSpecified")); //$NON-NLS-1$
		}
	}

	public int compareTo(Object o) {
		return getID().compareTo(((ServerDefinition) o).getID());
	}

	public boolean equals(Object obj) {
		boolean retVal;
		if (this == obj) {
			retVal = true;
		}
		else if (obj instanceof ServerDefinition) {
			retVal = getID().equals(((ServerDefinition) obj).getID());
		}
		else {
			retVal = false;
		}
		return retVal;
	}

	public int hashCode() {
		return mID.hashCode();
	}

	private static class DecendingVersionOrder implements Comparator {

		public int compare(Object o1, Object o2) {
			return -(((Comparable) o1).compareTo(o2));
		}
	}

}