/*******************************************************************************
 * Copyright (c) 2005-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.dialogs;

import java.net.URL;
import java.util.Collection;
import java.util.Comparator;
import java.util.TreeSet;

import org.eclipse.datatools.connectivity.ICategory;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.ProfileManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IPackage;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.ui.ConnectionProfileTreeViewer;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.DialogPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;

/**
 * @author rcernich
 * 
 * Created on Jan 28, 2004
 */
public class ConnectionProfileSelectionDialogPage extends DialogPage {

	public static final String TITLE;
	public static final String ICON_PATH;
	public static final ImageDescriptor ICON_DESC;

	public static final String LABEL_SELECT_PACKAGES;
	public static final int MIN_TREE_HEIGHT;

	private ConnectionProfileTreeViewer mCPViewer;
	private IConnectionProfile mSelectedCP;
	private IPackage mPackage;
	private Collection mExistingProfiles;

	public ConnectionProfileSelectionDialogPage() {
		this(TITLE, ICON_DESC);
	}

	/**
	 * @param title
	 * @param image
	 */
	public ConnectionProfileSelectionDialogPage(String title,
												ImageDescriptor image) {
		super(title, image);
		setNeedsApplyButton(false);
	}

	public void init(Root root, IPackage pkg) {
		mExistingProfiles = new TreeSet(new ProfileComparator());
		mExistingProfiles.addAll(root.getServer());
		mPackage = pkg;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		Composite content = new Composite(parent, SWT.NULL);
		content.setLayout(new GridLayout(1, false));
		content.setLayoutData(new GridData(GridData.FILL_BOTH));

		Label label = new Label(content, SWT.NULL);
		label.setText(LABEL_SELECT_PACKAGES);
		GridData data = new GridData();
		label.setLayoutData(data);

		Tree tree = new Tree(content, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL);
		mCPViewer = new ConnectionProfileTreeViewer(tree);
		mCPViewer.getViewer().setAutoExpandLevel(3);
		data = new GridData(GridData.FILL_BOTH);
		data.heightHint = tree.getItemHeight() * MIN_TREE_HEIGHT;
		tree.setLayoutData(data);
		mCPViewer.getViewer().addSelectionChangedListener(
				new ISelectionChangedListener() {

					public void selectionChanged(SelectionChangedEvent event) {
						handleCPSelectionChanged();
					}
				});
		mCPViewer.getViewer().addDoubleClickListener(
				new IDoubleClickListener() {

					public void doubleClick(DoubleClickEvent event) {
						handleCPDoubleClick();
					}
				});
		mCPViewer.getViewer().addFilter(new ViewerFilter() {

			public boolean select(Viewer viewer, Object parentElement,
					Object element) {
				boolean retVal;
				if (element instanceof IConnectionProfile) {
					retVal = !mExistingProfiles.contains(element);
				}
				else {
					retVal = element instanceof ICategory;
				}
				return retVal;
			}
		});
		if (mPackage != null) {
			mCPViewer.getViewer().addFilter(new ViewerFilter() {

				public boolean select(Viewer viewer, Object parentElement,
						Object element) {
					boolean retVal;
					if (element instanceof IConnectionProfile) {
						retVal = null != DeploymentExtensionManager
								.getInstance().getDeployDriver(
										mPackage,
										((IConnectionProfile) element)
												.getProviderId());
					}
					else {
						retVal = true;
					}
					return retVal;
				}
			});
		}

		initControls();

		setControl(content);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.internal.deploy.ui.dialogs.DialogPage#performOk()
	 */
	public boolean performOk() {
		boolean retVal;
		IStructuredSelection selection = (IStructuredSelection) mCPViewer
				.getViewer().getSelection();
		if (selection.isEmpty()) {
			mSelectedCP = null;
			retVal = false;
		}
		else {
			mSelectedCP = (IConnectionProfile) selection.getFirstElement();
			retVal = true;
		}
		return retVal;
	}

	public IConnectionProfile getSelection() {
		return mSelectedCP;
	}

	private void initControls() {
		mCPViewer.getViewer().setInput(ProfileManager.getInstance());

		updateControls();

		validate();

		setErrorMessage(null);
	}

	private void updateControls() {
		// Nothing to do here
	}

	private void validate() {
		IStructuredSelection selection = (IStructuredSelection) mCPViewer
				.getViewer().getSelection();
		if (selection.isEmpty()
				|| !(selection.getFirstElement() instanceof IConnectionProfile)) {
			setErrorMessage(DeployCorePlugin.getDefault().getResourceString("ConnectionProfileSelectionDialogPage.error.selectConnectionProfile")); //$NON-NLS-1$
			setIsValid(false);
		}
		else {
			setErrorMessage(null);
			setIsValid(true);
		}
	}

	private void handleCPDoubleClick() {
		updateControls();

		validate();

		if (isValid()) {
			getContainer().closeOK();
		}
	}

	private void handleCPSelectionChanged() {
		updateControls();

		validate();
	}

	private static class ProfileComparator implements Comparator {

		public int compare(Object o1, Object o2) {
			String s1;
			String s2;

			if (o1 instanceof IConnectionProfile) {
				s1 = ((IConnectionProfile) o1).getName();
			}
			else if (o1 instanceof DeployServer) {
				s1 = ((DeployServer) o1).getProfileName();
			}
			else {
				s1 = (String) o1;
			}

			if (o2 instanceof IConnectionProfile) {
				s2 = ((IConnectionProfile) o2).getName();
			}
			else if (o2 instanceof DeployServer) {
				s2 = ((DeployServer) o2).getProfileName();
			}
			else {
				s2 = (String) o2;
			}

			return s1.compareTo(s2);
		}
	}

	static {
		TITLE = DeployCorePlugin.getDefault().getResourceString("ConnectionProfileSelectionDialogPage.title.selectConnectionProfile"); //$NON-NLS-1$
		ICON_PATH = "icons/blank_wizard_graphic.gif"; //$NON-NLS-1$
		URL url = null;
		try {
			url = DeployCorePlugin.getDefault().getBundle().getEntry(
					ICON_PATH);
		}
		catch (RuntimeException e) {
			e.printStackTrace();
		}
		if (url == null) {
			ICON_DESC = ImageDescriptor.getMissingImageDescriptor();
		}
		else {
			ICON_DESC = ImageDescriptor.createFromURL(url);
		}
		LABEL_SELECT_PACKAGES = DeployCorePlugin.getDefault().getResourceString("ConnectionProfileSelectionDialogPage.LABLE.selectConnectionProfile"); //$NON-NLS-1$
		MIN_TREE_HEIGHT = 15;
	}

}