/*******************************************************************************
 * Copyright (c) 2005-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.properties;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Properties;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.FolderSelectionDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.PropertyPage;
import org.eclipse.ui.internal.ide.misc.ContainerContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;


/**
 * A property page, used to specify one output folder for deployment.
 */
public class DeployOutputFolderPropertyPage extends PropertyPage {

    // Stashed reference to the project
    private IProject project = null;

    private Text outputFolderText = null;

    private DeploymentResource deploymentResource = null;

    public DeployOutputFolderPropertyPage() {
        super();
    }

    protected Control createContents(Composite parent) {
        this.noDefaultAndApplyButton();
        Composite content = new Composite(parent, SWT.NONE);
        GridData data = new GridData(GridData.FILL_BOTH);
        content.setLayoutData(data);

        GridLayout layout = new GridLayout(1, false);
        content.setLayout(layout);

        Label label = new Label(content, SWT.NONE);
        label.setText(DeployCorePlugin.getDefault().getResourceString(
                "LABEL.DefineDefaultFolder"));//$NON-NLS-1$

        // Creates one composite to hold the output folder text and the browse button
        Composite blockComposite = new Composite(content, SWT.NONE);
        blockComposite.setLayoutData(new GridData(GridData.FILL_BOTH));

        GridLayout gridLayout = new GridLayout(2, false);
        blockComposite.setLayout(gridLayout);

        outputFolderText = new Text(blockComposite, SWT.SINGLE | SWT.BORDER);
        outputFolderText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        outputFolderText.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                validateChange(e);
            }
        });
        
        Button button = new Button(blockComposite, SWT.PUSH);
        button.setText(DeployCorePlugin.getDefault().getResourceString(
                "Button.Browse"));//$NON-NLS-1$
        button.addSelectionListener(new SelectionListener() {
            public void widgetSelected(SelectionEvent e) {
                chooseFolder(e);
            }
            public void widgetDefaultSelected(SelectionEvent e) {
                chooseFolder(e);
            }
        });

        initailOutputFolder();

        return content;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IWorkbenchPropertyPage#setElement(org.eclipse.core.runtime.IAdaptable)
     */
    public void setElement(IAdaptable element) {
        super.setElement(element);

        if (element instanceof IProject) {
            IProject aproject = (IProject) element;
            this.project = aproject;
            this.deploymentResource = new DeploymentResource(this.project);
        }
    }
    
    /**
     * Do validates when the outuput folder changes
     * @param e
     */
    private void validateChange(ModifyEvent e) {
        Text text = (Text) e.widget;
        String outputFolder = text.getText();

        // Checks if the value is empty
        if ((outputFolder == null) || (outputFolder.trim().length() == 0)) {
            this.setErrorMessage(DeployCorePlugin.getDefault()
                    .getResourceString(
                            "DeploymentOutputFolderPage.error.emptyPath"));//$NON-NLS-1$
            this.setValid(false);
            return;
        }

        // Checks if the output folder is under current project
        IPath projectPath = this.project.getFullPath();
        IPath outputFolderPath = new Path(outputFolder).makeAbsolute();

        if (!projectPath.isPrefixOf(outputFolderPath)
                || (projectPath.equals(outputFolderPath))) {
            String name = this.project.getName();
            this.setErrorMessage(DeployCorePlugin.getDefault()
                    .getResourceString(
                            "DeploymentOutputFolderPage.error.invalidPath",//$NON-NLS-1$
                            new Object[] { outputFolder, name }));
            this.setValid(false);
            return;
        }

        this.setErrorMessage(null);
        this.setValid(true);
    }
    
    private void chooseFolder(SelectionEvent e) {
        final ContainerContentProvider cp = new ContainerContentProvider();
        cp.showClosedProjects(false);

        FolderSelectionDialog dialog = new FolderSelectionDialog(getShell(),
                new WorkbenchLabelProvider(), cp);

        dialog.setTitle(DeployCorePlugin.getDefault().getResourceString(
                "FolderSelectionDialog.TITLE"));//$NON-NLS-1$	
        dialog.setMessage(DeployCorePlugin.getDefault().getResourceString(
                "FolderSelectionDialog.MESSAGE"));//$NON-NLS-1$	

        // Gets all the projects that don't need to be showed on the dialog
        IWorkspaceRoot root = project.getWorkspace().getRoot();
        IProject[] allProjects = root.getProjects();
        ArrayList rejectedElements = new ArrayList();
        for (int i = 0; i < allProjects.length; i++) {
            if (!allProjects[i].equals(project)) {
                rejectedElements.add(allProjects[i]);
            }
        }
 
        dialog.addFilter(new ProjectFilter(rejectedElements.toArray(),
                new Class[] { IProject.class, IFolder.class }));
        dialog.setInput(project.getWorkspace().getRoot());
        IContainer initialSel = getInitialSelecionForDialog();
        if (initialSel != null) {
            dialog.setInitialSelection(initialSel);
        }
        if (dialog.open() == FolderSelectionDialog.OK) {
            // Gets the first element as the deployment folder
            IContainer selection = (IContainer) dialog.getFirstResult();
            outputFolderText.setText(selection.getFullPath().toString());
        }
    }

    private IContainer getInitialSelecionForDialog() {
        String text = outputFolderText.getText();
        IWorkspaceRoot workspaceRoot = this.project.getWorkspace().getRoot();
        IResource initSelection = workspaceRoot.findMember(text);

        if (initSelection instanceof IContainer) {
            return (IContainer) initSelection;
        }

        return null;
    }

    private void initailOutputFolder() {
    	String outputFolder = deploymentResource.getOutputFolder();
        
        if (outputFolder != null) {
            outputFolderText.setText(outputFolder);
        }
    }

    /*
     * @see IPreferencePage#performOk
     */
    public boolean performOk() {
        String folderText = outputFolderText.getText();
        IWorkspaceRoot workspaceRoot = this.project.getWorkspace().getRoot();
        IPath destFolderPath = workspaceRoot.getFullPath().append(folderText);
        IFolder destFolder = workspaceRoot.getFolder(destFolderPath);

        if ((destFolder == null) || (!destFolder.getLocation().toFile().exists())) {
            try {
                // Creates the output folder
                Utilities.createFolder(destFolder, true, true);
            } catch (CoreException e) {
				MessageDialog
                        .openError(
                                null,
                                DeployCorePlugin
                                        .getDefault()
                                        .getResourceString(
                                                "DeploymentOutputFolderPage.moveFolderDialog.title"), e.getMessage());//$NON-NLS-1$
            }
        }

        // Checks if need to move the old deployment files to the new folder
        String oldOutputFolder = deploymentResource.getOutputFolder();
        IPath oldFolderPath = new Path(oldOutputFolder).makeAbsolute();
        if (!oldFolderPath.equals(destFolderPath)) {
            String message = DeployCorePlugin.getDefault()
                    .getResourceString(
                            "DeploymentOutputFolderPage.message.moveFolder",//$NON-NLS-1$	
                            new Object[] { oldOutputFolder });
            String title = DeployCorePlugin
                    .getDefault()
                    .getResourceString(
                            "DeploymentOutputFolderPage.moveFolderDialog.title");//$NON-NLS-1$
            
            boolean moveFiles = MessageDialog.openQuestion(this.getShell(),
                    title, message);
            if (moveFiles) {
                IFolder oldFolder = workspaceRoot.getFolder(oldFolderPath);
                if (oldFolder.exists()) {
                    try {
                        Utilities.moveFolder(oldFolder, destFolder,
                                "svcpkg", false);//$NON-NLS-1$	
                    } catch (CoreException e) {
                        MessageDialog
                                .openError(
                                    null,
                                    DeployCorePlugin
                                        .getDefault()
                                        .getResourceString(
                                        "DeploymentOutputFolderPage.moveFolderDialog.title"), e.getMessage());//$NON-NLS-1$
                    }
                }
            }
        }

        deploymentResource.setOutputFolder(outputFolderText.getText().trim());

        return super.performOk();
    }

    public static File getOutputFolder(IProject project) {
        DeploymentResource resource = new DeploymentResource(project);
        String folderPath = resource.getOutputFolder();
        IFolder folder = project.getWorkspace().getRoot().getFolder(
                new Path(folderPath));

        if (!folder.exists()) {
            try {
                Utilities.createFolder(folder, true, true);
            } catch (CoreException e) {
                System.err.println(e.getMessage());
            }
        }
        
        return new File(folder.getRawLocation().toOSString());
    }

    public static String getOutputContainerPath(IProject project) {
        DeploymentResource resource = new DeploymentResource(project);
        String folderPath = resource.getOutputFolder();
        
        return folderPath;
    }
}

/**
 * Reads/Writes the deployment setting from/to the source. 
 */
class DeploymentResource {
    private IProject project = null;

    private IWorkspaceRoot workspaceRoot = null;

    public static final String DEPLOYMENT_OUTPUT_FOLDER = "deploymenet.output.folder";//$NON-NLS-1$	

    public static final String DEFAULT_DEPLOYMENT_FOLDER = "Deployment";//$NON-NLS-1$	

    private static final String DEPLOYMENT_OUTPUT_FILE = ".deployment";//$NON-NLS-1$	

    public DeploymentResource(IProject project) {
        this.project = project;
        this.workspaceRoot = this.project.getWorkspace().getRoot();
    }

    /**
     * Gets the output folder setting, if haven't do setting before, it will
     * generate one default output setting.
     * 
     * @return
     * @throws CoreException
     * @throws IOException
     */
    public String getOutputFolder(){
        Properties properties = new Properties();
        
        try {
            properties.load(getInputStream());
        } catch (Exception e) {
            System.err.println(e.getMessage());
            return null;
        }
        
        String folderPath = properties.getProperty(DEPLOYMENT_OUTPUT_FOLDER);
        String projectName = this.project.getName();
        StringBuffer pathBuf = new StringBuffer();
        pathBuf.append("/").append(projectName).append("/").append(folderPath);

        return pathBuf.toString();
    }

    /**
     * Applies the output folder setting to source. 
     * 
     * @param path
     * @throws CoreException
     * @throws IOException
     */
    public void setOutputFolder(String path){
        String projectName = this.project.getName();
        Path tempPath = new Path(path);

        // Removes the first segment, for it is related with project name.
        IPath resultPath = tempPath.removeFirstSegments(1);
        InputStream inputStream = generateInputStream(resultPath.toString());

        IPath filePath = workspaceRoot.getFullPath().append(projectName);
        filePath = filePath.append("/");
        filePath = filePath.append(DEPLOYMENT_OUTPUT_FILE);
        IFile deploymentFile = workspaceRoot.getFile(filePath);

       try {
            if (deploymentFile == null) {
                deploymentFile.create(inputStream, true,
                        new NullProgressMonitor());
            } else {
                deploymentFile.setContents(inputStream, true, false,
                        new NullProgressMonitor());
            }
            inputStream.close();
        } catch (Exception e) {
            System.err.println(e.getMessage());
        }       
    }

    private InputStream getInputStream() throws CoreException {
        String projectName = this.project.getName();
        IPath filePath = workspaceRoot.getFullPath().append(projectName);
        filePath = filePath.append("/");
        filePath = filePath.append(DEPLOYMENT_OUTPUT_FILE);
        IFile deploymentFile = workspaceRoot.getFile(filePath);

        if ((deploymentFile == null) || (!deploymentFile.exists())) {
            deploymentFile.create(
                    generateInputStream(DEFAULT_DEPLOYMENT_FOLDER), true,
                    new NullProgressMonitor());
        }
        
        return deploymentFile.getContents(true);
    }

    private InputStream generateInputStream(String path) {
        StringBuffer strBuf = new StringBuffer();
        strBuf.append(DEPLOYMENT_OUTPUT_FOLDER);
        strBuf.append("=");
        strBuf.append(path);

        return new ByteArrayInputStream(strBuf.toString().getBytes());
    }
}

/**
 * Filter class.  
 */
class ProjectFilter extends ViewerFilter {
    private Object[] rejectProjects;

    private Class[] acceptClasses;

    public ProjectFilter(Object[] rejectProjects, Class[] acceptClasses) {
        this.rejectProjects = rejectProjects;
        this.acceptClasses = acceptClasses;
    }
    
    /**
     * @see ViewerFilter#select
     */
    public boolean select(Viewer viewer, Object parentElement, Object element) {
        if (rejectProjects != null) {
            for (int i = 0; i < rejectProjects.length; i++) {
                if (element.equals(rejectProjects[i])) {
                    return false;
                }
            }
        }

        if (acceptClasses != null) {
            for (int i = 0; i < acceptClasses.length; i++) {
                if (acceptClasses[i].isInstance(element)) {
                    return true;
                }
            }
        }

        return false;
    }
}