/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.mainengine;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.net.InetAddress;
import java.net.URL;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Vector;

import org.eclipse.stp.b2j.core.jengine.internal.Version;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.core.bpel.WSEndpointReference;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPFactory;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPServerTransportListener;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.SOAPUtils;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPException;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPNotFoundException;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPServer;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPServerListener;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.soap.http.HTTPUtils;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.GCThread;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.VMFork;
import org.eclipse.stp.b2j.core.misc.internal.HexData;
import org.eclipse.stp.b2j.core.publicapi.B2jConfig;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * 
 * @author amiguel
 * 
 * The distributed engine Daemon implementation.
 * 
 * This class deliberately imports as few classes as possible to mimimise footprint.
 */
public class SoapDaemon extends Thread implements SOAPServerTransportListener, HTTPServerListener {

private Object TRANSACTION_LOCK = new Object();
	
public static String STDOUT_LOGFILE = null;
public static FileOutputStream stdlog = null;

public static boolean FORK_PROCESSES = true;
public static int DAEMON_PORT = 11000;
private static boolean USE_SSL = false;
private static boolean HAS_PASSWORD = false;
private static String PASSWORD = "password";

public static boolean SOAP_HTTP_PROXY_ON = false;
public static String SOAP_HTTP_PROXY_HOST = "";
public static int SOAP_HTTP_PROXY_PORT = 8080;
public static boolean SOAP_HTTPS_PROXY_ON = false;
public static String SOAP_HTTPS_PROXY_HOST = "";
public static int SOAP_HTTPS_PROXY_PORT = 8080;
public static String[] SOAP_PROXY_EXCLUDED_HOSTS = new String[0];

public static final String SOAPACTION_CREATE_CONTROLLER = "Daemon/CreateController";
public static final String SOAPACTION_CREATE_SUBCONTROLLER = "Daemon/CreateSubcontroller";
public static final String SOAPACTION_GET_VERSION = "Daemon/GetVersion";
public static final String SOAPACTION_SUPPORTS_VERSION = "Daemon/SupportsVersion";
public static final String SOAPACTION_ADD_VERSION = "Daemon/AddVersion";
public static final String SOAPACTION_LIST_ENGINES = "Daemon/ListEngines";
public static final String SOAPACTION_KILL_ENGINE = "Daemon/TerminateEngine";

static Object VERSIONS_LOCK = new Object();
static String VERSION_PREFIX = "b2j";
static ArrayList version_jars = new ArrayList();
static ArrayList version_tmp_jars = new ArrayList();

static Object CONF_LOCK = new Object();
static long last_loaded = -1;

//static Object PROPERTIES_LOCK = new Object();
//static long last_loaded = -1;

Object LIVE_PROCESSES_LOCK = new Object();
long live_process_id = 0;
Vector live_processes = new Vector(1,1);

WSEndpointReference epr;
WSEndpointReference daemon_service_epr;
int port = DAEMON_PORT;

static String dprefix = "(DAEMON)";

private SecureRandom sr = new SecureRandom();

/*private static String PROP_DEF = 
	"#B2J Distributed Engine Properties File\n"+
	"#\n"+
	"#fork_jvm - whether to start a new JVM for each controller (non-forked engines are not supported)\n"+
	"#daemon_port - the port the engine daemon should listen on\n"+
	"#\n"+
	"#java_exe - the java executable path\n"+
	"#java_classpath_arg - the java classpath arg\n"+
	"#java_vmargN - a VM level argument to append\n"+
	"#\n"+
	"\n"+
	"fork_jvm=true\n"+
	"daemon_port=11000\n"+
	"daemon_ssl=false\n"+
	"daemon_haspassword=false\n"+
	"daemon_password=password\n"+
	"\n"+
	"soap_http_proxy_on=false\n"+
	"soap_http_proxy_host=myproxy\n"+
	"soap_http_proxy_port=8080\n"+
	"soap_https_proxy_on=false\n"+
	"soap_https_proxy_host=myproxy\n"+
	"soap_https_proxy_port=8080\n"+
	"soap_proxy_excluded_hosts=localhost,127.*\n"+
	"\n"+
	"#Uncomment these to specify loggin level\n"+
	"#\n"+
	"#log_info=true\n"+
	"\n"+
	"#Uncomment these to specify extra engine classpath JARs\n"+
	"#java_classpath_extra1=myjarA.jar\n"+
	"#java_classpath_extra2=myjarB.jar\n"+
	"\n"+
	"#Uncomment these to specify an exact java command line\n"+
	"#\n"+
	"#java_exe=java\n"+
	"#java_classpath_arg=-cp\n"+
	"#java_vmarg0=-Xmx128m\n";
*/
static String java_exe;
static String java_cp;
static ArrayList java_extrajars;
static String[] java_vmargs;

int jvm_count = 0;

static String cd = null;
static String basejar = null;

	private SoapDaemon(String[] args) {
		cd = "."+File.separatorChar;
		
		basejar = cd+"b2j.jar";
		
		try {
			Runtime.getRuntime().addShutdownHook(new ProcessKillerThread());

			try {
				loadProperties();
			} catch (Exception e) {
				e.printStackTrace();
			}
			
			port = DAEMON_PORT;
			
			start_socket();
			run();
			
		} catch (Throwable t) {
			t.printStackTrace();
		}
		try {
			Thread.sleep(4000);
		} catch (Exception e) {
		}
	}

	public SoapDaemon(File enginedir, File base_jar, int port) {
		cd = enginedir.getAbsolutePath();
		if (!cd.endsWith(File.separator)) cd += File.separator;
		
		basejar = base_jar.getAbsolutePath();
		
		try {
			Runtime.getRuntime().addShutdownHook(new ProcessKillerThread());

			try {
				loadProperties();
			} catch (Exception e) {
				e.printStackTrace();
			}
			//constructor overrides properties file
			this.port = port;
			start_socket();
			start();
			Thread.sleep(1000);
			
		} catch (java.net.BindException t) {
			Logger.direct(dprefix+"Daemon could not start - something already bound to socket "+port);
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}

	public void killEngines() {
		new ProcessKillerThread().run();
	}
	
	private String[] filterEmpties(String[] commands) {
		int count = 0;
		for (int i = 0; i < commands.length; i++) {
			if (commands[i] == null) {
				count++;
			} else if (commands[i].length() == 0) {
				count++;
			}
		}
		String[] tmp = new String[commands.length-count];
		int index = 0;
		for (int i = 0; i < commands.length; i++) {
			if (commands[i] != null) {
				if (commands[i].length() > 0) {
					tmp[index++] = commands[i];
				}
			}
		}
		return tmp;
	}
	
	private Process forkJVMAsCommand(String[] vmargs, String classpath, String classname, String[] args, String[] jars, StringBuffer cmdline) throws Exception {

		ArrayList commands = new ArrayList();

		commands.add(java_exe);
		commands.add(java_cp);

		StringBuffer sb = new StringBuffer();
		String sep = System.getProperty("path.separator");

		sb.append(classpath).append(sep);
		for (int i = 0; i < jars.length; i++) {
			sb.append(jars[i]).append(sep);
		}
		
		commands.add(sb.toString());
		
		for (int i = 0; i < vmargs.length; i++) {
			commands.add(vmargs[i]);
		}
		
		commands.add(classname);

		for (int i = 0; i < args.length; i++) {
			commands.add(args[i]);
		};
		
		String[] fork_exe = new String[commands.size()];
		commands.toArray(fork_exe);

		Runtime r = Runtime.getRuntime();

		Logger.info(dprefix+"Launching engine component");
		
		fork_exe = filterEmpties(fork_exe);
		System.gc();

		for (int i = 0; i < fork_exe.length; i++) {
			Logger.info(dprefix+"arg "+i+": "+fork_exe[i]);	
			
			cmdline.append("["+fork_exe[i]+"]");
		}
		
		Process p = r.exec(fork_exe,null,new File(cd));
		
		return p;
	}
	private Process forkJVMAsVMFork(String[] vmargs, String classpath, String classname, String[] args, String[] jars, StringBuffer cmdline) throws Exception {
		VMFork vmfork = new VMFork(classname,args,new File(cd));

		vmfork.setVMSpecificArgs(vmargs);
		vmfork.setBaseClasspath(new String[]{classpath});
		
		//include the jars specified in the conf file
		String[] tmp = new String[jars.length+java_extrajars.size()];
		System.arraycopy(jars,0,tmp,0,jars.length);
		for (int i = 0; i < java_extrajars.size(); i++) {
			tmp[jars.length+i] = (String)java_extrajars.get(i);
		}
		jars = tmp;
		
		vmfork.setClasspathExtras(jars);
		
		cmdline.append("[(VMFork java?)]");
		for (int i = 0; i < vmargs.length; i++) {
			cmdline.append("["+vmargs[i]+"]");
		}

		cmdline.append("[(VMFork -cp?)]");
		cmdline.append("[");
		cmdline.append(classpath);
		cmdline.append(System.getProperty("path.separator"));
		for (int i = 0; i < jars.length; i++) {
			cmdline.append(jars[i]);
			cmdline.append(System.getProperty("path.separator"));
		}
		cmdline.append("]");

		cmdline.append("["+classname+"]");
		for (int i = 0; i < args.length; i++) {
			cmdline.append("["+args[i]+"]");
		}
		
		Process p = vmfork.fork();

		System.gc();

		return p;
	}
	
	private SessionAddress forkJVM(String[] vmargs, String classpath, String classname, String[] classargs, JARDependency[] deps, Message deps_msg, String desc, String req_address_hex, String description) throws Exception {
		System.gc();
		
		StringBuffer sb = new StringBuffer();
		Process p;
		
		String[] dependencies = new String[deps.length + java_extrajars.size()];
		for (int i = 0; i < deps.length; i++) {
			JARDependency.writeTemporaryDependency(deps[i]);
			dependencies[i] = deps[i].getFilePath();
		}
		
		for (int i = 0; i < java_extrajars.size(); i++) {
			dependencies[i + deps.length] = (String)java_extrajars.get(i);
		}
		
		if (java_vmargs.length > 0) {
			//conf file has specified VM args
			vmargs = new String[java_vmargs.length];
			for (int i = 0; i < java_vmargs.length; i++) {
				String s = java_vmargs[i];
				int index = s.indexOf("$TIME");
				if (index != -1) {
					vmargs[i] = s.substring(0,index) + System.currentTimeMillis() + s.substring(index+5);
				} else {
					vmargs[i] = s;
				}
			}
		}
		
		if (java_exe == null) {
			p = forkJVMAsVMFork(vmargs,classpath,classname,classargs,dependencies,sb);
		} else {
			p = forkJVMAsCommand(vmargs,classpath,classname,classargs,dependencies,sb);
		}
		
		BufferedReader bread = new BufferedReader(new InputStreamReader(p.getInputStream()));
		BufferedReader eread = new BufferedReader(new InputStreamReader(p.getErrorStream()));

		//write the address that we want this controller to pop up on
		PrintStream pout = new PrintStream(p.getOutputStream());
		pout.println(req_address_hex);
		pout.flush();
		
		Logger.info(dprefix+"wrote address "+req_address_hex);

		//write the dependencies message (if we have anything to write)
		if (deps_msg != null) {
			String deps_hex = HexData.byteArrayToHexString(MessageUtils.messageToBytes(deps_msg));

			int depsize = (deps_hex.length()/2)/1024;

			Logger.info(dprefix+"writing "+depsize+"k of JAR dependencies");
			
			pout.println(deps_hex);
			pout.flush();
		} else {
			String deps_hex = HexData.byteArrayToHexString(MessageUtils.messageToBytes(new Message()));

			int depsize = (deps_hex.length()/2)/1024;

			Logger.info(dprefix+"writing "+depsize+"k of JAR dependencies");
			
			pout.println(deps_hex);
			pout.flush();
		}

		Logger.info(dprefix+"wrote JAR dependencies");
		
		//write the classpath for the B2J component
		pout.println(classpath);
		pout.flush();
		
		Logger.info(dprefix+"waiting for actual address");
		
		//read the address it has bound to
		String act_address = bread.readLine();
		if (act_address == null) {
			try {
				throw new Exception("Daemon started engine but it terminated unexpectedly with exit code "+p.exitValue());
			} catch (Exception e) {
				
				//print the output of the failed process
				InstanceReader br = new InstanceReader("FAILED PROCESS"+"("+desc+"):",bread);
				br.start(); 
				InstanceReader er = new InstanceReader("FAILED PROCESS"+"("+desc+"):",eread);
				er.start(); 	
				
				try {
					Thread.sleep(1000);
				} catch (Exception x) {}
				
				throw new Exception("Expected port number for engine process but got '"+br.info+"' / '"+er.info+"', command was "+sb+"");
			}
		}
//		int nport = Integer.parseInt(sport);
		SessionAddress actual_address = null;
		try {
			actual_address = SessionAddress.fromString(HexData.hexStringToString(act_address.trim()));
		} catch (Exception e) {
			throw new Exception("Expected SessionAddress of bound controller/subcontroller but received: "+act_address.trim());
		}
		
		Logger.info(dprefix+"Got back address of component: "+actual_address);

		//get a prefix for this jvm instance
		long instance_time = System.currentTimeMillis();
		String prefix = jvm_count+"-"+instance_time;
		jvm_count++;

		//get the output streams and print them with the above prefix
		InstanceReader nr;
		nr = new InstanceReader(prefix+"("+desc+"):",bread);
		nr.start(); 		
		nr = new InstanceReader(prefix+"("+desc+"):",eread);
		nr.start(); 		

		JEngineProcess jp = new JEngineProcess(p,prefix,classname,description);

		synchronized(LIVE_PROCESSES_LOCK) {
			long jpid = ++live_process_id;
			jp.setId(jpid);
			
			live_processes.add(jp);
		}	

		System.gc();

		//do I need to wait for the process to die now? probably not		
		InstanceTracker tr = new InstanceTracker(prefix+":",jp);
		tr.start();

		System.gc();

//		return nport;
		return actual_address;
	}

	

	private SessionAddress forkController(int id, SessionAddress address, Message versions, JARDependency[] deps, Message deps_msg, String description) throws Exception {
		String jarpath = getVersion(versions);
		
		if (jarpath == null) {
			throw new Exception("Unsupported version - "+Version.toString(versions));
		}
		
		if (!FORK_PROCESSES) {
			if (!versions.equals(Version.getVersion())) {
				throw new Exception("Disallowing fork means controller version not available - "+Version.toString(versions)+" (only "+Version.getVersionAsString()+")");
			}
			Controller c = new Controller(address,id,deps_msg,jarpath);
			return c.getAddress();	
		}
		
		String[] vmargs = new String[] {"-Xmx128m"};
		String classname = Util.jengine_package+".mainengine.Controller";
		String[] args = new String[]{ ""+id };
		
		if (HAS_PASSWORD) {
			//we are supposed to be secure
			address.setRequiresPassword(true);
			address.setPassword(sr.nextLong()+"_"+sr.nextLong()+"_"+sr.nextLong());
		}
		
		return forkJVM(vmargs, jarpath, classname, args, deps, deps_msg, "Controller", HexData.stringToHexString(SessionAddress.toString(address)),description);
	}

	private SessionAddress forkSubController(int id, SessionAddress address, Message versions, JARDependency[] deps, Message deps_msg) throws Exception {
		String jarpath = getVersion(versions);
		
		if (jarpath == null) {
			throw new Exception("Unsupported version - "+versions);
		}
		
		if (!FORK_PROCESSES) {
			if (!versions.equals(Version.getVersion())) {
				throw new Exception("Disallowing fork means controller version not available - "+Version.toString(versions)+" (only "+Version.getVersionAsString()+")");
			}
			SubController c = new SubController(address,id);
			return c.getAddress();	
		}
		
		String[] vmargs = new String[] {"-Xmx256m"};
		String classname = Util.jengine_package+".mainengine.SubController";
		String[] args = new String[]{ ""+id };
	
		if (HAS_PASSWORD) {
			//we are supposed to be secure
			address.setRequiresPassword(true);
			address.setPassword(sr.nextLong()+"_"+sr.nextLong()+"_"+sr.nextLong());
		}
		
		return forkJVM(vmargs, jarpath, classname, args, deps, deps_msg, "SubController@"+address.getListenerHost(),HexData.stringToHexString(SessionAddress.toString(address)),"");
	}

	private static void loadProperties() throws Exception {
		synchronized (CONF_LOCK) {
			//
			// Only load the properties if the file has been modified since the last load
			// or if this is the first time
			//
			B2jConfig config = B2jPlatform.getB2jConfig();
			
			if (B2jPlatform.getB2jConfigLastModified() <= last_loaded) {
				return;
			}
			last_loaded = B2jPlatform.getB2jConfigLastModified();
//			File f = new File(cd+"b2j.conf");
//			
//			if (!f.exists()) {
//				FileOutputStream fout = new FileOutputStream(cd+"b2j.conf");
//				fout.write(PROP_DEF.getBytes());
//				fout.close();
//			}
//			
//			if (f.lastModified() <= last_loaded) {
//				return;
//			}
//			last_loaded = f.lastModified();
	
			//
			// Load the properties
			//		
//			File props = new File(cd+"b2j.conf");
			Logger.info(dprefix+"Engine configuration file path = "+new File("./conf/Default/conf.xml").getCanonicalPath());
//			FileInputStream fin = new FileInputStream(props);
			
			
//			Properties conf = new Properties();
//			conf.load(fin);	
			
//			ByteArrayOutputStream bout = new ByteArrayOutputStream();
//			conf.store(bout,"");
//			Logger.info(dprefix+"Engine Configuration Properties:\n"+new String(bout.toByteArray()));
			
			try {
				java_exe = config.getProperty(B2jConfig.PATH_JVMFORK_FORK_COMMAND_JAVAEXE);
				java_cp = config.getProperty(B2jConfig.PATH_JVMFORK_FORK_COMMAND_JAVA_CLASSPATH_ARG);
//				java_exe = conf.getProperty("java_exe");
//				java_cp = conf.getProperty("java_classpath_arg");
				
				ArrayList tmp = new ArrayList();
				String[] extra_jars = config.getProperties(B2jConfig.PATH_JVMFORK_EXTRA_CLASSPATH);
				for (int i = 0; i < extra_jars.length; i++) {
					tmp.add(extra_jars[i].trim());
				}
				java_extrajars = tmp;
				
/*				ArrayList extra_jars = new ArrayList();
				int istart = 0;
				boolean found = false;
				do {
					found = false;
					for (int i = istart; i < istart + 20; i++) {
						String tmp = conf.getProperty("java_classpath_extra"+i);
						if (tmp != null) {
							if (tmp.trim().length() > 0)  {
								extra_jars.add(tmp);
								found = true;
							}
						}
					}
					istart += 20;
				} while (found);
				
				java_extrajars = extra_jars;
				*/
//				java_jars = conf.getProperty("java_classpath");

				java_vmargs = config.getProperties(B2jConfig.PATH_JVMFORK_FORK_COMMAND_JAVA_VM_ARG);
/*				
				ArrayList vmargs = new ArrayList();
				
				int max = 10;
				for (int i = 0; i < max; i++) {
					String tmp = conf.getProperty("java_vmarg"+i);
					if (tmp != null) {
						if (tmp.length() > 0) vmargs.add(tmp);
						max = i+10;
					}
				}
				
				java_vmargs = new String[vmargs.size()];
				vmargs.toArray(java_vmargs);
				*/
				/*
				int argc = Integer.parseInt(conf.getProperty("java_args"));
				if (argc < 1) throw new Exception("Java exe argument count must be > 0");
				java_exe = new String[argc];
				
				for (int i = 0; i < argc; i++) {
					java_exe[i] = conf.getProperty("java_args"+i);
					if (java_exe[i] == null) java_exe[i] = "";	
				}
				*/
			} catch (Exception e) {
				e.printStackTrace();
//				fin.close();
				
				java_exe = null;
				java_cp = null;
				java_extrajars = null;
				java_vmargs = null;
				
				throw new Exception("error reading java command from config file");
			}

			try {
				if (config.getProperty(B2jConfig.PATH_SOAPDAEMON_LOG_INFO).equals("true")) {
					Logger.PRINT_INFO = true;
				}
//				if (conf.getProperty("log_info").trim().equalsIgnoreCase("true")) {
//					Logger.PRINT_INFO = true;
//				}
			} catch (Exception e) {
				//default
				Logger.PRINT_INFO = false;
			}
			
			try {
				FORK_PROCESSES = config.getProperty(B2jConfig.PATH_JVMFORK_FORK).equals("true");
//				FORK_PROCESSES = conf.getProperty("fork_jvm").trim().equalsIgnoreCase("true");
			} catch (NullPointerException e) {
				//default
				FORK_PROCESSES = true;
			}

			try {
				DAEMON_PORT = Integer.parseInt(config.getProperty(B2jConfig.PATH_SOAPDAEMON_PORT));
//				DAEMON_PORT = Integer.parseInt(conf.getProperty("daemon_port"));
			} catch (Exception e) {
				//default
				DAEMON_PORT = 11000;
			}

			try {
				USE_SSL = config.getProperty(B2jConfig.PATH_SOAPDAEMON_SSL).equals("true");
//				USE_SSL = conf.getProperty("daemon_ssl").trim().equalsIgnoreCase("true");
			} catch (NullPointerException e) {
				//default
				USE_SSL = false;
			}

			try {
				HAS_PASSWORD = config.getProperty(B2jConfig.PATH_SOAPDAEMON_USE_PASSWORD).equals("true");
//				HAS_PASSWORD = conf.getProperty("daemon_haspassword").trim().equalsIgnoreCase("true");
			} catch (NullPointerException e) {
				//default
				HAS_PASSWORD = false;
			}

			try {
				PASSWORD = config.getProperty(B2jConfig.PATH_SOAPDAEMON_PASSWORD);
//				PASSWORD = conf.getProperty("daemon_password").trim();
			} catch (NullPointerException e) {
				//default
				PASSWORD = null;
			}

			try {
				ArrayList tmp = new ArrayList();
//				String proxy_hosts = conf.getProperty("soap_proxy_excluded_hosts").trim();
				String proxy_hosts = config.getProperty(B2jConfig.PATH_SOAP_EXCLUDED_PROXY_HOSTS);
				
				int index;
				do {
					index = proxy_hosts.indexOf(',');
					String proxy_host;
					if (index != -1) {
						proxy_host = proxy_hosts.substring(0,index);
						proxy_hosts = proxy_hosts.substring(index+1);
					} else {
						proxy_host = proxy_hosts;
					}
					
					proxy_host = proxy_host.trim();
					if (proxy_host.length() > 0) {
						tmp.add(proxy_host);
					}
				} while (index != -1);
				
				SOAP_PROXY_EXCLUDED_HOSTS = new String[tmp.size()];
				tmp.toArray(SOAP_PROXY_EXCLUDED_HOSTS);
				
			} catch (NullPointerException e) {
				SOAP_PROXY_EXCLUDED_HOSTS = new String[0];
			}
			
			try {
				SOAP_HTTPS_PROXY_ON = config.getProperty(B2jConfig.PATH_SOAP_HTTPS_PROXY_USE).equals("true");
//				SOAP_HTTPS_PROXY_ON = conf.getProperty("soap_https_proxy_on").trim().equalsIgnoreCase("true");
			} catch (NullPointerException e) {
				SOAP_HTTPS_PROXY_ON = false;
			}
			
			try {
				SOAP_HTTPS_PROXY_HOST = config.getProperty(B2jConfig.PATH_SOAP_HTTPS_PROXY_HOST);
//				SOAP_HTTPS_PROXY_HOST = conf.getProperty("soap_https_proxy_host").trim();
			} catch (NullPointerException e) {
				SOAP_HTTPS_PROXY_ON = false;
			}
			
			try {
				SOAP_HTTPS_PROXY_PORT = Integer.parseInt(config.getProperty(B2jConfig.PATH_SOAP_HTTPS_PROXY_PORT));
//				SOAP_HTTPS_PROXY_PORT = Integer.parseInt(conf.getProperty("soap_https_proxy_port").trim());
			} catch (Exception e) {
				SOAP_HTTPS_PROXY_ON = false;
			}
			
			try {
				SOAP_HTTP_PROXY_ON = config.getProperty(B2jConfig.PATH_SOAP_HTTP_PROXY_USE).equals("true");
//				SOAP_HTTP_PROXY_ON = conf.getProperty("soap_http_proxy_on").trim().equalsIgnoreCase("true");
			} catch (NullPointerException e) {
				SOAP_HTTPS_PROXY_ON = false;
			}
			
			try {
				SOAP_HTTP_PROXY_HOST = config.getProperty(B2jConfig.PATH_SOAP_HTTP_PROXY_HOST);
//				SOAP_HTTP_PROXY_HOST = conf.getProperty("soap_http_proxy_host").trim();
			} catch (NullPointerException e) {
				SOAP_HTTP_PROXY_ON = false;
			}
			
			try {
				SOAP_HTTP_PROXY_PORT = Integer.parseInt(config.getProperty(B2jConfig.PATH_SOAP_HTTP_PROXY_PORT));
//				SOAP_HTTP_PROXY_PORT = Integer.parseInt(conf.getProperty("soap_http_proxy_port").trim());
			} catch (Exception e) {
				SOAP_HTTP_PROXY_ON = false;
			}
			
			
			if (HAS_PASSWORD) {
				if (PASSWORD == null) {
					PASSWORD="password";
				}
				if (PASSWORD == "") {
					PASSWORD="password";
				}
			}
			
			/*
			try {
				STDOUT_LOGFILE = conf.getProperty("logfile");
				if (STDOUT_LOGFILE.length() == 0) STDOUT_LOGFILE = null;
				
				if (STDOUT_LOGFILE != null) {
					FileOutputStream fout = new FileOutputStream(STDOUT_LOGFILE);
					PrintStream pout = new PrintStream(fout);
					System.setOut(pout);
					System.setErr(pout);
					
					System.out.println("STDOUT set to ");
				}
			} catch (Exception e) {
			}
			*/
	
			//
			// Print out the properties so the user can see them
			//		
			Logger.info(dprefix+"Loaded new Configuration:");

			Logger.info(dprefix+"\t(By Default) Daemon will listen on port "+DAEMON_PORT);

			Logger.info(dprefix+"\t");
			
			if (FORK_PROCESSES) {
				Logger.info(dprefix+"\tDaemon will fork JVM for engine components");
				Logger.info(dprefix+"\t");

				if (java_extrajars != null && java_extrajars.size() > 0) {
					for (int i = 0; i < java_extrajars.size(); i++) {
						Logger.info(dprefix+"\tForced Required JAR: "+java_extrajars.get(i));
					}
					Logger.info(dprefix+"\t");
				}
				
				if (java_exe != null) {
					Logger.info(dprefix+"\tUsing specific command to spawn controllers");
					Logger.info(dprefix+"\tJava executable: "+java_exe);
					Logger.info(dprefix+"\tJava classpath arg: "+java_cp);
//					Logger.info(dprefix+"\tJava base classpath: "+java_jars);
					for (int i = 0; i < java_vmargs.length; i++) {
						Logger.info(dprefix+"\tJava VM arg: "+java_vmargs[i]);
					}
					Logger.info(dprefix+"\t");
				} else {
					Logger.info(dprefix+"\tUsing cross-platform JVM fork instead of specific command");
					Logger.info(dprefix+"\t");
				}
			} else {
				Logger.info(dprefix+"\tDaemon will run engine components in current JVM");
				Logger.info(dprefix+"\t");
			}
			
			if (SOAP_HTTP_PROXY_ON) {
				Logger.info(dprefix+"\tSOAP HTTP proxy is ON ("+SOAP_HTTP_PROXY_HOST+":"+SOAP_HTTP_PROXY_PORT+")");
				Logger.info(dprefix+"\t");
			} else {
				Logger.info(dprefix+"\tSOAP HTTP proxy is OFF (direct HTTP connections)");
				Logger.info(dprefix+"\t");
			}

			if (SOAP_HTTPS_PROXY_ON) {
				Logger.info(dprefix+"\tSOAP HTTPS proxy is ON ("+SOAP_HTTPS_PROXY_HOST+":"+SOAP_HTTPS_PROXY_PORT+")");
				Logger.info(dprefix+"\t");
			} else {
				Logger.info(dprefix+"\tSOAP HTTPS proxy is OFF (direct HTTPS connections)");
				Logger.info(dprefix+"\t");
			}

			if (SOAP_HTTP_PROXY_ON || SOAP_HTTPS_PROXY_ON) {
				for (int i = 0;i < SOAP_PROXY_EXCLUDED_HOSTS.length; i++) {
					Logger.info(dprefix+"\tHost excluded from Proxies: "+SOAP_PROXY_EXCLUDED_HOSTS[i]);
				}
				Logger.info(dprefix+"\t");
			}

			if (HAS_PASSWORD) {
				Logger.info(dprefix+"\tPassword IS required");
				Logger.info(dprefix+"\t");
			} else {
				Logger.info(dprefix+"\tPassword IS NOT required");
				Logger.info(dprefix+"\t");
			}

			if (USE_SSL) {
				Logger.info(dprefix+"\tDaemon will accept ONLY SSL connections");
				Logger.info(dprefix+"\tDaemon will NOT accept standard TCP/IP connections");
				Logger.info(dprefix+"\t");
			} else {
				Logger.info(dprefix+"\tDaemon will NOT accept SSL connections");
				Logger.info(dprefix+"\tDaemon will accept ONLY standard TCP/IP connections");
				Logger.info(dprefix+"\t");
			}
			
			if (!HAS_PASSWORD || !USE_SSL) {
				Logger.info(dprefix+"\t************************");
				Logger.info(dprefix+"\t* Daemon IS NOT SECURE *");
				Logger.info(dprefix+"\t************************");
				Logger.info(dprefix+"\t");
			}
			
//			fin.close();

		}//end synch
	}


	public void run() {
		Thread th = Thread.currentThread();
		th.setName("B2J Daemon [Listener] Thread (PORT: "+port+")");
		
		readVersions();
		
		String postfix = "";
		if (USE_SSL) {
			postfix += " (HTTPS)";
		} else {
			postfix += " (HTTP)";
		}
		if (HAS_PASSWORD) {
			postfix += " (Password Required)";
		} else {
			postfix += " (No Password)";
		}
		Logger.direct(dprefix+"B2J daemon listening for connections on port "+port+postfix);

		System.gc();

		try {
//			Logger.direct(dprefix+"Adding HTTP listener to "+new URL(epr.getAddress()+"/public.wsdl"));
			HTTPServer.addHttpListener(new URL(epr.getAddress()+"/public.wsdl"),this);
			HTTPServer.addHttpListener(new URL(epr.getAddress()+"/"),this);
			
			SOAPFactory.addServerTransportListenerForEPR(daemon_service_epr,null,false,this);

			//wait forever (until the user closes the daemon)
			while (true) {
				try {
					 Thread.sleep(99999999);
				} catch (Exception e) {}
			}
		} catch (Throwable t) {
			Logger.direct("Failed to listen for incoming connections: "+t);
			Logger.error("Failed to listen for incoming connections",t);
		}
//		try {
//			serve();			
//		} catch (Throwable t) {
//			Logger.error(dprefix+"daemon socket server thread died",t);
//			try {
//				Thread.sleep(3000);
//			} catch (Exception e) {}
//		}
//		Logger.error(dprefix+"daemon exiting");
	}
	
	private void start_socket() throws IOException {
		System.gc();
		
		if (USE_SSL) {
			epr = new WSEndpointReference("https://localhost:"+port);
			daemon_service_epr = new WSEndpointReference("https://localhost:"+port+"/services/engine/");
		} else {
			epr = new WSEndpointReference("http://localhost:"+port);
			daemon_service_epr = new WSEndpointReference("http://localhost:"+port+"/services/engine/");
		}
	}

//	private void serve() throws Exception {
		
//		SOAPFactory.addServerTransportListenerForEPR(epr,null,false,this);
		
		/*
		int connection = 0;
		while (true) {
			
			
			Session session = null;
			try {
				session = SessionFactory.newSession(session_address,false);
			} catch (Exception e) {
				Logger.error(dprefix+"failed to create session from session address");
				//fatal
				throw e;
			}

			try {
				session.beginNonBlocking();
				session.waitUntilSessionTransportBound();
			} catch (Exception e) {
				Logger.warning(dprefix+"unable to bind to specified port: "+e);
				//fatal
				throw e;
			}
			
			try {
				session.waitUntilSessionTransportReady();
				try {
					Handler handler = new Handler(session,connection++);
					handler.start();
				} catch (Exception e) {
					Logger.warning(dprefix+"daemon connection handler died");
				}
			} catch (Exception e) {
				try {
					Thread.sleep(1000);
				} catch (Exception x) {
				}
				Logger.warning(dprefix+"problem accepting new session");
			}
			System.gc();
		}	
		*/
//	}

	public static void main(String[] args) {
		if (Switches.CONSTANT_GARBAGE_COLLECTION) {
			GCThread.startThread();
		}
		new SoapDaemon(args);
	} 

	public static String getStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method
	
	public static void addVersion(Message versions, byte[] dat) throws Exception {
		synchronized(VERSIONS_LOCK) {
			StringBuffer name = new StringBuffer(VERSION_PREFIX);
			for (int i = 0; i < versions.length(); i++) {
				name.append("-");
				name.append((Integer)versions.get(i));
			}
			name.append(".jar");
			
			File jar_file;
			FileOutputStream fout = null;
			
			boolean permanent = false;
			
			try {
				//try writing it to the cached versions directory
				jar_file = new File(cd+"versions/"+name.toString());
				fout = new FileOutputStream(jar_file);
				
				permanent = true;
			} catch (IOException e) {
				//if that fails then write it to a temp directory
				jar_file = File.createTempFile(VERSION_PREFIX,".jar");
				fout = new FileOutputStream(jar_file);
				
				permanent = false;
			}
			
			fout.write(dat,0,dat.length);
			fout.flush();
			fout.close();
			
			JARVersion ver = new JARVersion();
			ver.version = versions;
			ver.path = jar_file.getAbsolutePath();
			
			if (permanent) {
				version_jars.add(ver);
			} else {
				version_tmp_jars.add(ver);
			}
		}
	}
	
	public static void readVersions() {
		synchronized(VERSIONS_LOCK) {
			try {
				version_jars.clear();
				
				File version_dir = new File(cd+"versions");

				if (!version_dir.exists()) {
					version_dir.mkdirs();
				} else {
					//already exists - search for versions
					File[] files = version_dir.listFiles();
					for (int i = 0; i < files.length; i++) {
						String name = files[i].getName();
						
						Message versions = new Message();
						
						
						int until = 0;
						while (until < name.length()) {
							name = name.substring(until);
							
							int nextHyphen = name.indexOf('-');
							if (nextHyphen == -1) nextHyphen = name.length();
							
							int nextDot = name.indexOf('.');
							if (nextDot == -1) nextDot = name.length();
							
							until = Math.min(nextHyphen,nextDot);
							
							//get the substring
							String str = name.substring(0,until);
							
							//skip the delimiter
							until++;

							try {
								int n = Integer.parseInt(str);
								versions.append(n);
							} catch (Exception e) {
							}
						}

						if (versions.length() > 0) {
							//this seems to be a valid versioned jar
							JARVersion ver = new JARVersion();
							ver.version = versions;
							ver.path = files[i].getAbsolutePath();

							Logger.direct(dprefix+"Detected support for version "+ver);
							
							//add it to the list of versions we support
							version_jars.add(ver);
						}
					}
				}
				
			} catch (Exception e) {
			}
		}
	}

	public static String getVersion(Message versions) {
		if (versions.equals(Version.getVersion())) {
			//we support this version
			return basejar;
		}
		
		synchronized(VERSIONS_LOCK) {
			for (int i = 0; i < version_jars.size(); i++) {
				JARVersion ver = (JARVersion)version_jars.get(i);
				if (ver.version.equals(versions)) {
					//found the jar for this version - return the path
					return ver.path;
				}
			}
			
			for (int i = 0; i < version_tmp_jars.size(); i++) {
				JARVersion ver = (JARVersion)version_tmp_jars.get(i);
				if (ver.version.equals(versions)) {
					//found the jar for this version - return the path
					return ver.path;
				}
			}
			
			return null;
		}
	}

	public static class JARVersion {
		Message version;
		String path;
		
		public String toString() {
			StringBuffer sb = new StringBuffer();

			for (int i = 0; i < version.length(); i++) {
				if (i > 0) {
					sb.append("-");
				}
				sb.append(version.get(i));
			}
			
			sb.append(" (");
			sb.append(path.substring(path.lastIndexOf(File.separatorChar)+1));
			sb.append(")");
			return sb.toString();
		}
	}

	public String doRequest(int httpport, boolean rpc_type, String operationNamespace, String operation, String soapAction, String payload) throws Exception {

		try {
			/*
			boolean invalid = false;
			
			if (payload == null) {
				invalid = true;
			}
			if (payload.trim().length() == 0) {
				invalid = true;
			}
			
			if (invalid) {
				throw new SOAPNotFoundException("");
			}*/
			
			synchronized(TRANSACTION_LOCK) {
			
				try {
					loadProperties();
				} catch (Exception e) {
				}
				
				Element body_elem = SoapDaemonUtils.getDocumentEncodedMessageBody(payload);
	
				Element part_Password = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_PASSWORD);
				
				if (HAS_PASSWORD) {
					if (part_Password != null) {
						String pword = SoapDaemonUtils.elementToPassword(part_Password);
						if (pword.equals(PASSWORD)) {
							//OK - password matches
						} else {
							throw new Exception("password incorrect");
						}
					} else {
						throw new Exception("password required");
					}
				}
				
				StringBuffer responseXml = new StringBuffer();
				
				if (soapAction.equals(SOAPACTION_CREATE_CONTROLLER)) {
					//part - hex encoded SessionAddress XML
					Element part_SessionAddress = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_SESSION_ADDRESS);
					
					//part - dependencies
					Element part_Dependencies = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_DEPENDENCIES);
					
					//part - versions
					Element part_Versions = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_VERSIONS);

					//part - versions
					Element part_Description = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_DESCRIPTION);
					
					//Build SessionAddress
					SessionAddress address = SoapDaemonUtils.elementToSessionAddress(part_SessionAddress);
					
					//Build JAR dependencies
					JARDependency[] deps = SoapDaemonUtils.elementToJARDependency(part_Dependencies);
					
					String description = "";
					if (part_Description != null) {
						description = SoapDaemonUtils.elementToDescription(part_Description);
					} else {
						SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmssSSS");
						description = sdf.format(new Date());
					}
					
					Message dependencies = new Message();
					for (int i = 0; i < deps.length; i++) {
						dependencies.append(JARDependency.toMessage(deps[i]));
					}
					
					//Build Versions
					Message versions = SoapDaemonUtils.elementToVersion(part_Versions);
	
					address.setListenerPortMinimum(port+1);
					address.setListenerPortMaximum(SessionAddress.TRANSPORT_PORT_ANY);
					
					address = forkController(0,address,versions,deps,dependencies,description);
					
					responseXml.append(SoapDaemonUtils.sessionAddressToElement(address));
	
					System.gc();
					
				} else if (soapAction.equals(SOAPACTION_CREATE_SUBCONTROLLER)) {
					//part - integer ID
					Element part_ID = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_ID);
	
					//part - hex encoded SessionAddress XML
					Element part_SessionAddress = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_SESSION_ADDRESS);
					
					//part - dependencies
					Element part_Dependencies = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_DEPENDENCIES);
					
					//part - versions
					Element part_Versions = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_VERSIONS);
					
					//Build ID
					Integer id = new Integer(SoapDaemonUtils.elementToId(part_ID));
					
					//Build SessionAddress
					SessionAddress address = SoapDaemonUtils.elementToSessionAddress(part_SessionAddress);
					
					//Build JAR dependencies
					JARDependency[] deps = SoapDaemonUtils.elementToJARDependency(part_Dependencies);
					
					Message dependencies = new Message();
					for (int i = 0; i < deps.length; i++) {
						dependencies.append(JARDependency.toMessage(deps[i]));
					}
					
					//Build Versions
					Message versions = SoapDaemonUtils.elementToVersion(part_Versions);
					
					address.setListenerPortMinimum(port+1);
					address.setListenerPortMaximum(SessionAddress.TRANSPORT_PORT_ANY);
					
					address = forkSubController(id.intValue(),address,versions,deps,null);
	
					responseXml.append(SoapDaemonUtils.sessionAddressToElement(address));
	
					System.gc();
	
				} else if (soapAction.equals(SOAPACTION_GET_VERSION)) {
	
					Message version = Version.getVersion();
					
					responseXml.append(SoapDaemonUtils.versionToElement(version));
					
					System.gc();
					
				} else if (soapAction.equals(SOAPACTION_SUPPORTS_VERSION)) {
					//part - versions
					Element part_Versions = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_VERSIONS);
					
					//Build Versions
					Message versions = SoapDaemonUtils.elementToVersion(part_Versions);
					
					boolean supported = false;
	
					if (getVersion(versions) != null) {
						supported = true;
					} else {
						supported = false;
					}
					
					responseXml.append(SoapDaemonUtils.supportedToElement(supported));
					
					System.gc();
					
				} else if (soapAction.equals(SOAPACTION_ADD_VERSION)) {
					//part - versions
					Element part_Versions = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_VERSIONS);
					
					//part - JAR
					Element part_JAR = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_JAR);
					
					//Build Versions
					Message versions = SoapDaemonUtils.elementToVersion(part_Versions);
					
					//Build JAR
					byte[] dat = HexData.hexStringToByteArray(Util.getTextDirectlyUnder(part_JAR));
					
					addVersion(versions,dat);
					
					System.gc();

				} else if (soapAction.equals(SOAPACTION_LIST_ENGINES)) {
					//empty
					
					responseXml.append("<"+SoapDaemonUtils.TAG_ENGINES+">");
					
					synchronized(LIVE_PROCESSES_LOCK) {

						for (int i = 0; i < live_processes.size(); i++) {
							JEngineProcess proc = (JEngineProcess)live_processes.get(i);
							
							if (proc.getDescription().length() > 0) {
								responseXml.append(SoapDaemonUtils.engineToElement(proc.getId(),proc.getDescription()));
							}
						}
					}	
					
					responseXml.append("</"+SoapDaemonUtils.TAG_ENGINES+">");
					
					System.gc();
					
				} else if (soapAction.equals(SOAPACTION_KILL_ENGINE)) {
					
					//part - EngineInstance
					Element part_Instance = Util.getFirstElement(body_elem,SoapDaemonUtils.TAG_ENGINE);
					
					long id = Long.parseLong(Util.getTextDirectlyUnder(Util.getFirstElement(part_Instance,"id")));
					
					responseXml.append("<"+SoapDaemonUtils.TAG_ENGINES+">");

					synchronized(LIVE_PROCESSES_LOCK) {
						for (int i = 0; i < live_processes.size(); i++) {
							JEngineProcess proc = (JEngineProcess)live_processes.get(i);
							if (proc.getId() == id && proc.getDescription().length() > 0) {
								try {
									proc.destroy();
								} catch (Throwable t) {}
								live_processes.remove(i--);
							} else {
								if (proc.getDescription().length() > 0) {
									responseXml.append(SoapDaemonUtils.engineToElement(proc.getId(),proc.getDescription()));
								}
							}
						}
					}						

					responseXml.append("</"+SoapDaemonUtils.TAG_ENGINES+">");
					
					System.gc();
				}
				
				return SOAPUtils.basicWrapInSoapEnvelopeDocument(responseXml.toString());
			}
		
		} catch (Throwable t) {
			
			String s = t.getMessage();
			if (s == null) {
				s = SOAPUtils.getStackTrace(t);
			} else if (s.trim().length() == 0) {
				s = SOAPUtils.getStackTrace(t);
			}
			
			return SOAPUtils.basicWrapInSoapEnvelopeDocument(SoapDaemonUtils.errorToElement(s));
			
		}
	}
	
	private class InstanceReader extends Thread {
		String prefix;
		BufferedReader bread;
		String info = null;
		
		public InstanceReader(String prefix, BufferedReader bread) {
			this.prefix = prefix;
			this.bread = bread;
			setDaemon(true);
		}
		public void run() {
			try {
				while (true) {
					String s = bread.readLine();
					if (s == null) break;
					if (info == null) info = s; 
					Logger.log(prefix+s);
				}
			} catch (Exception e) {
			}
		}
	}

	/**
	 * Keeps track of live engine components and removes them from the live list
	 * when they die
	 */
	private class InstanceTracker extends Thread {
		JEngineProcess p;
		String prefix;
		public InstanceTracker(String prefix, JEngineProcess p) {
			this.prefix = prefix;
			this.p = p;
		}
		public void run() {
			try {
				p.waitFor();
				synchronized(LIVE_PROCESSES_LOCK) {
					p.destroy();	
				}
			} catch (Throwable e) {
				synchronized(LIVE_PROCESSES_LOCK) {
					p.destroy();	
				}
			}
			
			try {
				int n = p.exitValue();
				Logger.direct(prefix+" PROCESS EXITED, returning "+n);
			} catch (Exception e) {
				Logger.direct(prefix+" PROCESS EXITED, returning <unknown>");
			}

			try {
				synchronized(LIVE_PROCESSES_LOCK) {
					live_processes.remove(p);
				}
			} catch (Exception e) {
			}
		}
	}

	private class ProcessKillerThread extends Thread {
		public void run() {
			Logger.direct(dprefix+"Destroying any live engine components...");
			
			synchronized (LIVE_PROCESSES_LOCK) {
				for (int i = 0; i < live_processes.size(); i++) {
					try {
						JEngineProcess p = (JEngineProcess)live_processes.get(i);
						
						try {
							p.destroy();
	
							try {
								int n = p.exitValue();
								Logger.direct(dprefix+"PROCESS "+p.prefix+" DESTROYED <"+n+"> "+p);
							} catch (Exception e) {
								Logger.direct(dprefix+"PROCESS "+p.prefix+" DESTROYED <unknown> "+p);
							}
						} catch (Throwable e) {
							Logger.direct(dprefix+"UNABLE TO DESTROY PROCESS "+p.prefix+" "+p);
						}	
					} catch (Throwable t) {
						//?
					}
				}	
			}
				
		}
	}

	private class JEngineProcess {
		Process p;
		String prefix,name;
		String description;
		long id;
		JEngineProcess(Process p, String prefix, String name, String description) {
			this.p = p;
			this.prefix = prefix;
			this.name = name;
			this.description = description;
		}
		
		public void setId(long l) {
			id = l;
		}
		
		public long getId() {
			return id;
		}
		
		public String getDescription() {
			return description;
		}

		public void waitFor() throws InterruptedException{
			p.waitFor();	
		}
		
		public void destroy() {
			p.destroy();	
		}	

		public int exitValue() {
			return p.exitValue();	
		}
		
		public String toString() {
			return name;
		}
	}

	public String doRequest(int port, String resource, String payload, HashMap lc_headers) throws HTTPException {
		try {
		if (resource.toLowerCase().indexOf("wsdl") != -1) {
			String protocol = "http";
			if (USE_SSL) {
				protocol = "https";
			}
			
			return 
				"<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
			+ 	"<definitions xmlns=\"http://schemas.xmlsoap.org/wsdl/\" "
			+	"targetNamespace=\"http://www.eclipse.org/stp/b2j/2006/02\" "
			+	"xmlns:tns=\"http://www.eclipse.org/stp/b2j/2006/02\" " 
			+	"xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" " 
			+	"xmlns:wsdl=\"http://schemas.xmlsoap.org/wsdl/\" " 
			+	"xmlns:soap=\"http://schemas.xmlsoap.org/wsdl/soap/\" "
			+	">"
			
			+	"<types> "
			+	"<schema "
			+	"targetNamespace=\"http://www.eclipse.org/stp/b2j/2006/02\" "
			+	"xmlns:tns=\"http://www.eclipse.org/stp/b2j/2006/02\" "
			+	"xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" "
			+	"xmlns=\"http://www.w3.org/2001/XMLSchema\"> "

			+	"<complexType name=\"Versions\"> "
			+	"<sequence> "
			+	"<element name=\"Version\" type=\"xsd:int\" minOccurs=\"1\" maxOccurs=\"unbounded\" nillable=\"false\" /> "
			+	"</sequence> "
			+	"</complexType> "

			+	"<complexType name=\"EngineInstance\"> "
			+	"<sequence> "
			+	"<element name=\"id\" type=\"xsd:long\" minOccurs=\"1\" maxOccurs=\"1\" nillable=\"false\" /> "
			+	"<element name=\"description\" type=\"xsd:string\" minOccurs=\"0\" maxOccurs=\"1\" nillable=\"false\" /> "
			+	"</sequence> "
			+	"</complexType> "
			
			+	"<complexType name=\"EngineInstances\"> "
			+	"<sequence> "
			+	"<element name=\"EngineInstance\" type=\"tns:EngineInstance\" minOccurs=\"0\" maxOccurs=\"unbounded\" nillable=\"false\" /> "
			+	"</sequence> "
			+	"</complexType> "
			
			+	"</schema> "
			+	"</types> "
			
			+	"<message name=\"GetVersionRequest\"> "
			+	"<part name=\"Password\" type=\"xsd:string\" /> "
			+	"</message> "
			
			+	"<message name=\"GetVersionResponse\"> "
			+	"<part name=\"Versions\" type=\"tns:Versions\" /> "
			+	"<part name=\"Error\" type=\"xsd:string\" /> "
			+	"</message> "

			+	"<message name=\"ListEnginesRequest\"> "
			+	"</message> "

			+	"<message name=\"ListEnginesResponse\"> "
			+	"<part name=\"EngineInstances\" type=\"tns:EngineInstances\" /> "
			+	"</message> "

			+	"<message name=\"TerminateEnginesRequest\"> "
			+	"<part name=\"EngineInstance\" type=\"tns:EngineInstance\" /> "
			+	"</message> "

			+	"<message name=\"TerminateEnginesResponse\"> "
			+	"<part name=\"EngineInstances\" type=\"tns:EngineInstances\" /> "
			+	"</message> "
			
			+	"<portType name=\"B2jDaemon\"> "
			+	"<operation name=\"getVersion\"> "
			+	"<input message=\"tns:GetVersionRequest\" name=\"GetVersionRequest\" /> "
			+	"<output message=\"tns:GetVersionResponse\" name=\"GetVersionResponse\" /> "
			+	"</operation> "
			+	"<operation name=\"listEngines\"> "
			+	"<input message=\"tns:ListEnginesRequest\" name=\"ListEnginesRequest\" /> "
			+	"<output message=\"tns:ListEnginesResponse\" name=\"ListEnginesResponse\" /> "
			+	"</operation> "
			+	"<operation name=\"terminateEngine\"> "
			+	"<input message=\"tns:TerminateEnginesRequest\" name=\"TerminateEnginesRequest\" /> "
			+	"<output message=\"tns:TerminateEnginesResponse\" name=\"TerminateEnginesResponse\" /> "
			+	"</operation> "
			+	"</portType> "

		    +	"<binding name=\"B2jDaemonSoapBinding\" type=\"tns:B2jDaemon\"> "
			+	"<soap:binding transport=\"http://schemas.xmlsoap.org/soap/http\" style=\"document\"/> "

			+	"<operation name=\"getVersion\"> "
			+	"<soap:operation soapAction=\""+SOAPACTION_GET_VERSION+"\" style=\"document\"/> "
			+	"<wsdl:input name=\"GetVersionRequest\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:input> "
			+	"<wsdl:output name=\"GetVersionResponse\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:output> "
			+	"</operation> "

			+	"<operation name=\"listEngines\"> "
			+	"<soap:operation soapAction=\""+SOAPACTION_LIST_ENGINES+"\" style=\"document\"/> "
			+	"<wsdl:input name=\"ListEnginesRequest\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:input> "
			+	"<wsdl:output name=\"ListEnginesResponse\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:output> "
			+	"</operation> "

			+	"<operation name=\"terminateEngine\"> "
			+	"<soap:operation soapAction=\""+SOAPACTION_KILL_ENGINE+"\" style=\"document\"/> "
			+	"<wsdl:input name=\"TerminateEnginesRequest\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:input> "
			+	"<wsdl:output name=\"TerminateEnginesResponse\" > "
			+	"<soap:body use=\"literal\"/> "
			+	"</wsdl:output> "
			+	"</operation> "
			
			+	"</binding> "

			+	"<service> "
			+	"<port name=\"B2jDaemonPort\" binding=\"tns:B2jDaemonSoapBinding\"> "
			+	"<soap:address location=\""+protocol+"://"+InetAddress.getLocalHost().getHostAddress()+":"+port+"/services/engine/\"/> "
			+	"</port> "
			+	"</service> "			

			+	"</definitions> "			
			;
		} else if (payload == null) {
			StringBuffer sb = new StringBuffer();
			
			sb.append("<ul>");
			sb.append("<li><a href=\"/public.wsdl\">Daemon Public WSDL (/public.wsdl)</a></li>");
			sb.append("</ul>");
			
			return HTTPUtils.getHtml("B2J Engine Daemon",sb.toString());
		}
		} catch (Exception e) {
			throw new HTTPException(""+e);
		}
		throw new HTTPNotFoundException("Not Found");
	}

}