/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.multiplex;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;

/**
 * @author amiguel
 *
 * An InputStream implementation that can have blocks of byte array level data
 * dropped in to the buffer at any time.
 * 
 * Automatically manages read blocking and exceptions etc.
 */
public class ByteArrayInBuffer extends InputStream {

public static final int INFINITE_BUFFER = -1;
public static final int DEFAULT_BUFFER = 400000;

Object buffer_LOCK = new Object();

LinkedList bufs = new LinkedList();
int len = 0;

byte[] buffer = new byte[0];	
int ptr = 0;

boolean closed = false;
IOException ioexception = null;

boolean waiting_for_data = false;
boolean waiting_for_space = false;

int max_buffer = DEFAULT_BUFFER;

	public ByteArrayInBuffer() {
	}
	public ByteArrayInBuffer(int max_buffer) {
		this.max_buffer = max_buffer;
	}

	public int length() {
		synchronized(buffer_LOCK) {
			return len + (buffer.length-ptr);	
		}
	}

	private void waitForData() {
		while(len <= 0 && !closed) {
			try {
				waiting_for_data = true;
				buffer_LOCK.wait();
				waiting_for_data = false;
			} catch (InterruptedException e) {}
		}
	}
	
	public void setClosed() {
		closed = true;
		synchronized(buffer_LOCK) {
			if (waiting_for_data) {
				buffer_LOCK.notifyAll();
			}
		}
	}
	public void setClosed(IOException t) {
		closed = true;
		ioexception = t;
		synchronized(buffer_LOCK) {
			if (waiting_for_data) {
				buffer_LOCK.notifyAll();
			}
		}
	}

	public int read() throws IOException {
		synchronized(buffer_LOCK) {
			if (closed && len == 0) {
				if (ioexception != null) throw ioexception;
				return -1;
			}
			
			get();
	
			if (closed && len == 0) {
				if (ioexception != null) throw ioexception;
				return -1;
			}
		
			return 0x000000FF & ((int)buffer[ptr++]);
		}
	} 
	public int read(byte[] b) throws IOException {
		return read(b,0,b.length);
	} 
	public int read(byte[] b, int off, int length) throws IOException {
		synchronized(buffer_LOCK) {
			if (closed && len == 0) {
				if (ioexception != null) throw ioexception;
				return -1;
			}
			
			get();
	
			if (closed && len == 0) {
				if (ioexception != null) throw ioexception;
				return -1;
			}
	
			int tmplen = buffer.length - ptr;
			
			if (tmplen <= length) {
				//buffer can take all available data
				System.arraycopy(buffer,ptr,b,off,tmplen);
	
				ptr+= tmplen;

				
				return tmplen;
			} else {
				//we can fill buffer
				System.arraycopy(buffer,ptr,b,off,length);
				
				ptr += length;
				
				return length;
			}
		}
	}  

	private void get() {
		if (ptr >= buffer.length) {
			//this should really be ==
			waitForData();
			if (!closed) {
				buffer = (byte[])bufs.removeFirst();
				len -= buffer.length;
				ptr = 0;
			}

			if (waiting_for_space) {
				buffer_LOCK.notifyAll();
			}
		} 	
	}

	public void add(byte[] dat) {
		if (dat.length > 0) {
			synchronized(buffer_LOCK) {
				
				while (dat.length + len + (buffer.length-ptr) > max_buffer
						&& len > 0
						&& max_buffer >= 0) {
					
					waiting_for_space = true;
					try {
						buffer_LOCK.wait();
					} catch (InterruptedException e) {}
					waiting_for_space = false;
				}
				
				bufs.addLast(dat);
				len += dat.length;
				
				if (waiting_for_data) {
					buffer_LOCK.notify();
				}
			}
		}
	}	
}