/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.multiplex;

import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.stp.b2j.core.jengine.internal.mutex.UnqueuedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;

/**
 * @author amiguel
 * 
 * A class that takes any InputStream and multiplexes it into multiple
 * streams.  Assumes that the underlying data has been written using a
 * MultiplexerOutputStream.
 */
public class MultiplexerInputStream implements MultiplexingInput {
InputStream instream;

Object lock = new Object();
Object notifier = new Object();
HashMap instreams = new HashMap();
HashMap locks = new HashMap();

boolean closed = false;

IOException ioexception;

	public MultiplexerInputStream(InputStream in) {
		instream = new BufferedInputStream(in);
		
		ReaderThread rthread = new ReaderThread();
		rthread.setDaemon(true);
		rthread.start();
	}

	public MultiplexerInputStream(InputStream in, String readername) {
		instream = new BufferedInputStream(in);
		
		ReaderThread rthread = new ReaderThread(readername);
		rthread.setDaemon(true);
		rthread.start();
	}

	public MultiplexerInputStream(InputStream in, String readername, ThreadGroup tg) {
		instream = new BufferedInputStream(in);
		
		ReaderThread rthread = new ReaderThread(tg,readername);
		rthread.setDaemon(true);
		rthread.start();
	}
	
	public UnqueuedMutex getLock(short n) {
		Integer N = new Integer(n);
		synchronized(lock) {
			UnqueuedMutex nlock = (UnqueuedMutex)locks.get(N);
			if (nlock == null) {
				nlock = new UnqueuedMutex();
				locks.put(N,nlock);
			}
			return nlock;
		}
	}

	public void closeAll() {
		closed = true;
		synchronized(lock) {
			Collection tmp = instreams.values();
			Iterator it = tmp.iterator();
			while (it.hasNext()) {
				((ByteArrayInBuffer)it.next()).setClosed();	
			}	
		}
	}
	public void closeAll(IOException e) {
		closed = true;
		ioexception = e;
		synchronized(lock) {
			Collection tmp = instreams.values();
			Iterator it = tmp.iterator();
			while (it.hasNext()) {
				((ByteArrayInBuffer)it.next()).setClosed(e);	
			}	
		}
	}
	
	public InputStream getInputStream(short n) {
		Integer N = new Integer(n);
		synchronized(lock) {
			ByteArrayInBuffer in = (ByteArrayInBuffer)instreams.get(N);
			if (in == null) {
				in = new ByteArrayInBuffer();
				instreams.put(N,in);	
			}
			if (closed) {
				if (ioexception != null) {
					in.setClosed(ioexception);
				} else {
					in.setClosed();
				}
			}
			return in;
		}
	}
	
	class ReaderThread extends Thread {
		ThreadGroup tg = null;
		public ReaderThread() {
		}
		public ReaderThread(String name) {
			super(name);
		}
		public ReaderThread(ThreadGroup tg, String name) {
			super(tg,name);
			this.tg = tg;
		}
		
		public void run() {
			try {
				int n = 0;
				byte[] dat = null;
				while (!closed) {
					n = StreamUtils.readShort(instream);
					dat = StreamUtils.readNBytes(instream,10000000);//~10 megs
					
					if (dat.length > 0) {
						//route this to the correct place
						Integer N = new Integer(n);
						synchronized(lock) {
							ByteArrayInBuffer in = (ByteArrayInBuffer)instreams.get(N);
							if (in == null) {
								in = new ByteArrayInBuffer();
								instreams.put(N,in);
							}
							in.add(dat);
						}
					}
				}
			} catch (EOFException e) {
				closeAll();/*
				synchronized(lock) {
					//set each baib to closed
					Iterator it = instreams.values().iterator();
					while (it.hasNext()) {
						ByteArrayInBuffer bain = (ByteArrayInBuffer)it.next();
						bain.setClosed();	
					}
				}*/
				if (tg != null) {
					tg.uncaughtException(this,e);
				}
				
			} catch (IOException e) {
				closeAll(e);/*
				synchronized(lock) {
					//set each baib to closed(e)
					Iterator it = instreams.values().iterator();
					while (it.hasNext()) {
						ByteArrayInBuffer bain = (ByteArrayInBuffer)it.next();
						bain.setClosed(e);	
					}
				}*/
				if (tg != null) {
					tg.uncaughtException(this,e);
				}
				
			} catch (Throwable t) {
				closeAll(new IOException("unknown error "+t));
				
				Logger.error("unexpected error in MultiplexerInputStream",t);
				if (tg != null) {
					tg.uncaughtException(this,t);
				}
				
			}
		}	
	}
}