/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.multiplex;

import java.io.BufferedOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.stp.b2j.core.jengine.internal.mutex.UnqueuedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;

/**
 * @author amiguel
 * 
 * A class that takes any OutputStream and multiplexes it into multiple
 * streams.  Assumes that the underlying data will be read using a
 * MultiplexerInputStream.
 */
public class MultiplexerOutputStream implements MultiplexingOutput {
OutputStream outstream;
private Object lock = new Object();

HashMap streams = new HashMap();
HashMap locks = new HashMap();

IOException ioexception;

private static final int MINCHUNK = 1024;

private static final int MAXCHUNK = 4096;

	public MultiplexerOutputStream(OutputStream out) {
		this.outstream = new BufferedOutputStream(out);
	}

	public MultiplexerOutputStream(OutputStream out, String name) {
		this.outstream = new BufferedOutputStream(out);
	}

	public MultiplexerOutputStream(OutputStream out, String name, ThreadGroup tg) {
		this.outstream = new BufferedOutputStream(out);
	}
	
	public void closeAll(IOException err) {
		ioexception = err;
	}
	
	public UnqueuedMutex getLock(short n) {
		Integer N = new Integer(n);
		synchronized(lock) {
			UnqueuedMutex nlock = (UnqueuedMutex)locks.get(N);
			if (nlock == null) {
				nlock = new UnqueuedMutex();
				locks.put(N,nlock);
			}
			return nlock;
		}
	}
	
	public OutputStream getOutputStream(short n) {
		synchronized(streams) {
			Integer N = new Integer(n);
			OutputStream out = (OutputStream)streams.get(N);
			if (out == null) {
				out = new MultiplexingOutput((short)n);
				streams.put(N,out);
			}
			return out;
		}
	}
	
	public void flushAll() throws IOException {
		Collection tmp = streams.values();
		Iterator it = tmp.iterator();
		while (it.hasNext()) {
			((MultiplexingOutput)it.next()).flush();	
		}	
	}

	class MultiplexingOutput extends FilterOutputStream {
		short n;
		Object boutlock = new Object();
		
		public MultiplexingOutput(short n) {
			super(outstream);
			this.n = n;
		}
		
		OpenByteArrayOutputStream bout = new OpenByteArrayOutputStream();

		private void writeChunk() throws IOException {
			synchronized(lock) {
				if (bout.size() > 0) {
/* 					
					byte[] tosend = bout.getByteArray();
					int tosendlength = bout.size();
					
					//dont send it all at once - send it Nk at a time
					int sent = 0;
					while (sent + MAXCHUNK < tosendlength) {
						StreamUtils.writeShort(outstream,n);
						StreamUtils.writeBytes(outstream,tosend,sent,MAXCHUNK);
						sent += MAXCHUNK;
					}
					
					if (sent < tosend.length) {
						//send remainder
						StreamUtils.writeShort(outstream,n);
						StreamUtils.writeBytes(outstream,tosend,sent,tosendlength-sent);
					}
*/				
					StreamUtils.writeShort(outstream,n);
					StreamUtils.writeBytes(outstream,bout.getByteArray(),0,bout.size());

					bout.reset();
				}
			}
		}

		public void flush() throws IOException {
			if (ioexception != null) throw ioexception;
			writeChunk();
			outstream.flush();
			
//			synchronized(lock) { 
//				if (bout.size() > 0) {
//					StreamUtils.writeShort(outstream,n);
//					StreamUtils.writeBytes(outstream,bout.getByteArray());
//
//					outstream.flush();
//
//					bout.reset();
//				}
//			}
		} 

		//
		// NOTE - we dont synchronize when we write to bytearrayoutputstream
		// only one thread should write to this stream at once, therefore
		// we only need to synchronize when we write data to the single 
		// main output stream
		//
		
		public void write(byte[] b) throws IOException {
			if (ioexception != null) throw ioexception;
			bout.write(b); 
			if (bout.size() > MINCHUNK) {
				flush();	
			}
		} 
		public void write(byte[] b, int off, int len) throws IOException {
			if (ioexception != null) throw ioexception;
			bout.write(b,off,len); 
			if (bout.size() > MINCHUNK) {
				flush();	
			}
		} 
		public void write(int b) throws IOException {
			if (ioexception != null) throw ioexception;
			bout.write(b); 
			if (bout.size() > MINCHUNK) {
				flush();	
			}
		}  		
		
	}
}