/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi.engine;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Properties;

import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.ControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.DaemonInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.TraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.utils.EngineListenerTraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.DependencyInfo;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.program.BPELProgram;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * A public API which represents an engine capable of running a BPELProgram program
 * 
 * @author aem
 *
 */
public class IndependantLocalBPELEngine {
	
	BPELProgram program;
	
	Object running_LOCK = new Object();
	boolean running = false;
	
	ControllerInterface controller = null;
	Message root_runner_ids = null;

	/**
	 * Create a new BPEL engine to run a BPEL program
	 * @param program the BPEL program to run
	 */
	public IndependantLocalBPELEngine(BPELProgram program) {
		this.program = program;
	}

	/**
	 * Run a BPEL program (will use the JVM local engine)
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram() throws Exception {
		runProgram(null,false,false,true);
	}

	/**
	 * Run a BPEL program
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(boolean terminateEngineOnDisconnect) throws Exception {
		runProgram(null,false,false,terminateEngineOnDisconnect);
	}
	
	/**
	 * Run a BPEL program
	 * @param debug whether to run the engine in debug mode (slower)
	 * @param verbose whether to produce verbose output when processing the input BPEL file etc.
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(boolean debug, boolean verbose, boolean terminateEngineOnDisconnect) throws Exception {
		runProgram(null,debug,verbose,terminateEngineOnDisconnect);
	}

	/**
	 * Run a BPEL program
	 * @param listener the trace listener to attach to the engine
	 * @param debug whether to run the engine in debug mode (slower)
	 * @param verbose whether to produce verbose output when processing the input BPEL file etc.
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(BPELEngineListener listener, boolean debug, boolean verbose, boolean terminateEngineOnDisconnect) throws Exception {
		synchronized(running_LOCK) {
			runInternal(program,listener,debug,verbose,terminateEngineOnDisconnect,false);
		}
	}
	
	/**
	 * Forcibly terminate this BPEL program
	 * @throws Exception if an error occurs while waiting for this BPEL program to terminate
	 */
	public void terminate() throws Exception {
		controller.terminate();
	}
	
	/**
	 * Wait for the BPEL program to terminate
	 * @throws Exception if an error occurs while waiting for this BPEL program to terminate
	 */
	public void waitForProgramCompletion() throws Exception {
		
		synchronized(running_LOCK) {
			if (!running) {
				return;
			}
		
			for (int i = 0; i < root_runner_ids.length(); i++) {
				Long id = (Long)root_runner_ids.get(i);	
	//			try {
					controller.joinRunner(id);
	//			} catch (Exception e) {
	//				console.debug("Failed to wait on program finish");
	//				console.debug(getStacktrace(e));
	//				B2jPlugin.DBG.logVisibleError(e,"Failed to wait on program finish",false);
	//				return;
	//			}
			}
	
			//give the data reading and printing thread a chance to finish reading any stuff
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {}
	
			//close the connection to the engine
			controller.closeConnection();
			
			running = false;
		}
	}

	
	private Program runInternal(BPELProgram bpel_program, BPELEngineListener console, boolean debug, boolean verbose, boolean exitOnDisconnect, boolean validateOnly) throws Exception {

		if (console == null) {
			console = new IgnoredTraceListener();
		}
		
		TraceListener console_tracel = new EngineListenerTraceListener(console);
		
		Program prog = (Program)bpel_program.getCompiledData("Program");
		ArrayList program_deps = (ArrayList)bpel_program.getCompiledData("Program Dependencies");

		SessionAddress[] workers = bpel_program.getWorkerHosts();
		SessionAddress[] worker_daemons = bpel_program.getWorkerDaemons();
		
		for (int i = 0; i < workers.length; i++) {
			prog.addHostAddress(workers[i],worker_daemons[i]);
		}

		ArrayList engine_deps = fetchAllEngineDependencies(console);
		
		//
		//remove any multiples from Program deps that are included in Engine deps
		//
		for (int i = 0; i < program_deps.size(); i++) {
			JARDependency dep = (JARDependency)program_deps.get(i);
			for (int k = 0; k < engine_deps.size(); k++) {
				JARDependency tmp = (JARDependency)engine_deps.get(k);
				if (dep.getFilePath().equals(tmp.getFilePath())) {
					program_deps.remove(i--);
					break;
				}
			}
		}
		
		JARDependency[] engine_deps_array = dependencyListToArray(engine_deps);
		JARDependency[] program_deps_array = dependencyListToArray(program_deps);
		
		if (engine_deps_array.length > 0 || program_deps_array.length > 0) {
			console.printInfo("");
			console.printInfo("Dependencies:");
		}

		for (int i = 0; i < engine_deps_array.length; i++) {
			console.printInfo("  (Engine Dependency)  "+engine_deps_array[i].getFilePath());
		}
		for (int i = 0; i < program_deps_array.length; i++) {
			console.printInfo("  (Program Dependency)  "+program_deps_array[i].getFilePath());
		}

		if (engine_deps_array.length > 0 || program_deps_array.length > 0) {
			console.printInfo("");
		}
		
		DaemonInterface daemon = null;
		try {
			//connect to the engine daemon 
			console.printInfo("Connecting to mini engine daemon");
			daemon = new org.eclipse.stp.b2j.core.jengine.internal.miniengine.api.DaemonConnector();
		} catch (Exception e) {
			printFatalProblem(console,"Failed to connect to engine daemon",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to connect to engine daemon",false);
			return null;
		}
		
		//start up the engine
		controller = null;
		try {
			console.printInfo("Creating engine instance");
			controller = daemon.newEngine(bpel_program.getName(),console_tracel,engine_deps_array,bpel_program.getCoordinatorHost());
			
			//set JAR dependencies
			
			daemon.close();
		} catch (Exception e) {
			printFatalProblem(console,"Failed to create new engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to create new engine instance",false);
			return null;
		}
		
		try {
			console.printInfo("Setting engine program");
			controller.setProgram(prog);
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set engine program in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set engine program in engine instance",false);
			return null;
		}

		try {
			if (!exitOnDisconnect) {
				controller.setHeadless(true);
				console.printInfo("Headless mode, engine will continue to run after workbench disconnects");
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set headless in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set headless in engine instance",false);
			return null;
		}
		
		try {
			if (debug) {
				console.printInfo("Debug mode, turning on all engine logging");
				controller.setLogLevel(true,true,false);
//				controller.setLogLevel(true,true,true);
			} else {
				console.printInfo("Turning on only warning and error logging");
				controller.setLogLevel(true,true,false);
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set log level in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set log level in engine instance",false);
			return null;
		}

		try {
			console.printInfo("Launching engine program");
			console.printInfo("");
			root_runner_ids = controller.launchRunner(1,"engine_main",0,new String[0]);
		} catch (Exception e) {
			printFatalProblem(console,"Failed to start engine program",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to start engine program",false);
			return null;
		}
		
		running = true;
		
		return null;
	}
	
	//
	// Utility methods
	//
	
	/**
	 * Convert a list of JARDependency objects to an array
	 * @param list
	 * @return
	 */
	private JARDependency[] dependencyListToArray(ArrayList list) {
		JARDependency[] array = new JARDependency[list.size()];
		list.toArray(array);
		return array;
	}
		
	private void printFatalProblem(BPELEngineListener console, String description, Exception e) {
		console.printDebug("\n\nERROR: "+description);
		console.printInfo("\nProblem Description:");
		if (e.getMessage() != null) {
			console.printDebug(e.getMessage());
		} else {
			console.printDebug("");
		}
		console.printInfo("\nProblem Details:");
		console.printDebug(getStacktrace(e));
	}
	
	private static String getStacktrace(Throwable t) {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		PrintStream print = new PrintStream(bout);
		t.printStackTrace(print);
		return new String(bout.toByteArray()).replace('\t',' ');
	}


	/**
	 * Fetch all the JAR dependencies which the underlying java engine has (e.g. transport
	 * extension point JARs etc)
	 * @param console
	 * @return
	 */
	private ArrayList fetchAllEngineDependencies(BPELEngineListener console) throws IOException {
		DependencyInfo[] infos = B2jPlatform.getEngineDependencyInfo();
		return fetchAllDependencies(console,infos);
	}
	
	private ArrayList fetchAllDependencies(BPELEngineListener console, DependencyInfo[] infos) throws IOException {
		ArrayList jardeps = new ArrayList();
		
		for (int i = 0; i < infos.length; i++) {
			Properties[] props = infos[i].getResources();
			for (int k = 0; k < props.length; k++) {
				String path = props[k].getProperty("JAR");
				
				path = infos[i].getRelativePath(path);

				InputStream in = new FileInputStream(path);
				in = new BufferedInputStream(in);
				
				byte[] dat = StreamUtils.readAll(in);
				
				in.close();
				
				if (path != null) {
					JARDependency dep = new JARDependency(dat,path);
					jardeps.add(dep);
				}
			}
		}
		return jardeps;
	}
	
	private class IgnoredTraceListener implements BPELEngineListener {

		public void printInfo(String s) {
		}

		public void printDebug(String s) {
		}

		public void printEngineInfo(String s) {
		}

		public void printEngineDebug(String s) {
		}
	}
}