/******************************************************************************
* Copyright (c) 2006, Intalio Inc.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     Intalio Inc. - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.bpmn.validation.specification;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.validation.AbstractModelConstraint;
import org.eclipse.emf.validation.IValidationContext;
import org.eclipse.stp.bpmn.Activity;
import org.eclipse.stp.bpmn.ActivityType;
import org.eclipse.stp.bpmn.BpmnPackage;
import org.eclipse.stp.bpmn.MessagingEdge;

/**
 * Messaging edges constraints computed from the BPMN specification.
 * See BPMN 1.0 paragraph 8.4.2
 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulm�</a>
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class MessagingEdgeConstraint extends AbstractModelConstraint {

	@Override
	public IStatus validate(IValidationContext ctx) {
		if (ctx.getTarget() instanceof MessagingEdge) {
			if (ctx.getFeature() == null || // batch mode
					ctx.getFeature(). // live mode
						getFeatureID() == BpmnPackage.MESSAGING_EDGE) {
				MessagingEdge edge = (MessagingEdge) ctx.getTarget();
				Activity source = edge.getSource();
				Activity target = edge.getTarget();
				if (source == null || target == null) { // invalid message.
					return ctx.createSuccessStatus();
				}
				switch (source.getActivityType().getValue()) {
				case ActivityType.EVENT_INTERMEDIATE_COMPENSATION:
				case ActivityType.EVENT_INTERMEDIATE_EMPTY:
				case ActivityType.EVENT_INTERMEDIATE_ERROR:
				case ActivityType.EVENT_INTERMEDIATE_MESSAGE:
				case ActivityType.EVENT_INTERMEDIATE_RULE:
				case ActivityType.EVENT_INTERMEDIATE_TIMER:
					if (((FeatureMap.Entry) source.
							getOrderedMessages().get(0)).getValue() == edge) {
						// a little bent to the spec
						// let's let the events be able to reply
						return ctx.createFailureStatus(new Object[]{
								"Intermediate events should not be the source of a " +
								"messaging flow, unless they are replying to a previous" +
						" messaging flow "});
					}
					break;
				case ActivityType.EVENT_START_EMPTY:
				case ActivityType.EVENT_START_MESSAGE:
				case ActivityType.EVENT_START_RULE:
					if (((FeatureMap.Entry) source.
							getOrderedMessages().get(0)).getValue() == edge) {
						return ctx.createFailureStatus(new Object[]{
								"Start events should not be the source of a " +
								"messaging flow, unless they are replying to a previous" +
								" messaging flow "
						});
					}
					break;
				}

				switch (target.getActivityType().getValue()) {	
				case ActivityType.EVENT_END_COMPENSATION:
				case ActivityType.EVENT_END_EMPTY:
				case ActivityType.EVENT_END_ERROR:
				case ActivityType.EVENT_END_MESSAGE:
				case ActivityType.EVENT_END_TERMINATE:
					if (((FeatureMap.Entry) target.
							getOrderedMessages().get(0)).getValue() == edge) {
						return ctx.createFailureStatus(new Object[]{
								"End events should not be the target of a messaging flow, " +
								"unless they are receiving the reply of the messaging flow they sent"
						});
					}
					break;
				}
			}
			
		}
		return ctx.createSuccessStatus();
	}

}
