/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.introspection;

import org.eclipse.emf.common.util.URI;
import org.eclipse.stp.core.internal.introspection.ModelIntrospectionManager;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Implementation;

/**
 * Provides a hook into the introspection framework to bind ComponentTypes to
 * the introspector that understands the underlying model.
 * 
 * <p>
 * The methods declared on this interface do not throw checked exceptions
 * (although Runtime exceptions such as
 * {@link org.eclipse.stp.core.infrastructure.assertion.AssertionFailedException}
 * could be thrown), but instead will set the status of the given SCAObject if a
 * problem occurs. Clients may check for problems using
 * {@link org.eclipse.stp.core.sca.SCAObject#getStatus()}.
 * </p>
 * 
 * <p>
 * This interface is not intended to be implemented by clients.
 * </p>
 * 
 * @since 1.0
 */
public interface IModelIntrospectionManager {

   /**
    * The singleton instance for the introspection manager.
    */
   IModelIntrospectionManager eINSTANCE = new ModelIntrospectionManager();

   /**
    * Bind a ComponentType to a specific implementation. In general, clients
    * will not need to call this method.
    * 
    * <p>
    * Clients which acquire the ComponentType from
    * {@link org.eclipse.stp.core.sca.Component#resolveComponentType()} should
    * NOT call this method.
    * </p>
    * 
    * @param componentType
    *           A non-null componentType
    * @param implementation
    *           The implementation URI of the backing model (e.g. a fully
    *           qualified Java class)
    */
   void bind(ComponentType componentType, URI implementation);

   /**
    * Strip the modeled ComponentType from the implementation which backs it.
    * The ComponentType will have no properties, services, or references after
    * this method.
    * 
    * <p>
    * No changes will be made to the underlying model as a result of this
    * method.
    * </p>
    * 
    * @param componentType
    *           A non-null componenttype that has been bound.
    */
   void unbind(ComponentType componentType);

   /**
    * Resolve will return a {@link ComponentType} bound to the given
    * implementation.
    * 
    * <p>
    * The ComponentType can be shared in some cases. If the implementation
    * introspector defines an {@link IShareableComponentTypeFactory}, then the
    * implementation model will be converted to URI and requested through the
    * ResourceSet of the Implementation that is supplied.
    * </p>
    * <p>
    * Therefore, resolve() should only be supplied implementations that are
    * contained by a Component, which are contained by a ResourceSet. (The
    * actual containment is Implementation > Component > Composite/Module >
    * SCACoreRoot > Resource > ResourceSet).
    * </p>
    * <p>
    * The one special case is AbstractImplementation ({@link org.eclipse.stp.core.sca.SCAPackage#getAbstractImplementation()})
    * which does not need to be contained to be resolved.
    * </p>
    * <p>
    * The feature from the implementation will be determined using the
    * FeatureMap of the containing {@link org.eclipse.stp.core.sca.Component}.
    * </p>
    * 
    * @param implementation
    *           A contained Implementation
    * @return The (possibly shared) bound ComponentType derived from the given
    *         Implementation.
    */
   ComponentType resolve(Implementation implementation);

   /**
    * @param implementation
    *           A contained Implementation
    * @return The (possibly shared) bound ComponentType derived from the given
    *         Implementation.
    */
   // Not ready yet ComponentType resolve(Implementation implementation,
   // IProject context);
}
