/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.internal.infrastructure.emf;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EventObject;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.EditModelDisposedException;
import org.eclipse.stp.core.infrastructure.emf.EditModelEvent;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModel;
import org.eclipse.stp.core.infrastructure.emf.IEditModelListener;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IHumanInteractionController;
import org.eclipse.stp.core.infrastructure.emf.IResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.infrastructure.emf.ITechnologyFlavor;
import org.eclipse.stp.core.infrastructure.emf.WorkbenchResourceHelper;
import org.eclipse.stp.core.infrastructure.validateedit.ResourceStateInputProvider;
import org.eclipse.wst.common.internal.emf.utilities.ExtendedEcoreUtil;

/**
 * An implementation of the IEditModelScribbler API interface.
 * 
 * This class should never be referenced directly as public methods which are
 * not part of the IEditModelScribbler interface are subject to change.
 * 
 */
public class EditModelScribbler implements IEditModelScribbler,
      CommandStackListener, ResourceStateInputProvider {

   private static final IScribblerDomain[] NO_DOMAINS                = new IScribblerDomain[0];

   private static final Class              EMF_COMMAND_STACK_CLASS   = CommandStack.class;

   private static final Class              IEDITMODELSCRIBBLER_CLASS = IEditModelScribbler.class;

   // this may should be converted to an array here instead of a set
   private final Set                       scribblerDomains;

   private IHumanInteractionController     userInterfaceController;

   private ScribblerEMFCommandStack        emfCmdStck;

   private final ScribblerResourceManager  internalState;

   // TODO Complete edit model listener
   private static class ScribblerResourceManager implements IEditModelListener {

      protected static class ResourcePartition {
         private static final Resource[] NO_RESOURCES = new Resource[0];

         protected final Resource[]      deletedResources;

         protected final Resource[]      modifiedResources;

         protected final Resource[]      readOnlyResources;

         protected ResourcePartition(Resource[] modified, Resource[] readOnly,
               Resource[] deleted) {

            modifiedResources = modified != null ? modified : NO_RESOURCES;
            deletedResources = deleted != null ? deleted : NO_RESOURCES;
            readOnlyResources = readOnly != null ? readOnly : NO_RESOURCES;
         }
      }

      protected final ReadOnlyAdapter        readOnlyAdapter        = new ReadOnlyAdapter();

      protected final DeletedResourceAdapter deletedResourceAdapter = new DeletedResourceAdapter();

      private WeakReference                  scribblerReference;

      protected final boolean                isReadOnly;

      private final EditModel                editModel;

      protected final Set                    resources              = new HashSet();

      /*
       * resources that are deleted locally are not in this scribbler's
       * "internalState.resources" so we must track them in a separate set.
       * other scribblers can detected deleted resources in their
       * "internalState.resources" set by looking for an instance of the
       * deletedResourceAdapter on the resource.
       */
      protected final Set                    deletedResources       = new HashSet();

      private boolean                        isVaporized            = false;

      private ScribblerResourceManager(EditModelScribbler aScribbler,
            EditModel theEditModel, boolean toMakeReadOnly) {

         scribblerReference = new WeakReference(aScribbler);
         isReadOnly = toMakeReadOnly;
         editModel = theEditModel;
         try {
            editModel.addListener(this);
         } catch (EditModelException e) {
            EMFInfrastructurePlugin.log(EMFInfrastructurePlugin
                  .createErrorStatus(0, e.getLocalizedMessage(), e));
         }
      }

      public void editModelChanged(IEditModel notifyingEditModel,
            EditModelEvent anEvent) {
         Assert.isTrue(notifyingEditModel == editModel);
         EditModelScribbler scribbler = getEnclosingScribbler();
         if (scribbler == null) {
            EMFInfrastructurePlugin.logError(0,
                  Messages.A_scribbler_has_not_been_, null);
            UnreleasedScribblerException[] exceptions = editModel
                  .computeUnreleasedScribblers();
            for (int i = 0; i < exceptions.length; i++)
               EMFInfrastructurePlugin.logError(0,
                     Messages.The_following_exception_, exceptions[i]);
            vaporize();
         } else {
            switch (anEvent.getEventCode()) {
               case EditModelEvent.LOADED_RESOURCE: {
                  Iterator itr = scribbler.scribblerDomains.iterator();
                  IScribblerDomain domain;
                  List changedResources = anEvent.getChangedResources();
                  Resource resource = null;
                  while (itr.hasNext()) {
                     domain = (IScribblerDomain) itr.next();
                     for (int i = 0; i < changedResources.size(); i++) {
                        resource = (Resource) changedResources.get(i);
                        Assert.isTrue(!deletedResourceAdapter
                              .isDeleted(resource));
                        // TODO Maybe change the following to
                        // domain.getResourceDescriptor() to find the
                        // value of isLoadAsReadOnly
                        if (domain.isContained(resource)) {
                           /*
                            * false indicates that the edit model does not need
                            * to process the resource as the notification
                            * originated from the edit model
                            */
                           manageResource(resource, false);
                        }
                     }
                  }
                  break;
               }
            }
         }
      }

      public final ResourcePartition partitionResources() {

         Set modifiedResources = new HashSet();
         Set readOnlyResources = new HashSet();
         /*
          * scribblers coordinated shared deleted resources with each other.
          * resources deleted by other scribblers will be handled by this search
          * algorithm as well.
          */
         Set allDeletedResources = new HashSet();

         for (Iterator iter = resources.iterator(); iter.hasNext();) {
            Resource resource = (Resource) iter.next();
            if (isReadOnly(resource))
               readOnlyResources.add(resource);
            else if (resource.isModified()) {
               /* check if one of the other scribblers deleted the resource */
               if (DeletedResourceAdapter.findAdapter(resource) == null)
                  modifiedResources.add(resource);
               else
                  allDeletedResources.add(resource);
            }
         }

         Resource[] modifiedArray = null;
         if (modifiedResources.size() > 0)
            modifiedArray = (Resource[]) modifiedResources
                  .toArray(new Resource[modifiedResources.size()]);

         Resource[] readOnlyArray = null;
         if (readOnlyResources.size() > 0)
            readOnlyArray = (Resource[]) readOnlyResources
                  .toArray(new Resource[readOnlyResources.size()]);

         /*
          * resources that are deleted locally are not in this scribbler's
          * "internalState.resources" so we must track them in a separate set
          */
         if (deletedResources.size() > 0)
            allDeletedResources.addAll(deletedResources);

         Resource[] deletedArray = null;
         if (allDeletedResources.size() > 0)
            deletedArray = (Resource[]) allDeletedResources
                  .toArray(new Resource[allDeletedResources.size()]);

         return new ResourcePartition(modifiedArray, readOnlyArray,
               deletedArray);
      }

      public final Set getResources() {
         return Collections.unmodifiableSet(resources);
      }

      public boolean hasDeletedResources() {
         return deletedResources.size() > 0;
      }

      /**
       * @param aResource
       * @return
       */
      public boolean isResourceManaged(Resource aResource) {
         return resources.contains(aResource);
      }

      public boolean delete(Resource aResource, IProgressMonitor aMonitor)
            throws CoreException {
         Assert.isNotNull(getEnclosingScribbler());
         synchronized (getEnclosingScribbler()) {
            unprotectedUnmanageResource(aResource);
            // set the resource to modified so
            // that other scribblers will notice
            aResource.setModified(true);
            deletedResources.add(aResource);
            IFile physicalFile = WorkbenchResourceHelper.getFile(aResource);
            if (physicalFile.isAccessible()) {
               physicalFile.delete(true, true, aMonitor);
               return true;
            }
            return false;
         }
      }

      public void revert(IProgressMonitor aMonitor) throws EditModelException,
            CoreException {
         if (isReadOnly || isVaporized)
            throw new EditModelException(Messages.Cannot_revert_changes_t_);

         ScribblerResourceManager.ResourcePartition partition = partitionResources();

         for (int i = 0; i < partition.deletedResources.length; i++)
            deletedResourceAdapter.undeleteResource(
                  partition.deletedResources[i], aMonitor);

         editModel.handleRevert(partition.modifiedResources, aMonitor);

         /*
          * we must only clear the deleted resources *after* the call to
          * partitionResources()
          */
         deletedResources.clear();

      }

      public void vaporize() {

         /*
          * vaporize() can be called when the scribbler wants to dispose its
          * state or when the scribbler has been garbage collected. Therefore,
          * vaporize() must adapt its lock to the appropriate monitor
          */
         Object lock = getEnclosingScribbler();
         if (lock == null)
            lock = this;

         synchronized (lock) {
            if (isVaporized)
               return;
            isVaporized = true;
            EditModelScribbler scribbler = getEnclosingScribbler();
            if (scribbler != null)
               editModel.removeScribbler(scribbler);

            // remove listeners and remove state, deref resources
            editModel.removeListener(this);
            Resource[] managedResources = (Resource[]) resources
                  .toArray(new Resource[resources.size()]);

            // here we can call unprotected since we have protected
            // with the above sync(lock)
            for (int i = 0; i < managedResources.length; i++)
               unprotectedUnmanageResource(managedResources[i]);
         }
      }

      /**
       * @return
       */
      private EditModelScribbler getEnclosingScribbler() {
         return (EditModelScribbler) scribblerReference.get();
      }

      /**
       * @param aResource
       */
      private void manageResource(Resource aResource, boolean toProcessResource) {

         Assert.isNotNull(getEnclosingScribbler());
         synchronized (getEnclosingScribbler()) {
            if (!isResourceManaged(aResource)) {
               ScribblerAdapter adapter = (ScribblerAdapter) EcoreUtil
                     .getAdapter(aResource.eAdapters(),
                           ScribblerAdapter.ADAPTER_TYPE);
               Assert.isNotNull(adapter);
               try {
                  if (isReadOnly)
                     adapter.accessForRead();
                  else
                     adapter.accessForWrite();
               } catch (EditModelException e) {
                  editModel.getErrorHandler()
                        .handleGeneralFailure(aResource, e);
               }

               resources.add(aResource);

               if (toProcessResource)
                  editModel.enableResourceTracking(aResource);
            }
         }
      }

      /*
       * This method should only be called from unmanageResource or from
       * vaporize(). vaporize() can be called when the enclosing scribbler has
       * gone null, and hence the assertion
       * Assert.isNotNull(getEnclosingScribbler()) in unmanageResource() will
       * fail. We open this method up for vaporize to call to avoid this case.
       */
      private void unprotectedUnmanageResource(Resource aResource) {
         try {
            Assert.isTrue(isResourceManaged(aResource),
                  Messages.An_attempt_was_made_to_r_);
            ScribblerAdapter adapter = (ScribblerAdapter) EcoreUtil.getAdapter(
                  aResource.eAdapters(), ScribblerAdapter.ADAPTER_TYPE);
            Assert.isNotNull(adapter);

            if (isReadOnly)
               adapter.releaseFromRead();
            else
               adapter.releaseFromWrite();
         } catch (EditModelException e) {
            editModel.getErrorHandler().handleGeneralFailure(aResource, e);
         }

         resources.remove(aResource);
      }

      public boolean isDirty() {
         if (isReadOnly)
            return false;
         if (hasDeletedResources())
            return true;
         ResourcePartition partition = partitionResources();

         Resource[] theResources = partition.modifiedResources;
         Resource resource;
         for (int i = 0; i < theResources.length; i++) {
            resource = theResources[i];
            if (resource.isLoaded() && resource.isModified())
               return true;
         }

         return false;
      }

      public boolean isReadOnly(Resource resource) {
         return readOnlyAdapter.findAdapter(resource) == readOnlyAdapter;
      }

   }

   EditModelScribbler(EditModel anEditModel, IScribblerDomain[] theDomains,
         boolean toMakeReadOnly) {
      internalState = new ScribblerResourceManager(this, anEditModel,
            toMakeReadOnly);

      if (theDomains == null || theDomains.length == 0)
         scribblerDomains = Collections.EMPTY_SET;
      else
         scribblerDomains = new HashSet(Arrays.asList(theDomains));

      init();
   }

   private void init() {

      for (Iterator scribblerIter = scribblerDomains.iterator(); scribblerIter
            .hasNext();) {
         IScribblerDomain scribbler = (IScribblerDomain) scribblerIter.next();
         IResourceDescriptor[] descriptors = scribbler.getResourceDescriptors();
         for (int i = 0; i < descriptors.length; i++) {
            IPath localPath = descriptors[i].getLocalPath();
            Resource resource;
            try {
               resource = internalState.editModel.getResource(URI
                     .createPlatformResourceURI(localPath.toString()), true,
                     false);

               if (resource != null) {
                  if (descriptors[i].isLoadAsReadOnly())
                     markResourceAsReadOnly(resource);
                  internalState.manageResource(resource, true); /*
                                                                   * true =
                                                                   * request
                                                                   * editmodel
                                                                   * process
                                                                   * resource
                                                                   */
               }

            } catch (EditModelDisposedException emde) {
               // This case shouldn't occur; but could happen in a
               // multi-threaded environment
               EMFInfrastructurePlugin.log(EMFInfrastructurePlugin
                     .createErrorStatus(0, emde.getLocalizedMessage(), emde));
               break; // exit the for loop
            } catch (EditModelException eme) {
               EMFInfrastructurePlugin.log(EMFInfrastructurePlugin
                     .createErrorStatus(0, eme.getLocalizedMessage(), eme));
            }
         }
      }
   }

   public Resource getResource(IPath aWorkspaceRelativePath) {
      return getResource(URI.createPlatformResourceURI(aWorkspaceRelativePath
            .toString()), true);
   }

   public Resource getResource(IPath aWorkspaceRelativePath, int options) {
      Resource resource = getResource(URI
            .createPlatformResourceURI(aWorkspaceRelativePath.toString()),
            (options & OPTION_LOAD_RESOURCE) != 0);

      if ((options & OPTION_LOAD_AS_READ_ONLY) != 0)
         markResourceAsReadOnly(resource);

      return resource;
   }

   public Resource getResource(IResourceDescriptor aDescriptor) {
      Resource resource = getResource(aDescriptor, OPTION_LOAD_RESOURCE);
      if (aDescriptor.isLoadAsReadOnly())
         markResourceAsReadOnly(resource);

      return resource;
   }

   public Resource getResource(IResourceDescriptor aDescriptor, int options) {

      String workspaceRelativePath = aDescriptor.getLocalPath().toString();
      URI descriptorURI = URI.createURI(aDescriptor.getRequestProtocol()
            + workspaceRelativePath);
      Resource resource = getResource(descriptorURI,
            (options & OPTION_LOAD_RESOURCE) != 0);

      if ((options & OPTION_LOAD_AS_READ_ONLY) != 0
            || aDescriptor.isLoadAsReadOnly())
         markResourceAsReadOnly(resource);

      return resource;
   }

   public Resource getResource(URI aUri) {
      return getResource(aUri, true);
   }

   public Resource getResource(URI aUri, int options) {
      Resource resource = getResource(aUri,
            (options & OPTION_LOAD_RESOURCE) != 0);

      if ((options & OPTION_LOAD_AS_READ_ONLY) != 0)
         markResourceAsReadOnly(resource);

      return resource;
   }

   private Resource getResource(URI aUri, boolean loadOnDemand) {
      complainIfDisposed();

      Resource foundResource = null;
      synchronized (this) {
         if (aUri == null || aUri.segmentCount() == 0)
            return null;

         try {
            Resource resource = null;
            for (Iterator resourcesItr = internalState.resources.iterator(); resourcesItr
                  .hasNext()
                  && foundResource == null;) {
               resource = (Resource) resourcesItr.next();
               if (ExtendedEcoreUtil.endsWith(resource.getURI(), aUri))
                  foundResource = resource;
            }
            if (foundResource != null) {
               if (loadOnDemand && !foundResource.isLoaded())
                  try {
                     foundResource.load(null);
                  } catch (IOException e) {
                     internalState.editModel.getErrorHandler()
                           .handleLoadFailed(aUri, e);
                  }
               return foundResource;
            }

            try {
               foundResource = internalState.editModel.getResource(aUri,
                     false /* local only */, loadOnDemand);
            } catch (EditModelException e) {
               EMFInfrastructurePlugin.log(EMFInfrastructurePlugin
                     .createErrorStatus(0, e.getLocalizedMessage(), e));
            }

            if (foundResource != null)
               internalState
                     .manageResource(foundResource, false /* process resource */);
         } catch (RuntimeException e) {
            internalState.editModel.getErrorHandler().handleLoadFailed(aUri, e);
         }
      }

      return foundResource;
   }

   /**
    * @param resource
    */
   private void markResourceAsReadOnly(Resource resource) {
      if (!isReadOnly(resource))
         resource.eAdapters().add(internalState.readOnlyAdapter);
   }

   private void complainIfDisposed() {
      if (internalState.isVaporized)
         throw new IllegalStateException(Messages.The_IEditModelScribbler_);
   }

   public Resource createResource(IResourceDescriptor aResourceDescriptor,
         IProgressMonitor monitor) {
      // TODO Auto-generated method stub

      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public boolean deleteResource(IResourceDescriptor aResourceDescriptor,
         IProgressMonitor monitor) throws EditModelException {
      Resource resource = getResource(aResourceDescriptor, OPTION_NONE);
      return deleteResource(resource, monitor);
   }

   public boolean deleteResource(Resource aResource, IProgressMonitor monitor)
         throws EditModelException {
      if (isReadOnly(aResource))
         throw new EditModelException(NLS.bind(
               Messages.Attempt_to_delete_a_readonly_resource, aResource));

      boolean result = false;
      synchronized (this) {
         monitor.beginTask(NLS.bind(Messages.Deleting_resource_0, aResource
               .getURI()), 1);
         try {
            result = internalState.delete(aResource, monitor);
         } catch (CoreException e) {
            throw new EditModelException(e.getMessage());
         }
         monitor.worked(1);
         monitor.subTask(Messages.Finished_deleting_resour_);
         monitor.done();
      }
      return result;
   }

   private boolean isReadOnly(Resource resource) {
      if (internalState.isReadOnly(resource))
         return true;

      return false;
   }

   // implement me correctly please! help!
   public Resource[] getSortedResources() {
      return (Resource[]) internalState.getResources().toArray(
            new Resource[internalState.getResources().size()]);
   }

   public String getLabel() {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public boolean matches(IUndoContext context) {
      if (context instanceof EditModelScribbler) {
         EditModelScribbler otherScribbler = ((EditModelScribbler) context);
         // check that the edit model matches by reference
         if (otherScribbler.internalState.editModel == internalState.editModel) {
            IScribblerDomain[] otherDomains = createDomainArray(((IEditModelScribbler) context));
            IScribblerDomain[] myDomains = createDomainArray(this);
            boolean matchedOne = false;
            // verify that for each of myDomains
            for (int myDomainsIndx = 0; myDomainsIndx < myDomains.length; myDomainsIndx++) {
               // that at least one of the other domains
               for (int otherDomainsIndx = 0; otherDomainsIndx < otherDomains.length
                     && !matchedOne; otherDomainsIndx++) {
                  // provides a satisfactory match for my domain
                  if (myDomains[myDomainsIndx]
                        .matches(otherDomains[otherDomainsIndx]))
                     matchedOne = true;
               }
               // if there wasn't a match for my last then stop looking
               if (!matchedOne)
                  return false;
               // otherwise reset 'matched' for another round
               matchedOne = false;
            }
            return true;
         }
      }
      return false;
   }

   private static IScribblerDomain[] createDomainArray(
         IEditModelScribbler scribbler) {
      if (scribbler == null || scribbler.getScribblerDomains().size() == 0)
         return NO_DOMAINS;
      Set domains = scribbler.getScribblerDomains();
      return (IScribblerDomain[]) domains.toArray(new IScribblerDomain[domains
            .size()]);
   }

   public void commandStackChanged(EventObject event) {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public void close(IProgressMonitor monitor) throws EditModelException {
      complainIfDisposed();
      synchronized (this) {
         if (!internalState.isReadOnly && isDirty()) {
            IStatus directionStatus = EMFInfrastructurePlugin.createStatus(
                  IStatus.INFO, 0, Messages.Would_you_like_to_save_y_, null);
            if (getUserInterfaceController() != null) {
               IHumanInteractionController.Response response = getUserInterfaceController()
                     .requestDirection(directionStatus);
               if (response.shouldProceed() != null
                     && response.shouldProceed().booleanValue()) {
                  try {
                     save(true, monitor);
                  } finally {
                     disconnect();
                  }
               } else
                  discard(monitor);
            } else {
               try {
                  save(false, monitor);
               } finally {
                  disconnect();
               }
            }
         } else {
            disconnect();
         }
      }
   }

   public void discard(IProgressMonitor monitor) throws EditModelException {

      if (monitor == null)
         monitor = new NullProgressMonitor();

      synchronized (this) {
         try {
            monitor.beginTask(Messages.Discarding_changes_to_resources,
                  internalState.resources.size() + 2);
            monitor.worked(1);

            internalState.revert(monitor);
         } catch (CoreException e) {
            EMFInfrastructurePlugin.logError(0, e.getMessage(), e);
            throw new EditModelException(e.getMessage());
         } finally {
            monitor.subTask(Messages.Disconnecting_from_the_u_);
            disconnect();
            monitor.worked(1);
            monitor.subTask(Messages.Completed_discarding_cha_);
            monitor.done();
         }
      }

   }

   public void save(boolean force, IProgressMonitor monitor)
         throws EditModelException {

      if (monitor == null)
         monitor = new NullProgressMonitor();

      synchronized (this) {
         if (internalState.isReadOnly || internalState.isVaporized)
            throw new EditModelException(Messages.Cannot_save_resources_lo_);

         if (force || !internalState.editModel.isShared(this)) {
            Resource[] resourcesToSave = internalState.partitionResources().modifiedResources;
            internalState.editModel.handleSave(resourcesToSave, monitor);
            internalState.deletedResources.clear();
         }
      }
   }

   public void revert(IProgressMonitor monitor) throws EditModelException {

      if (monitor == null)
         monitor = new NullProgressMonitor();

      synchronized (this) {
         try {
            internalState.revert(monitor);
         } catch (CoreException e) {
            EMFInfrastructurePlugin.logError(0, e.getMessage(), e);
            throw new EditModelException(e.getMessage());
         }
      }
   }

   public void setUserInterfaceController(
         IHumanInteractionController aUserInterfaceController) {
      userInterfaceController = aUserInterfaceController;
   }

   public IHumanInteractionController getUserInterfaceController() {
      return userInterfaceController;
   }

   public IStatus execute(IUndoableOperation operation,
         IProgressMonitor monitor, IAdaptable args) {
      if (monitor == null)
         monitor = new NullProgressMonitor();
      monitor.beginTask(Messages.Executing_operation, 10);
      IStatus status = null;
      try {
         operation.addContext(this);

         status = internalState.editModel.getOperationHistory().execute(
               operation, monitor, args);
      } catch (EditModelException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
         EMFInfrastructurePlugin.log(status);
      } catch (ExecutionException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
      }
      return status;
   }

   public IStatus undo(IUndoableOperation operation, IProgressMonitor monitor,
         IAdaptable args) {
      if (monitor == null)
         monitor = new NullProgressMonitor();
      IStatus status = null;

      try {
         status = internalState.editModel.getOperationHistory().undoOperation(
               operation, monitor, args);
      } catch (EditModelException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
         EMFInfrastructurePlugin.log(status);
      } catch (ExecutionException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
      }
      return status;
   }

   public IStatus redo(IUndoableOperation operation, IProgressMonitor monitor,
         IAdaptable args) {
      if (monitor == null)
         monitor = new NullProgressMonitor();
      IStatus status = null;
      try {
         status = internalState.editModel.getOperationHistory().redoOperation(
               operation, monitor, args);
      } catch (EditModelException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
         EMFInfrastructurePlugin.log(status);
      } catch (ExecutionException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
      }
      return status;
   }

   public IStatus undoLast(IProgressMonitor monitor, IAdaptable args) {
      if (monitor == null)
         monitor = new NullProgressMonitor();

      IStatus status = null;
      try {
         status = internalState.editModel.getOperationHistory().undo(this,
               monitor, args);
      } catch (EditModelException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
         EMFInfrastructurePlugin.log(status);
      } catch (ExecutionException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
      }
      return status;
   }

   public IStatus redoLast(IProgressMonitor monitor, IAdaptable args) {
      if (monitor == null)
         monitor = new NullProgressMonitor();
      IStatus status = null;
      try {
         status = internalState.editModel.getOperationHistory().redo(this,
               monitor, args);
      } catch (EditModelException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
         EMFInfrastructurePlugin.log(status);
      } catch (ExecutionException e) {
         status = EMFInfrastructurePlugin.createErrorStatus(0, e
               .getLocalizedMessage(), e);
      }
      return status;
   }

   public void flushOperationHistory(IProgressMonitor monitor) {
      if (monitor != null)
         monitor.beginTask(Messages.Flushing_the_operation_h_, 2);
      try {
         internalState.editModel.getOperationHistory()
               .dispose(this, true /* flushUndo */, true/* flushRedo */,
                     false/* flushCtx */);
      } catch (EditModelException e) {
         EMFInfrastructurePlugin.log(EMFInfrastructurePlugin.createErrorStatus(
               0, e.getLocalizedMessage(), e));
      }
      if (monitor != null)
         monitor.beginTask(Messages.Completed_flushing_the_o_, 2);
   }

   public boolean isDirty() {

      return internalState.isDirty();
   }

   public boolean isReadOnly() {
      return internalState.isReadOnly;
   }

   public List getResources() {
      return Collections.unmodifiableList(new ArrayList(internalState
            .getResources()));
   }

   public List getNonResourceFiles() {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public List getNonResourceInconsistentFiles() {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public void cacheNonResourceValidateState(List roNonResourceFiles) {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);

   }

   public ITechnologyFlavor createTechnologyFlavor(
         String technologyFlavorLifecycleId) {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public ITechnologyFlavor createTechnologyFlavor(
         String technologyFlavorLifecycleId, IAdaptable input) {
      // TODO Auto-generated method stub
      throw new UnsupportedOperationException(
            Messages.Method_not_yet_implement_);
   }

   public Set getScribblerDomains() {
      return Collections.unmodifiableSet(scribblerDomains);
   }

   public Object getAdapter(Class adapter) {
      if (adapter == EMF_COMMAND_STACK_CLASS) {
         if (emfCmdStck == null)
            emfCmdStck = new ScribblerEMFCommandStack(this);
         return emfCmdStck;
      } else if (adapter == IEDITMODELSCRIBBLER_CLASS) {
         return this;
      }

      return Platform.getAdapterManager().getAdapter(this, adapter);
   }

   public IUndoContext getUndoContext() {
      return this;
   }

   private void disconnect() {
      internalState.vaporize();
   }

}
