/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.internal.infrastructure.emf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.NotificationImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.resource.impl.URIConverterImpl;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.jem.util.emf.workbench.ProjectResourceSet;
import org.eclipse.jem.util.emf.workbench.ResourceHandler;
import org.eclipse.jem.util.emf.workbench.ResourceSetWorkbenchSynchronizer;
import org.eclipse.jem.util.emf.workbench.nature.EMFNature;
import org.eclipse.jem.util.plugin.JEMUtilPlugin;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.WorkbenchResourceHelper;
import org.eclipse.wst.common.internal.emf.resource.ReferencedResource;

/**
 * Implements a more rigorous ProjectResourceSet than the version available in
 * JEM.
 * 
 */
public class ProjectResourceSetEditImpl extends ResourceSetImpl implements
      ProjectResourceSet {

   private final IProject                   project;

   private final List                       resourceHandlers = new ArrayList();

   private ResourceSetWorkbenchSynchronizer synchronizer;

   private boolean                          released         = false;

   /**
    * 
    */
   protected ProjectResourceSetEditImpl() {
      init();
      project = null;
   }

   /**
    * Create a Project Resource set to manage resources for the given project.
    * 
    * @param aProject
    *           An accessible ({@link org.eclipse.core.resources.IResource#isAccessible()})
    *           project.
    */
   public ProjectResourceSetEditImpl(IProject aProject) {
      init();
      Assert.isNotNull(aProject);
      Assert.isTrue(aProject.isAccessible());
      project = aProject;

      initializeSharedCacheListener();
   }

   private void init() {
      setURIResourceMap(new HashMap(10)); // Tell it to cache uri->resource
      // access.
      getLoadOptions().put(XMLResource.OPTION_USE_PARSER_POOL,
            EMFNature.SHARED_PARSER_POOL);
   }

   public Resource createResource(URI uri) {
      Assert.isTrue(!released);
      // Check the map first when creating the resource and do not
      // normalize if a value is found.
      URIConverterImpl.URIMap uriMap = (URIConverterImpl.URIMap) getURIConverter()
            .getURIMap();
      URI mappedURI = uriMap.getURI(uri);

      boolean isMapped = !(mappedURI != null && mappedURI.equals(uri));

      URI converted = isMapped ? getURIConverter().normalize(uri) : uri;

      Resource result = createResourceFromHandlers(converted);
      if (result == null)
         result = super.createResource(converted);

      if (result != null
            && WorkbenchResourceHelper.isReferencedResource(result))
         WorkbenchResourceHelper
               .cacheSynchronizationStamp((ReferencedResource) result);

      return result;
   }

   /**
    * @return Returns a project that contains the resources managed by this
    *         Resource Set.
    */
   public IProject getProject() {
      return project;
   }

   public EObject getEObject(URI uri, boolean loadOnDemand) {
      Assert.isTrue(!released);

      Resource resource = getResource(uri.trimFragment(), loadOnDemand);
      EObject result = null;
      if (resource != null && resource.isLoaded())
         result = resource.getEObject(uri.fragment());
      if (result == null)
         result = getEObjectFromHandlers(uri, loadOnDemand);
      return result;
   }

   /*
    * Javadoc copied from interface.
    */
   public Resource getResource(URI uri, boolean loadOnDemand) {
      Assert.isTrue(!released);
      return super.getResource(uri, loadOnDemand);
   }

   public boolean add(ResourceHandler resourceHandler) {
      Assert.isTrue(!released);
      return resourceHandlers.add(resourceHandler);
   }

   public void addFirst(ResourceHandler resourceHandler) {
      Assert.isTrue(!released);
      resourceHandlers.add(0, resourceHandler);
   }

   public boolean remove(ResourceHandler resourceHandler) {
      Assert.isTrue(!released);
      return resourceHandlers.remove(resourceHandler);
   }

   /**
    * @return The synchronizer which keeps resources managed by this resource
    *         set up to date with changes from the workspace.
    */
   public ResourceSetWorkbenchSynchronizer getSynchronizer() {
      Assert.isTrue(!released);
      return synchronizer;
   }

   /**
    * 
    * The synchronizer may be set only once. Attempting to override a
    * synchronizer will result in assertion exceptions.
    * 
    * @param aSynchronizer
    *           The synchronizer which keeps resources managed by this resource
    *           set up to date with changes from the workspace.
    */
   public void setSynchronizer(ResourceSetWorkbenchSynchronizer aSynchronizer) {
      Assert.isTrue(!released);
      Assert.isTrue(synchronizer == null);
      synchronizer = aSynchronizer;
   }

   public void setResourceFactoryRegistry(Resource.Factory.Registry factoryReg) {
      Assert.isTrue(!released);
      if (resourceFactoryRegistry != null && factoryReg != null) {
         preserveEntries(factoryReg.getExtensionToFactoryMap(),
               resourceFactoryRegistry.getExtensionToFactoryMap());
         preserveEntries(factoryReg.getProtocolToFactoryMap(),
               resourceFactoryRegistry.getProtocolToFactoryMap());
      }
      super.setResourceFactoryRegistry(factoryReg);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.jem.util.emf.workbench.ProjectResourceSet#resetNormalizedURICache()
    */
   public void resetNormalizedURICache() {
      Assert.isTrue(!released);
      if (getURIResourceMap() != null)
         getURIResourceMap().clear();
   }

   public void release() {
      synchronized (this) {
         if (released)
            return;
         released = true;
         // Send out notification of release.
         if (eNotificationRequired()) {
            eNotify(new NotificationImpl(SPECIAL_NOTIFICATION_TYPE, null, null,
                  Notification.NO_INDEX, false) {
               /*
                * (non-Javadoc)
                * 
                * @see org.eclipse.emf.common.notify.impl.NotificationImpl#getFeatureID(java.lang.Class)
                */
               public int getFeatureID(Class expectedClass) {
                  return PROJECTRESOURCESET_ABOUT_TO_RELEASE_ID;
               }

               /*
                * (non-Javadoc)
                * 
                * @see org.eclipse.emf.common.notify.impl.NotificationImpl#getNotifier()
                */
               public Object getNotifier() {
                  return ProjectResourceSetEditImpl.this;
               }
            });
         }
         if (synchronizer != null)
            synchronizer.dispose();
         synchronizer = null;

         removeAndUnloadAllResources();

         if (resourceHandlers != null)
            resourceHandlers.clear();
         if (eAdapters != null)
            eAdapters.clear();

         JEMUtilPlugin.getSharedCache().stopListening(this);
      }
   }

   protected void removeAndUnloadAllResources() {

      if (resources == null || resources.isEmpty())
         return;
      Resource res;
      int size = resources.size();
      for (int i = 0; i < size; i++) {
         res = (Resource) resources.get(i);
         try {
            res.unload();
         } catch (RuntimeException ex) {
            EMFInfrastructurePlugin.logError(0, ex.getMessage(), ex);
         }
      }
      resources.clear();
   }

   protected void initializeSharedCacheListener() {
      JEMUtilPlugin.getSharedCache().beginListening(this);
   }

   protected Resource delegatedGetResource(URI uri, boolean loadOnDemand) {
      Assert.isTrue(!released);
      Resource res = super.delegatedGetResource(uri, loadOnDemand);
      if (res == null)
         res = getResourceFromHandlers(uri);
      return res;
   }

   protected void demandLoad(Resource resource) throws IOException {
      Assert.isTrue(!released);
      super.demandLoad(resource);
   }

   /**
    * See if any resource handlers from the WorkbenchContext decide to create
    * the Resource in another manner.
    */
   protected Resource createResourceFromHandlers(URI uri) {
      Assert.isTrue(!released);

      Resource resource = null;
      ResourceHandler handler = null;
      for (int i = 0; i < resourceHandlers.size(); i++) {
         handler = (ResourceHandler) resourceHandlers.get(i);
         resource = handler.createResource(this, uri);
         if (resource != null)
            return resource;
      }
      return null;
   }

   /**
    * See if any resource handlers from the WorkbenchContext can return a
    * Resource from a <code>uri</code>.
    */
   protected Resource getResourceFromHandlers(URI uri) {
      Assert.isTrue(!released);
      for (int i = 0; i < resourceHandlers.size(); i++) {
         Resource resource = ((ResourceHandler) resourceHandlers.get(i))
               .getResource(this, uri);
         if (resource != null)
            return resource;
      }
      return null;
   }

   /**
    * See if any resource handlers from the WorkbenchContext can return a
    * EObject from a <code>uri</code> after failing to find it using the
    * normal mechanisms.
    */
   protected EObject getEObjectFromHandlers(URI uri, boolean loadOnDemand) {
      EObject obj = null;
      ResourceHandler handler = null;
      for (int i = 0; i < resourceHandlers.size(); i++) {
         handler = (ResourceHandler) resourceHandlers.get(i);
         obj = handler.getEObjectFailed(this, uri, loadOnDemand);
         if (obj != null)
            return obj;
      }
      return null;
   }

   /*
    * Preserve the entries from map2 in map1 if no collision.
    */
   protected void preserveEntries(Map map1, Map map2) {
      if (map2.isEmpty())
         return;
      Iterator it = map2.entrySet().iterator();
      Map.Entry entry;
      while (it.hasNext()) {
         entry = (Map.Entry) it.next();
         if (!map1.containsKey(entry.getKey()))
            map1.put(entry.getKey(), entry.getValue());
      }
   }

}
