/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
/*
 * Created on May 30, 2006
 *
 * Copyright 2005 by IONA Technologies PLC, Inc.
 *
 * All rights reserved.
 *
 */
package org.eclipse.stp.sc.jaxws.builders;

import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.jaxws.ScJaxWsPlugin;
import org.eclipse.stp.sc.jaxws.generators.types.JaxBindTypeGenerator;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IJavaToWsdlGenerator;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;



/**
 * This builder is used to build java code with JWS annotation. It is done by call the javaToWsdl generator.
 */
public abstract class ScJavaBuilder extends IncrementalProjectBuilder {
        
    private static final String WSDL_OUTPUT_DIR = "wsdl";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(ScJavaBuilder.class);
    
    public ScJavaBuilder() {
        super();
    }

    @Override
    protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
        throws CoreException {
        if (kind == IncrementalProjectBuilder.FULL_BUILD) {
            fullBuild(monitor);
        } else {
            IResourceDelta delta = getDelta(getProject());

            if (delta == null) {
                fullBuild(monitor);
            } else {
                incrementalBuild(delta, monitor);
            }
        }

        return null;
    }

    /**
     * build every java with ws-annotation files within the project
     * @param monitor, the progress monitor
     * @throws CoreException get status within CoreException will have BUILD_FAILED 
     * code and error message string
     */
    protected void fullBuild(final IProgressMonitor monitor)
        throws CoreException {
        LOG.debug("fullBuild for project:" + getProject().getName());
        getProject().accept(new ScBuildVisitor());
        getProject().refreshLocal(IProject.DEPTH_INFINITE, monitor);
    }

    protected void incrementalBuild(IResourceDelta delta,
        IProgressMonitor monitor) throws CoreException {
        // the visitor does the work.
        LOG.debug("incrementalBuild for project:" + getProject().getName());
        delta.accept(new ScBuildDeltaVisitor());
        getProject().refreshLocal(IProject.DEPTH_INFINITE, monitor);
    }

    /**
     * clean the generated wsdl. This will delete /project/wsdl dir and all contains under that dir
     * @param monitor, the progress monitor
     * @throws CoreException
     */
    protected void clean(IProgressMonitor monitor) throws CoreException {
        LOG.debug("clean for the project:" + getProject().getName());

        IProject proj = getProject();
        IFolder wsdlFolder = proj.getFolder(WSDL_OUTPUT_DIR);
        wsdlFolder.delete(true, monitor);
        getProject().refreshLocal(IProject.DEPTH_INFINITE, null);
    }

    /**
     * call the javatowsdl to build one JWS style java source file
     * @param javaFile, the java file with JWS annotation
     */
    protected void buildOneJavaFile(IFile javaFile) throws CoreException {
        try {
            ICompilationUnit compUnit = JDTUtils.getJavaUnitFromFile(javaFile);

            if (JavaDocumentUtils.needToGenerateWrapper(compUnit)) {
                LOG.info("generate wrapper type classes for SEI:"
                         + compUnit.findPrimaryType().getFullyQualifiedName());

                JaxBindTypeGenerator typeGen = new JaxBindTypeGenerator();
                typeGen.createWrapperCls(compUnit);
            }

            generate(javaFile.getFullPath(), getProject(), null);

        } catch (Exception e) {
            LOG.debug("generation failure", e);

            Status status = new Status(IStatus.ERROR,
                                       ResourcesPlugin.PI_RESOURCES,
                                       IResourceStatus.BUILD_FAILED,
                                       e.getMessage(),
                                       e);
            throw new CoreException(status);
        }
    }

    /**
     * setup the javatowsdl generate in the initialize
     */
    protected void startupOnInitialize() {
    }

    /**
     * visit the resource file within project.
     * call javatowsdl if it is java file and has WebService annotation
     * @param res
     * @throws CoreException
     */
    private void visitResourceFile(IResource res) throws CoreException {
        if (!(res instanceof IFile)) {
            return;
        }

        IFile file = (IFile)res;

        if (!file.getFileExtension().equals("java")) {
            return;
        }

        if (JavaDocumentUtils.hasWebServiceAnnotation(file)) {
            buildOneJavaFile(file);
        }
    }

    /**
     *
     * @param res
     * @throws CoreException
     */
    private void removeResourceFile(IResource res) throws CoreException {
        if (!(res instanceof IFile)) {
            return;
        }

        IFile file = (IFile)res;

        if (!file.getFileExtension().equals("java")) {
            return;
        }

        if (JavaDocumentUtils.hasWebServiceAnnotation(file)) {
            LOG.debug("WebService java file has been removed:"
                + file.getFullPath());

            String clsName = JDTUtils.getJavaClassNameFromFile(file);

            if (clsName.lastIndexOf(".") > 0) {
                clsName = clsName.substring(clsName.lastIndexOf("."));
            }

            String wsdlName = clsName + ".wsdl";
            LOG.debug("  need to remove:" + wsdlName);

            IContainer folder = JaxWsWorkspaceManager.getWSDLFolder(getProject());
            IResource wsdlResource = folder.findMember(wsdlName);

            if (wsdlResource.exists()) {
                LOG.debug("delete wsdl resource:"
                    + wsdlResource.getFullPath());
                wsdlResource.delete(IResource.FORCE, null);
            }
        }
    }

    class ScBuildVisitor implements IResourceVisitor {
        public boolean visit(IResource res) throws CoreException {
            //build the specified resource.
            visitResourceFile(res);

            //return true to continue visiting children.
            return true;
        }
    }

    class ScBuildDeltaVisitor implements IResourceDeltaVisitor {
        public boolean visit(IResourceDelta delta) throws CoreException {
            IResource res = delta.getResource();

            switch (delta.getKind()) {
            case IResourceDelta.ADDED:
            case IResourceDelta.CHANGED:
                visitResourceFile(res);

                break;

            case IResourceDelta.REMOVED:
                LOG.debug("DeltaVistor, removed resource:"
                    + res.getFullPath());
                try {
                    removeResourceFile(res);
                } catch (Exception e) {
                    LOG.error("builder error", e);
                }

                break;
            default:
                break;
            }

            return true;
        }
    }
    
    public void generate(IPath path, IProject project, Object data) throws CoreException {
        IJavaToWsdlGenerator generator =
            ScJaxWsPlugin.getRuntimeProviderManager().getJavaToWsdlGenerator(project);
        if (data != null) {
            generator.setInitializationData(null, null, data);
        }
        generator.run(path, project);
    }

}
