/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.generators.types;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.jws.Oneway;
import javax.jws.WebParam;
import javax.jws.WebResult;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.ws.RequestWrapper;
import javax.xml.ws.ResponseWrapper;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IImportDeclaration;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageDeclaration;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.Annotation;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.NormalAnnotation;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.SingleMemberAnnotation;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.rewrite.ASTRewrite;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.Document;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.jaxws.annotations.WebParamAnnInfo;
import org.eclipse.stp.sc.jaxws.annotations.WebResultAnnInfo;
import org.eclipse.stp.sc.jaxws.utils.JaxBindAnnotationUtils;
import org.eclipse.stp.sc.jaxws.utils.ScJDTUtils;
import org.eclipse.text.edits.MalformedTreeException;
import org.eclipse.text.edits.TextEdit;



public class JaxBindTypeGenerator {
    public static final String ANNOTATION_WRAPPER_LOCALNAME = "localName";
    public static final String ANNOTATION_WRAPPER_CLASSNAME = "className";
    public static final String ANNOTATION_WRAPPER_TARGETNAMESPACE = "targetNamespace";
    public static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final String DEFAULT_PACKAGE_NAME = "jaxws";
    private static final String DEFAULT_ARG_PREFIX = "arg";
    private static final int MODE_UNKNOWN = 0;
    private static final int MODE_REQUEST = 1;
    private static final int MODE_RESPONSE = 2;
    
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(JaxBindTypeGenerator.class);
    
    String className = "";
    String targetNamespace = "";
    String localName = "";
    ICompilationUnit sei;
    IPackageFragment pkgFragment = null;
    private ArrayList<String> xmlTypeList = new ArrayList<String>();
    private ArrayList<String> importList = new ArrayList<String>();

    public void createWrapperCls(ICompilationUnit seiUnit) {
        IMethod[] methods;
        sei = seiUnit;

        try {
            methods = seiUnit.findPrimaryType().getMethods();

            for (IMethod method : methods) {
                List<Annotation> annotations = JDTUtils.getAnnotations(method);
                createWrappClsByAnnotations(method, annotations);

            }
        } catch (Exception e) {
            LOG.error("error during generate java class", e);
        }
    }

    @SuppressWarnings("unchecked")
    protected void createWrappClsByAnnotations(IMethod method, List<Annotation> annotations) throws Exception {

        boolean genRequestWrapper = false;
        boolean genResponseWrapper = false;
        boolean hasOneWayAnnotation = false;
        for (Annotation an : annotations) {
        	
            int mode = MODE_UNKNOWN;
            String name = an.getTypeName().getFullyQualifiedName();

            if (name.equals(RequestWrapper.class.getSimpleName())) {
                mode = MODE_REQUEST;
                genRequestWrapper = true;
                setupDefaultValue(method, MODE_REQUEST);
            
            } else if (name.equals(ResponseWrapper.class.getSimpleName())) {
                mode = MODE_RESPONSE;
                genResponseWrapper = true;
                setupDefaultValue(method, MODE_RESPONSE);
            
            } else {
            	if (name.equals(Oneway.class.getSimpleName())) {
            		hasOneWayAnnotation = true;
            	}
                continue;
            }

            NormalAnnotation normalAnn = (NormalAnnotation)an;
            List list = normalAnn.values();
            Iterator itor = list.iterator();

            while (itor.hasNext()) {
                MemberValuePair valuePair = (MemberValuePair)itor.next();
                String attrKey = valuePair.getName().getFullyQualifiedName();
                String value = ((StringLiteral)valuePair.getValue()).getLiteralValue();

                if (attrKey.equals(ANNOTATION_WRAPPER_LOCALNAME)) {
                    localName = value;
                } else if (attrKey.equals(ANNOTATION_WRAPPER_CLASSNAME)) {
                    className = value;
                } else if (attrKey.equals(ANNOTATION_WRAPPER_TARGETNAMESPACE)) {
                    targetNamespace = value;
                }
            }

            createWrapperCls(method, mode, null);
        }

        if (!genRequestWrapper) {
            setupDefaultValue(method, MODE_REQUEST);
            createWrapperCls(method, MODE_REQUEST, null);
        } 
        
        if (!genResponseWrapper && !hasOneWayAnnotation) {
            setupDefaultValue(method, MODE_RESPONSE);
            createWrapperCls(method, MODE_RESPONSE, null);
        }

    }
    
    protected void setupDefaultValue(IMethod method, int mode) throws JavaModelException {
        
        localName = (mode == MODE_REQUEST) ? method.getElementName() : method.getElementName() + "Response";

        String pkgName = "";
        
        IPackageDeclaration[] pkgDecs = sei.getPackageDeclarations();
        if(pkgDecs.length == 0)
        {
        	pkgName = "";
            className = DEFAULT_PACKAGE_NAME + "."
            			+ localName.substring(0, 1).toUpperCase() + localName.substring(1);
        }else{
        	pkgName = sei.getPackageDeclarations()[0].getElementName();
        	className = pkgName + "." 
        				+ DEFAULT_PACKAGE_NAME + "." 
        				+ localName.substring(0, 1).toUpperCase() + localName.substring(1);
        }
        
        LOG.debug("default wrapper cls name:" + className);
        
        targetNamespace = JDTUtils.getNamespace(pkgName);
        LOG.debug("default wrapper ns:" + targetNamespace);
    }
    
    
    protected void createWrapperCls(IMethod webMethod, int mode, IProgressMonitor monitor) throws Exception {

        xmlTypeList.clear();
        importList.clear();

        ICompilationUnit wrapperCls = createEmptyWrapperUnit(webMethod, monitor);
        
        addClsAnnotation(wrapperCls, webMethod);
        processFields(wrapperCls.getTypes()[0], webMethod, mode);
        addXmlTypeAnnotation(wrapperCls);
        addImport(wrapperCls);
        
        wrapperCls.save(monitor, true);
        LOG.debug("generated wrapper class:" + className);
    }
    

    private ICompilationUnit createEmptyWrapperUnit(IMethod webMethod,
                                                    IProgressMonitor monitor) throws CoreException {
        String unitName = className.substring(className.lastIndexOf(".") + 1);
        String packageName = className.substring(0, className.lastIndexOf("."));

        IPackageFragment pkg = getPackageFragment(webMethod.getResource(), packageName);

        //delete the file first it it is already exists
        String unitFileName = unitName + ".java";
        ICompilationUnit oldCls = pkg.getCompilationUnit(unitFileName);

        if (oldCls.getResource().exists()) {
            oldCls.getResource().delete(true, null);
            LOG.debug("delete previous generated type file:" + unitFileName);
        }
        

        ICompilationUnit wrapperCls = pkg.createCompilationUnit(unitFileName, "", true, monitor);
        wrapperCls.createPackageDeclaration(packageName, null);
        wrapperCls.createImport(XmlAccessType.class.getName(), null, monitor);
        wrapperCls.createImport(XmlAccessorType.class.getName(), null, monitor);
        wrapperCls.createImport(XmlElement.class.getName(), null, monitor);
        wrapperCls.createImport(XmlRootElement.class.getName(), null, monitor);
        wrapperCls.createImport(XmlType.class.getName(), null, monitor);

        StringBuffer contents = new StringBuffer();
        contents.append("public class " + unitName + " {" + LINE_SEPARATOR);

        contents.append("}" + LINE_SEPARATOR);
        wrapperCls.createType(contents.toString(), null, true, monitor);

        return wrapperCls;
    }

    private IPackageFragment getPackageFragment(IResource seiFile,
                                                String packageName) throws JavaModelException {
        
        IJavaProject javaProject = JDTUtils.findJavaProject(seiFile.getProject().getName());
        IPackageFragmentRoot[] roots = javaProject.getPackageFragmentRoots();
        IPackageFragmentRoot srcRoot = null;

        for (IPackageFragmentRoot root : roots) {
            if (root.getKind() == IPackageFragmentRoot.K_SOURCE) {
                srcRoot = root;
                break;
            }
        }

        pkgFragment = srcRoot.getPackageFragment(packageName);

        if (!pkgFragment.exists()) {
            pkgFragment = srcRoot.createPackageFragment(packageName, true, null);
        }

        if (!pkgFragment.isOpen()) {
            pkgFragment.open(null);
        }

        return pkgFragment;
    }

    private CompilationUnit createAstTree(ICompilationUnit unit) throws JavaModelException {

        ASTParser parser = ASTParser.newParser(AST.JLS3);
        parser.setSource(unit);

        CompilationUnit astRoot = (CompilationUnit)parser.createAST(null);

        return astRoot;
    }

    private void addClsAnnotation(ICompilationUnit unit, IMethod method) throws JavaModelException, 
                                                                                MalformedTreeException,
                                                                                BadLocationException {
        CompilationUnit astRoot = createAstTree(unit);
        AST ast = astRoot.getAST();
        QualifiedName qName = ast.newQualifiedName(ast.newSimpleName("XmlAccessType"),
                                                   ast.newSimpleName(XmlAccessType.FIELD.name()));
        SingleMemberAnnotation singleAnn = JaxBindAnnotationUtils.newXmlAccessorTypeAnnotation(astRoot, qName);
        addAnnotation(unit, astRoot, singleAnn, unit.findPrimaryType());

        //XmlRootElement
        Annotation xmlRootElementAnn = JaxBindAnnotationUtils.newXmlRootElemntAnnotation(astRoot, localName);
        addAnnotation(unit, astRoot, xmlRootElementAnn, unit.findPrimaryType());
    }

    /**
     * we need to add xml type annotation after processes all fields
     * @param unit
     * @throws Exception
     */
    private void addXmlTypeAnnotation(ICompilationUnit unit) throws Exception {

        CompilationUnit astRoot = createAstTree(unit);
        Annotation xmlTypeAnn = 
            JaxBindAnnotationUtils.newXmlTypeAnnotation(astRoot, "", xmlTypeList.toArray(new String[0]));

        addAnnotation(unit, astRoot, xmlTypeAnn, unit.findPrimaryType());
    }

    private void processFields(IType wrapperClsType, IMethod method, int mode) throws Exception {

        List<Annotation> annotations = JDTUtils.getAnnotations(method);
        boolean hasWebResultAnn = false;
        if (mode == MODE_RESPONSE) {
            // process @WebResult
            for (Annotation an : annotations) {
                if (an.getTypeName().getFullyQualifiedName().equals(WebResult.class.getSimpleName())) {

                    WebResultAnnInfo resultAnn = new WebResultAnnInfo(an);
                    if (resultAnn.getTargetNamespace() == null) {
                        resultAnn.setTargetNamespace(targetNamespace);
                    }

                    createOneField(wrapperClsType,
                                   method.getReturnType(),
                                   resultAnn.getName(),
                                   resultAnn.getTargetNamespace());
                    hasWebResultAnn = true;
                    break;
                }
            }
        }

        String[] paraTypes = method.getParameterTypes();
        String[] paraNames = method.getParameterNames();

        for (int i = 0; i < paraNames.length; i++) {
            
            WebParamAnnInfo webParamAnn = JaxBindTypeGenerator.getWebParamAnnotation(method, paraNames[i]);

            if (webParamAnn == null) {
                LOG.debug("missing @WebParam annotation for method:" + method.getElementName());
                webParamAnn = createDefaultWebParam(paraNames[i], i);
            }

            if ((mode == MODE_REQUEST) && (webParamAnn.getMode() == WebParam.Mode.OUT)) {
                continue; //pass output parameter
            }

            if ((mode == MODE_RESPONSE) && (webParamAnn.getMode() == WebParam.Mode.IN)) {
                continue; //pass input parameter
            }

            String paramName = webParamAnn.getName();

            if ((paramName == null) || paramName.equals("")) {
                paramName = paraNames[i];
                //default value
            }

            if ((webParamAnn.getTargetNamespace() == null) || webParamAnn.getTargetNamespace().equals("")) {

                webParamAnn.setTargetNamespace(targetNamespace); //default value
            }

            createOneField(wrapperClsType, paraTypes[i], paramName, webParamAnn.getTargetNamespace());
        }
        
        if (mode == MODE_RESPONSE && !hasWebResultAnn) {

            String typeString = Signature.getSignatureSimpleName(method.getReturnType());
            if (!typeString.toLowerCase().startsWith("void")) {
                WebResultAnnInfo resultAnn = createDefaultWebResult();
                LOG.debug("create default WebResult:" + resultAnn.getName());
                createOneField(wrapperClsType,
                               method.getReturnType(),
                               resultAnn.getName(),
                               resultAnn.getTargetNamespace());
            }
        }
    }
    
    private WebResultAnnInfo createDefaultWebResult() {

        WebResultAnnInfo webResult = new WebResultAnnInfo();
        webResult.setName("return");
        webResult.setTargetNamespace(targetNamespace);

        return webResult;
    }
    
    
    
    private WebParamAnnInfo createDefaultWebParam(String paramName, int index) {

        LOG.debug("create default WebParam for parameter:" + paramName);
        WebParamAnnInfo annInfo = new WebParamAnnInfo();
        //according to jaxb spec, the default value should be arg0, arg1, arg2
        annInfo.setName(DEFAULT_ARG_PREFIX + index);
        annInfo.setTargetNamespace(targetNamespace);
        annInfo.setMode(WebParam.Mode.IN);

        return annInfo;
    }
    
    private void createOneField(IType wrapperClsType, 
                                String typeSignature, 
                                String paramName, 
                                String tns) throws Exception {
    	if (paramName.equals("return")) {
    		xmlTypeList.add("_" + paramName);
    	} else {
            xmlTypeList.add(paramName);
    	}

        String fieldDesc = Modifier.ModifierKeyword.PROTECTED_KEYWORD.toString();
        String typeString = Signature.getSignatureSimpleName(typeSignature);

        
        fieldDesc += " " + typeString + " ";
        if (paramName.equals("return")) {
        	fieldDesc += "_" + paramName + ";";
        } else {
            fieldDesc += paramName + ";";
        }

        LOG.debug("fielDesc:" + fieldDesc);
        IField field = wrapperClsType.createField(fieldDesc, null, true, null);
        CompilationUnit astRoot = createAstTree(wrapperClsType.getCompilationUnit());
        FieldDeclaration fieldDeclaration = JDTUtils.getFieldDeclaration(astRoot, field);
        LOG.debug("fieldDeclaration:" + fieldDeclaration);

        Type type = fieldDeclaration.getType();
        if ((fieldDeclaration != null) && (type != null) && !type.isPrimitiveType()) {

        	String importTypeStr = typeString;
        	if (importTypeStr.indexOf("<") > 0) {
        		importTypeStr = importTypeStr.substring(0, 
        				importTypeStr.indexOf("<"));
            }
        	if (importTypeStr.indexOf("[]") > 0) {
        		importTypeStr = importTypeStr.substring(0, 
        				importTypeStr.indexOf("[]"));
            }
            String pkg = Signature.getSignatureQualifier(typeSignature);

            if (pkg.indexOf(".") < 0) {
                //we need to find the right package here.
                pkg = getPackageNameForClassType(sei.getTypes()[0], importTypeStr);
            }

            if ((pkg != null) && !pkg.startsWith("java.lang")) {
                // add import if needed
                String importStat = pkg + "." + importTypeStr;
                
                LOG.debug("import stat to add:" + importStat);
                importList.add(importStat);
            }
        }

        NormalAnnotation annotation = JaxBindAnnotationUtils.newXmlElementAnnotation(astRoot, 
                                                                                     paramName,
                                                                                     null,
                                                                                     tns,
                                                                                     null,
                                                                                     type); //default value? nillable?
        addAnnotation(wrapperClsType.getCompilationUnit(), astRoot, annotation, field);
        createGetterSetter(wrapperClsType, typeString, paramName);
    }

    private void addImport(ICompilationUnit unit) throws JavaModelException, 
                                                         MalformedTreeException, 
                                                         BadLocationException {
        CompilationUnit astRoot = createAstTree(unit);

        //AST ast = astRoot.getAST();
        for (String importStat : importList) {
            addImport(unit, astRoot, importStat);
        }
    }

    @SuppressWarnings("unchecked")
    private boolean isClsDefined(String pkg, String type) {

        String clsName = pkg.equals("") ? type : (pkg + "." + type);

        try {
            Class cls = Class.forName(clsName);

            if (cls != null) {
                return true;
            }

            return false;
        } catch (Throwable th) {
            return false;
        }
    }

    private String getPackageNameForClassType(IType wrapperClsType, String type) throws JavaModelException {

        if (isClsDefined("java.lang", type)) {
            return "java.lang";
        }

        IImportDeclaration[] imports = wrapperClsType.getCompilationUnit().getImports();

        for (IImportDeclaration importDesc : imports) {
            String importStat = importDesc.getElementName();

            if (importStat.endsWith(".*")) {
                String pkg = importStat.substring(0, importStat.lastIndexOf("."));

                if (isClsDefined(pkg, type)) {
                    return pkg;
                } else {
                    continue;
                }
            } else if (importStat.endsWith(type)) {
                return importStat.substring(0, importStat.lastIndexOf("."));
            }
        }

        //check current package
        if (pkgFragment.getCompilationUnit(type + ".java") != null) {
            return wrapperClsType.getPackageFragment().getElementName();
        }

        return null;
    }

    private void createGetterSetter(IType wrapperClsType,
                                    String type,
                                    String param) throws JavaModelException {
        
        String upperName = Character.toString(param.charAt(0)).toUpperCase();
        upperName += param.substring(1);

        StringBuffer sb = new StringBuffer();
        sb.append("public ");
        sb.append(type);
        sb.append(" get");
        sb.append(upperName);
        sb.append("(){");
        sb.append(LINE_SEPARATOR);
        sb.append("    return ");
        if (param.equals("return")) {
            sb.append("_" + param);
        } else {
        	sb.append(param);  
        }
        
        sb.append(";");
        sb.append(LINE_SEPARATOR);
        sb.append("}");
        sb.append(LINE_SEPARATOR);
        wrapperClsType.createMethod(sb.toString(), null, true, null);
        
        sb = new StringBuffer();
        sb.append("public void set")
          .append(upperName)
          .append("(")
          .append(type)
          .append(" value){")
          .append(LINE_SEPARATOR)
          .append("    this.");
        if (param.equals("return")) {  
          sb.append("_" + param);
        } else {
          sb.append(param);
        }
          sb.append(" = value;")
          .append(LINE_SEPARATOR)
          .append("}")
          .append(LINE_SEPARATOR);
        wrapperClsType.createMethod(sb.toString(), null, true, null);
    }

    private void addImport(ICompilationUnit unit,
                           CompilationUnit astRoot,
                           String importStat) throws JavaModelException,
                                                     MalformedTreeException,
                                                     BadLocationException {
        //creation of a Document
        String source = unit.getBuffer().getContents();
        Document document = new Document(source);

        //creation of ASTRewrite
        ASTRewrite rewrite = ASTRewrite.create(astRoot.getAST());

        JDTUtils.addImport(astRoot, importStat, rewrite);

        // computation of the text edits
        TextEdit edits = rewrite.rewriteAST(document, unit.getJavaProject().getOptions(true));

        // computation of the new source code
        edits.apply(document);

        String newSource = document.get();

        // update of the compilation unit
        unit.getBuffer().setContents(newSource);
    }

    //todo: improve performance??? do not create rewrite everytime?
    private void addAnnotation(ICompilationUnit unit,
                               CompilationUnit astRoot,
                               Annotation annotation,
                               IMember member) throws JavaModelException, 
                                                      MalformedTreeException, 
                                                      BadLocationException {
        //creation of a Document
        String source = unit.getBuffer().getContents();
        Document document = new Document(source);

        //creation of ASTRewrite
        ASTRewrite rewrite = ASTRewrite.create(astRoot.getAST());

        if (member instanceof IField) {
            ScJDTUtils.addAnnotationOnField(astRoot, annotation, (IField)member, rewrite);
        } else if (member instanceof IMethod) {
            ScJDTUtils.addAnnotationOnMethod(astRoot, annotation, (IMethod)member, rewrite, true);
        } else if (member instanceof IType) {
            ScJDTUtils.addAnnotationOnType(astRoot, annotation, (IType)member, rewrite, true);
        }

        // computation of the text edits
        TextEdit edits = rewrite.rewriteAST(document, unit.getJavaProject().getOptions(true));

        // computation of the new source code
        edits.apply(document);

        String newSource = document.get();

        // update of the compilation unit
        unit.getBuffer().setContents(newSource);
    }

    /**
     * retriecves "webParam" annotation data for a specific method parameter
     * @param method
     * @param parameterName 
     * @return the annotation data
     */
    public static WebParamAnnInfo getWebParamAnnotation(IMethod method,
                                                        String parameterName) {
    
        List<Annotation> annList = JDTUtils.getParameterAnnotations(JDTUtils.getDomRootCompilationUnit(method),
                                                           method,
                                                           parameterName);
    
        if ((annList == null) || (annList.size() == 0)) {
            return null;
        }
    
        for (Annotation an : annList) {
            String name = an.getTypeName().getFullyQualifiedName();
    
            if (name.equals(WebParam.class.getSimpleName())) {
                return new WebParamAnnInfo(an);
            }
        }
    
        return null;
    }
}
