/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.runtimeprovider;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.preferences.PreferencesAccessor;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.stp.sc.jaxws.workspace.ScNature;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ListDialog;

public class RuntimeProviderManager implements IRuntimeProviderManager {
    
    private static final String EXT_POINT_SC_RUNTIME_PROVIDER = "org.eclipse.stp.sc.jaxws.runtimeProvider";
    private static final String EXT_ELT_SC_KIT_PROCESSOR = "KitProcessor";
    private static final String EXT_ATT_SC_PROCESSOR_CLASS = "kit_processor_class";
    private static final String EXT_ELT_SC_W2J_GEN = "wsdltoJavaGenerator";
    private static final String EXT_ATT_SC_W2J_GEN_CLASS = "class";
    private static final String EXT_ATT_SC_W2J_PARAM_PAGE = "parameter_page";
    private static final String EXT_ELT_SC_J2W_GEN = "javaToWsdlGenerator";
    private static final String EXT_ATT_SC_J2W_GEN_CLASS = "class";
    private static final String EXT_ELT_SC_W2J_WIZ = "wsdlToJavaWizard";
    private static final String EXT_ATT_SC_W2J_WIZ_CLASS = "class";
    private static final String EXT_ELT_SC_DIR_TEMP = "projectDirTemplate";
    private static final String EXT_ATT_SC_DIR_TEMP_CLASS = "class";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(RuntimeProviderManager.class);

    private static RuntimeProviderManager instance;
    
    private static Map<IProject, IProjectDirTemplate> prjDirTempMap;
    
    private RuntimeProviderManager(){
    }
    
    public static RuntimeProviderManager getInstance() {
        if (instance == null) {
            instance = new RuntimeProviderManager();
        }
        if(prjDirTempMap == null){
        	prjDirTempMap = new HashMap<IProject, IProjectDirTemplate>();
        }
        return instance;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getWsdltoJavaGenerator(org.eclipse.core.resources.IProject)
     */
    public IWsdlToJavaGenerator getWsdltoJavaGenerator(IProject aProject){
        return getWsdltoJavaGenerator(getRuntimeProviderUID(aProject));
    }
    
    public IWsdlToJavaWizard getWsdlToJavaWizard(IProject aProject) {
    	return getWsdltoJavaWizard(getRuntimeProviderUID(aProject));
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getJavaToWsdlGenerator(org.eclipse.core.resources.IProject)
     */
    public IJavaToWsdlGenerator getJavaToWsdlGenerator(IProject aProject){
        return getJavaToWsdlGenerator(getRuntimeProviderUID(aProject));
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getRuntimeKitProcessor(org.eclipse.core.resources.IProject)
     */
    public IRuntimeKitProcessor getRuntimeKitProcessor(IProject aProject){
        String providerExtUID = getRuntimeProviderUID(aProject);
        return getRuntimeKitProcessor(providerExtUID);
    }

    public RuntimeKitProcessor getRuntimeKitProcessor(String providerExtUID) {
        return new RuntimeKitProcessor(getRuntimeKitTemplateProcessor(providerExtUID),
                                       getKitRootFolder(providerExtUID));
    }
    
    
    public IProjectDirTemplate getProjectDirTemplate(IProject aProject) {
    	if(prjDirTempMap.keySet().contains(aProject)){
    		return prjDirTempMap.get(aProject);
    	}    		
    	IProjectDirTemplate template = getProjectDirTemplate(getRuntimeProviderUID(aProject));
    	prjDirTempMap.put(aProject, template);
    	return template;
    }
    
    

    private String getRuntimeProviderUID(IProject aProject){
        try {
            if (ScNature.hasNature(aProject)) {
                ScNature theNature = new ScNature();
                theNature.setProject(aProject);
                String providerUID = theNature.getRuntimeProviderUID();
                if (providerUID == null) {
                    String pickedProviderId = pickAProvider();
                    theNature.setRuntimeProviderId(pickedProviderId);
                    return pickedProviderId;
                }
                return providerUID;
            } else {
            	//during project creation. will exec below code before setup ScNature.
            	String providerUID = pickAProvider();
            	LOG.debug("return provider id without ScNature:" + providerUID);
            	return providerUID;
            }
        } catch (CoreException e) {
            // project not accessible
        }
        return null;
    }
    
    public String pickAProvider() {
        IExtension[] allRegistredProviders = getAllRegistredProviders();
        int numberOfRegistredProviders = allRegistredProviders.length;
        return numberOfRegistredProviders < 1 
                ? null 
                : numberOfRegistredProviders == 1 
                    ? allRegistredProviders[0].getUniqueIdentifier()
                    : promptUserToPickAProvider(allRegistredProviders);
    }

    private String promptUserToPickAProvider(IExtension[] allRegistredProviders) {
        
        
        ListDialog dialog = new ListDialog(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell());
        dialog.setContentProvider(new ArrayContentProvider());
        
        dialog.setTitle(ScJaxWsResources.getString("RuntimeProvider.dialog.title"));
        dialog.setMessage(ScJaxWsResources.getString("RuntimeProvider.dialog.message"));
        
        dialog.setLabelProvider(new LabelProvider() {
                public String getText(Object element) {
                    if (element instanceof IExtension) {
                        return ((IExtension)element).getLabel();
                    }
                    return null;
                }
            });

        dialog.setBlockOnOpen(true);
        dialog.open();
        
        Object[] result = dialog.getResult();
        
        if (result.length > 0) {
            ((IExtension)result[0]).getUniqueIdentifier();
        }
        
        return null;

        /*
        cptViewer.getTree().getDisplay().asyncExec(new Runnable() {
            public void run() {
                cptViewer.setInput(IntrospectionManager.getAllComponentTypes());
                cptViewer.refresh();
            }
        });
         */
    }

    public IExtension getExtension(String ID){
        IExtensionRegistry reg = Platform.getExtensionRegistry();
        return reg.getExtension(EXT_POINT_SC_RUNTIME_PROVIDER, ID);
    }
    
    public IExtension[] getAllRegistredProviders() {
        IExtensionRegistry reg = Platform.getExtensionRegistry();
        return reg.getExtensionPoint(EXT_POINT_SC_RUNTIME_PROVIDER).getExtensions();
    }

    private IWsdlToJavaGenerator getWsdltoJavaGenerator(String ID){
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_GEN)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_GEN_CLASS);
                    if (kitProcessorObj instanceof IWsdlToJavaGenerator) {
                        return (IWsdlToJavaGenerator)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IWsdlToJavaGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    public IProjectDirTemplate getProjectDirTemplate(String providerExtUID) {
        IConfigurationElement[] extElts = getExtension(providerExtUID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_DIR_TEMP)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_DIR_TEMP_CLASS);
                    if (kitProcessorObj instanceof IProjectDirTemplate) {
                        return (IProjectDirTemplate)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(providerExtUID,
                    		EXT_ATT_SC_DIR_TEMP_CLASS,
                            IProjectDirTemplate.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(providerExtUID, EXT_ATT_SC_DIR_TEMP_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    private IWsdlToJavaWizard getWsdltoJavaWizard(String ID){
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_WIZ)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_WIZ_CLASS);
                    if (kitProcessorObj instanceof IWsdlToJavaWizard) {
                        return (IWsdlToJavaWizard)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IWsdlToJavaWizard.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }

    private IJavaToWsdlGenerator getJavaToWsdlGenerator(String ID){
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_J2W_GEN)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_J2W_GEN_CLASS);
                    if (kitProcessorObj instanceof IJavaToWsdlGenerator) {
                        return (IJavaToWsdlGenerator)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IJavaToWsdlGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    public IRuntimeKitProcessorTemplate getRuntimeKitTemplateProcessor(String uniqueIdentifier){
        IConfigurationElement[] extElts = getExtension(uniqueIdentifier).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_KIT_PROCESSOR)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_PROCESSOR_CLASS);
                    if (kitProcessorObj instanceof IRuntimeKitProcessorTemplate) {
                        return (IRuntimeKitProcessorTemplate)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(uniqueIdentifier,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IRuntimeKitProcessorTemplate.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(uniqueIdentifier, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    private String getLogErrMsgNotCLass(String extId, String attributeName) {
        // this msg is intended for the logs, no point to use resources bundles for this
        return "the 'IRuntimeKitProcessor' could not be instanciated from the extension "
                      + extId 
                      + ". check value of the attribute '"
                      + attributeName 
                      + "'";
    }

    @SuppressWarnings("unchecked")
    private String getLogErrMsgWrongClassType(String extId,
                                                     String attributeName,
                                                     Class superClass) {
        // this msg is intended for the logs, no point to use resources bundles for this
        return "the class specified in the attribute '"
                  + attributeName 
                  + "' of the extension '"
                  + extId
                  + "' does not extend the interface '"
                  + superClass.getCanonicalName()
                  + "'";
    }

    private IPath getKitRootFolder(String providerExtUID) {
        String kitPath = PreferencesAccessor.getKitInstallLocation(providerExtUID);
    
        if ((kitPath == null) || kitPath.equals("")) {
            kitPath = System.getProperty(SCPreferenceConstants.KEY_RUNTIME_INSTALL_ROOT); //test support
        }
    
        return new Path(kitPath);
    }

    /**
     * get parameter page according to uid
     * @param id
     * @return
     */
    public IParameterPage getParameterPage(String id) {
        IConfigurationElement[] extElts = getExtension(id).getConfigurationElements();

        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_GEN)) {
                try {
                    Object kitParamPageObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_PARAM_PAGE);
                    if (kitParamPageObj instanceof IParameterPage) {
                        return (IParameterPage)kitParamPageObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(id,
                                                         EXT_ATT_SC_W2J_PARAM_PAGE,
                                                         IJavaToWsdlGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(id, EXT_ATT_SC_W2J_PARAM_PAGE), e);
                }
                return null;
            }
        }
        return null;
    }
    
	public IParameterPage getParameterPage(IProject aProject) {
		String id = getRuntimeProviderUID(aProject);
        return getParameterPage(id);    
	}
}
