/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.wizards.annotations;


import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.dom.Annotation;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.NormalAnnotation;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.stp.sc.common.annotations.ScAnnotationSupportUtils;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.validator.AnnotationValidator;
import org.eclipse.stp.sc.common.wizards.AbstractScWizardPage;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;


public class AnnotationAttributePage extends AbstractScWizardPage {

    
    public static final String PAGE_NAME_PREFIX = "attributePage_";

    protected HashMap<String, AttributeControlProxy> attributeMap;
    protected CreateAnnotationWizard parentWizard;
    
    private String pageDesc;
    private Class<? extends java.lang.annotation.Annotation> annotationHandled;
    
    private Button btnAnnSpecified;
    private Group grpAttrib;
    private boolean isAttributeEnabledByDefault;
    
    private static ImageDescriptor defaultAnnPageImgDesc =
        ScJaxWsResources.getImageDescriptor("wizban/newannotation_wiz.gif");

    public AnnotationAttributePage(CreateAnnotationWizard aParentWizard,
                                   String aPageDesc,
                                   String aHelpContextId,
                                   Class<? extends java.lang.annotation.Annotation> anAnnotationHandled,
                                   boolean enabledByDefault) {
        
        super(aParentWizard,
              generatePageName(anAnnotationHandled),
              aHelpContextId);
        
        parentWizard = (CreateAnnotationWizard)aParentWizard;
        attributeMap = new HashMap<String, AttributeControlProxy>();
        pageDesc = aPageDesc;
        annotationHandled = anAnnotationHandled;
        isAttributeEnabledByDefault = enabledByDefault;
        
        setTitle(ScJaxWsResources.getString("annotation.wizard.attribute.page.title.start")
                 + " " 
                 + annotationHandled.getSimpleName()
                 + " "
                 + ScJaxWsResources.getString("annotation.wizard.attribute.page.title.end"));
        
        if (pageDesc == null) {
            setDescription(ScJaxWsResources.getString("annotation.wizard.attribute.page.desc.start")
                           + " "
                           + annotationHandled.getSimpleName());
        } else {
            setDescription(pageDesc);
        }
        
        setImageDescriptor(defaultAnnPageImgDesc);
    }

    public static String generatePageName(Class<? extends java.lang.annotation.Annotation> annClass) {
        return PAGE_NAME_PREFIX + annClass.getSimpleName();
    }

    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        container.setLayout(new GridLayout(2, false));

        btnAnnSpecified = new Button(container, SWT.CHECK);
        btnAnnSpecified.setText(ScJaxWsResources.getString("annotation.wizard.attribute.page.chk.lbl"));

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.horizontalSpan = 2;
        gridData.grabExcessHorizontalSpace = true;
        btnAnnSpecified.setLayoutData(gridData);
        btnAnnSpecified.setSelection(isAttributeEnabledByDefault);
        
        btnAnnSpecified.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (btnAnnSpecified.getSelection()) {
                    initializeFromDefault();
                } else {
                    clearValuesForAttributeControls();    
                }

                setAttribGoupEnabled(btnAnnSpecified.getSelection());
            }
        });

        grpAttrib = createAttributeGroup(container, annotationHandled);

        setAttribGoupEnabled(isAttributeEnabledByDefault);
        
        setControl(container);
    }

    protected Group createAttributeGroup(Composite container, 
                                         Class<? extends java.lang.annotation.Annotation> declaringClass) {
        
        Method[] methods = declaringClass.getDeclaredMethods();
        if (methods == null || methods.length <= 0) {
            return null;
        }

        Group grpAttrib = new Group(container, SWT.NONE);
        grpAttrib.setText(ScJaxWsResources.getString("annotation.wizard.attribute.page.grp.lbl"));
        grpAttrib.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL
                                             | GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL));
        grpAttrib.setLayout(new GridLayout(2, false));

        // sort the methods by name
        List<Method> methodList = new ArrayList<Method>();
        for (int i = 0; i < methods.length; i++) {
            int index = 0;
            while ((index < methodList.size())
                    && (methods[i].getName().compareTo(methodList.get(index).getName()) > 0)) {
                index++;
            }
            methodList.add(index, methods[i]);
        }

        // create a label,field control pair for each attribute
        String attribName = "";
        Label lbl = null;

        for (int i = 0; i < methodList.size(); i++) {
        
            attribName = methodList.get(i).getName();

            lbl = new Label(grpAttrib, SWT.NONE);
            lbl.setText(attribName);

            AttributeControlProxy attribCtrl = 
                AttributeControlProxy.createInstance(grpAttrib,
                                                     attribName,
                                                     methodList.get(i).getReturnType(),
                                                     new UpdatePageStatusModifyListener());
            GridData gridData = new GridData();
            gridData.grabExcessHorizontalSpace = true;
            gridData.horizontalAlignment = GridData.FILL;
            attribCtrl.getControl().setLayoutData(gridData);
            
            attributeMap.put(attribName, attribCtrl);
        }

        initializeFromDefault();
        
        return grpAttrib;
    }

    @Override
    public void applyToData(Map<String, Object> data) throws Exception {
        
        List<MemberValuePair> annotValues = null;
        if (btnAnnSpecified.getSelection()) {
            annotValues = createAnnotationValues(parentWizard.getCuAstNodeFromData(),
                                                 parentWizard.getJavaMemberFromData());
        }
        
        parentWizard.setAnnAttributesToData(CreateAnnotationWizard.getAttributesDataKey(annotationHandled),
                                            annotValues);
    }

    /**
     * convenience method to initialize the page controls based on the default settings of the targetted
     * annotation
     * @param declaringClass
     */
    @SuppressWarnings("unchecked")
    private void initializeFromDefault() {
        // setup default values now retrieve the annotation defaults
        CompilationUnit astRoot = parentWizard.getCuAstNodeFromData();
        IMember member = parentWizard.getJavaMemberFromData();
        
        Annotation ann = ScAnnotationSupportUtils.getDefaultedAnnotationNode(annotationHandled,
                                                                         astRoot,
                                                                         member,
                                                                         null);
        // when initially loading the wizard the wrong type of IMember might be selected for the targeted
        // annotation
        if (ann != null) {
            List values = ((NormalAnnotation)ann).values();
            setValuesForAttributeControls((List<MemberValuePair>)values);
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void initializeFromData(Map<String, Object> data) throws Exception {
        
        // TODO the check box needs to be initialized too
        List<MemberValuePair> annotValues = 
            parentWizard.getAnnAttributesFromData(CreateAnnotationWizard.getAttributesDataKey(annotationHandled));
        CompilationUnit astRoot = parentWizard.getCuAstNodeFromData();
                  
        Annotation annotation;
        if (annotValues != null) {
            btnAnnSpecified.setSelection(true);
            annotation = JDTUtils.newNormalAnnotation(astRoot,
                                                      annotationHandled.getSimpleName(),
                                                      annotValues);
        } else {
            IMember member = parentWizard.getJavaMemberFromData();
            annotation = ScAnnotationSupportUtils.getDefaultedAnnotationNode(annotationHandled,
                                                                         astRoot,
                                                                         member,
                                                                         null);
            Annotation existAnnot = JDTUtils.findAnnotation(astRoot, member, annotation);
            if (existAnnot != null) {
                annotation = existAnnot;
                btnAnnSpecified.setSelection(true);
            }
        }        

        setValuesForAttributeControls(((NormalAnnotation)annotation).values());
    }


    /**
     * convenience method to initialize the page controls based on the attribute details provided
     * @param attributeValues
     */
    protected void setValuesForAttributeControls(List<MemberValuePair> attributeValues) {
        
        Iterator<MemberValuePair> iNewValues = attributeValues.iterator();
        while (iNewValues.hasNext()) {
            MemberValuePair newValue = iNewValues.next();
            AttributeControlProxy attrib = attributeMap.get(newValue.getName().toString());
            
            if (attrib != null) {
                attrib.setControlValue(newValue);
            }
        }
    }

    /**
     * convenience method to clear the attribute controls
     * @param attributeValues
     */
    protected void clearValuesForAttributeControls() {
        Iterator<AttributeControlProxy> iter = attributeMap.values().iterator();
        while (iter.hasNext()) {
            iter.next().clearControlValue();
        }
    }

    /**
     * convenience method to generate the model entities representing the attributes details captured in 
     * the page
     * @param astRoot
     * @param member
     * @return the attribute details as <code>List<MemberValuePair></code> 
     */
    protected List<MemberValuePair> createAnnotationValues(CompilationUnit astRoot, IMember member) {

        List<MemberValuePair> annotValues = new ArrayList<MemberValuePair>();

        // examine each entry in the attributeMap: if a value is specified, create a MemberValuePair entry
        Object newValue = null;
        Iterator<AttributeControlProxy> attributes = attributeMap.values().iterator();

        while (attributes.hasNext()) {
            AttributeControlProxy attrib = attributes.next();
            newValue = attrib.getControlValue();

            if (newValue != null) {
                MemberValuePair valPair = JDTUtils.newMemberValuePair(astRoot,attrib.getName(), newValue);
                annotValues.add(valPair);
            }
        }

        return annotValues;
    }
    
    @Override
    protected boolean validateInputs() {
        setErrorMessage(null);
        setMessage(null);
        
        if (!btnAnnSpecified.getSelection()) {
            return true;
        }
        
        NormalAnnotation annotationToAdd = 
            JDTUtils.newNormalAnnotation(parentWizard.getCuAstNodeFromData(),
                                         annotationHandled.getCanonicalName(),
                                         createAnnotationValues(parentWizard.getCuAstNodeFromData(),
                                                                parentWizard.getJavaMemberFromData()));
        AnnotationValidator validator = new AnnotationValidator();
        
        if (!validator.validateAnn(annotationToAdd)) {
            setErrorMessage(validator.getLastErrorMessage());
            return false;
        }
        
        /*if (attributeMap.containsKey("name")) {
            String name = ((Text)attributeMap.get("name").getControl()).getText();
            
            // TODO : maybe, just maybe, we need something a bit more advanced than than
            if (name == null || name.length() == 0) {
                setErrorMessage(CommonResources.getString("error.NameAttribute.Required"));
                return false;
            }
        }*/

        return true;

    }

    public HashMap<String, AttributeControlProxy> getAttributeMap() {
        return attributeMap;
    }
    
    private void setAttribGoupEnabled(boolean isEnabled) {
        if (grpAttrib != null) {
            Control[] grpChildren = grpAttrib.getChildren();

            for (int i = 0; i < grpChildren.length; i++) {
                grpChildren[i].setEnabled(isEnabled);
            }
        }
    }
    
    public boolean isEnabled() {
        return true;
    }

}
