/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.wizards.annotations.webmethod;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.stp.sc.common.wizards.AbstractScWizardPage;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;



public class MethodSelectionPage extends AbstractScWizardPage implements ICheckStateListener {
    private static final int METHOD_SELECTION_PREFERRED_HEIGHT = 200;
    private static final String SELECT_ALL = "all";
    private static final String SELECT_NONE = "none";
    protected CheckboxTableViewer methodViewer;
    protected Button selectAllBtn;
    protected Button deselectAllBtn;
    HashMap<String, IMethod> mMapMethods;
    
    protected CreateWebMethodWizard parentWizard;

    public MethodSelectionPage(CreateWebMethodWizard aParentWizard, String pageName, String aHelpContextId) {
        super(aParentWizard, pageName, aHelpContextId);
        parentWizard = aParentWizard;
        
    }

    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        container.setLayout(new GridLayout(2, false));

        /* The list panel */
        Group listPanel = new Group(container, SWT.SHADOW_ETCHED_IN);
        listPanel.setLayout(new GridLayout(4, true));
        listPanel.setText(ScJaxWsResources.getString("MethodSelectionPage.AvailableMethod.Label"));
        listPanel.setLayoutData(new GridData(GridData.FILL_BOTH));
        createCheckBoxList(listPanel);

        GridData gridData = new GridData();
        gridData.horizontalSpan = 1;
        selectAllBtn = new Button(listPanel, SWT.PUSH | SWT.END);
        selectAllBtn.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent e) {
                    updateTable(SELECT_ALL);
                }
            });
        selectAllBtn.setText(ScJaxWsResources.getString("JaxWs.Btn.SelectAll"));
        selectAllBtn.setLayoutData(gridData);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        deselectAllBtn = new Button(listPanel, SWT.PUSH | SWT.END);
        deselectAllBtn.setText(ScJaxWsResources.getString("JaxWs.Btn.DeselectAll"));
        deselectAllBtn.setLayoutData(gridData);
        deselectAllBtn.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent e) {
                    updateTable(SELECT_NONE);
                }
            });

        setControl(container);
    }

    public void createCheckBoxList(Composite panel) {
        methodViewer = CheckboxTableViewer.newCheckList(panel, SWT.FILL);

        GridData data = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL);
        data.horizontalSpan = 4;
        data.heightHint = METHOD_SELECTION_PREFERRED_HEIGHT;
        methodViewer.getTable().setLayoutData(data);
        methodViewer.setContentProvider(new MethodListContentProvider());
        methodViewer.setLabelProvider(new MethodLabelProvider());
        methodViewer.setSorter(new ViewerSorter());
        methodViewer.setInput(new Object());
        methodViewer.addCheckStateListener(this);
        methodViewer.getTable()
                    .setBackground(parentWizard.getShell().getDisplay()
                                               .getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
    }

    protected void updateTable(String selection) {
        //TODO        LOG.debug("updating table selection=" + selection);
        if (selection.equals(SELECT_ALL)) {
            methodViewer.setAllChecked(true);
        } else if (selection.equals(SELECT_NONE)) {
            methodViewer.setAllChecked(false);
        }

        updatePageState();
    }

    @Override
    public void applyToData(Map<String, Object> data) throws Exception {
        Object[] objects = methodViewer.getCheckedElements();
        ArrayList<IMethod> methods = new ArrayList<IMethod>();

        for (int i = 0; i < objects.length; i++) {
            methods.add((IMethod)objects[i]);
        }

        data.put(CreateWebMethodWizard.DATA_KEY_SELECTED_METHODS, methods.toArray(new IMethod[0]));
    }

    @Override
    public void initializeFromData(Map<String, Object> data) throws Exception {
        // check for any selected methods
        IMethod[] methods = (IMethod[])data.get(CreateWebMethodWizard.DATA_KEY_SELECTED_METHODS);

        if ((methods == null) || (methods.length == 0)) {
            updateTable(SELECT_ALL);
        } else {
            updateTable(SELECT_NONE);

            // select the specified methods
            for (int i = 0; i < methods.length; i++) {
                String signature = getMethodSignature(methods[i]);

                if (mMapMethods.containsKey(signature)) {
                    methodViewer.setChecked(methods[i], true);
                }
            }

            updatePageState();
        }
    }

    /**
     * @return
     * @see org.eclipse.stp.sc.common.wizards.AbstractScWizardPage#validateInputs()
     */
    protected boolean validateInputs() {
        setErrorMessage(null);
        setMessage(null);

        // must be at least one method selected
        if (methodViewer != null) {
            Object[] methods = methodViewer.getCheckedElements();

            if (methods.length < 1) {
                setErrorMessage(ScJaxWsResources.getString("MethodSelectionPage.MissingMethod.Error"));

                return false;
            }
        }

        return true;
    }

    public void checkStateChanged(CheckStateChangedEvent event) {
        updatePageState();
    }

    public Button getSelectAllBtn() {
        return selectAllBtn;
    }

    private String getMethodSignature(IMethod method) {
        try {
            return Signature.toString(method.getSignature(),
                method.getElementName(), null, false, false);
        } catch (Exception e) {
            return null;
        }
    }

    private IMethod[] getFileMethods() {
        mMapMethods = new HashMap<String, IMethod>();

        try {
            IJavaElement[] children = parentWizard.getJavaMemberFromData().getCompilationUnit()
                                       .findPrimaryType().getChildren();

            for (int i = 0; i < children.length; i++) {
                if (children[i].getElementType() == IJavaElement.METHOD) {
                    IMethod method = (IMethod)children[i];
                    int flags = method.getFlags();

                    if (Flags.isPrivate(flags) || Flags.isProtected(flags)) {
                        continue;
                    }

                    String methodSignature = getMethodSignature(method);

                    if (!mMapMethods.containsKey(methodSignature)) {
                        mMapMethods.put(methodSignature, method);
                    }
                }
            }

            return mMapMethods.values().toArray(new IMethod[0]);
        } catch (Exception e) {
            return null;
        }
    }

    public class MethodListContentProvider implements IStructuredContentProvider {
        private Object[] items = getFileMethods();

        public Object[] getElements(Object inputElement) {
            return items;
        }

        public void dispose() {
        }

        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }

    class MethodLabelProvider extends LabelProvider {
        public String getText(Object element) {
            if ((element == null) || !(element instanceof IMethod)) {
                return null;
            }

            return getMethodSignature((IMethod)element);
        }
    }
}
