/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core.operations;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.ui.RefreshProfileJob;
import org.eclipse.datatools.connectivity.ui.dse.DSEPlugin;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.IDeploySession;
import org.eclipse.stp.soas.deploy.core.IDeployTarget;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.core.cp.StpProfileManager;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.DeployStatusDialogPage;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.Dialog;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.PreDeployCheckDialogPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.navigator.CommonNavigator;
import org.eclipse.ui.progress.IProgressConstants;

/**
 * @author rcernich
 * 
 * Created on Feb 24, 2004
 */
public class ExecuteUndeploymentJob extends Job {

	private Root mRoot;
	private Map mDeployConfigToDeployTargetMap = new HashMap();
	private Map mDeployTargetToDeploySessionMap = new HashMap();
	private Map mDeployStatusMap = new HashMap();
	private boolean mHasErrorStatus = false;
	private CreateDeployPackagesJob mCreatePackages;
	private Shell mShell;

	/**
	 * 
	 */
	public ExecuteUndeploymentJob(Root root, Shell shell) {
		super(DeployCorePlugin.getDefault().getResourceString(
				"ExecuteUndeploymentJob.task.name"));
		mRoot = root;
		mShell = shell;
		mCreatePackages = new CreateDeployPackagesJob(mRoot);
		setRule(ResourcesPlugin.getWorkspace().getRoot());
		setUser(true);
		setSystem(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IStatus run(IProgressMonitor monitor) {
		monitor.beginTask(DeployCorePlugin.getDefault().getResourceString(
				"ExecuteUndeploymentJob.task.name"), 3000); //$NON-NLS-1$

		mDeployStatusMap.clear();
		mHasErrorStatus = false;

		IStatus status = Status.OK_STATUS;

		try {
			Map packageDescriptors = createPackages(monitor);

			preUndeployCheck(packageDescriptors, monitor);

			undeployPackages(packageDescriptors, monitor);

			if (mHasErrorStatus) {
				status = new Status(Status.ERROR, DeployCorePlugin
						.getDefault().getBundle().getSymbolicName(), -1,
						DeployCorePlugin.getDefault().getResourceString(
								"ExecuteUndeploymentJob.error.job"), null); //$NON-NLS-1$
			}
		}
		catch (CoreException e) {
			status = e.getStatus();
		}
		catch (InterruptedException e) {
			status = Status.CANCEL_STATUS;
		}
		finally {
			// Clean up an open sessions.
			for (Iterator it = mDeployTargetToDeploySessionMap.values()
					.iterator(); it.hasNext();) {
				IDeploySession session = (IDeploySession) it.next();
				if (session != null) {
					try {
						session.dispose();
					}
					catch (Exception e) {
						e.printStackTrace();
					}
				}
			}

			refreshDSE(monitor);

			mDeployConfigToDeployTargetMap.clear();
			mDeployTargetToDeploySessionMap.clear();
		}

		logStatus();

		monitor.done();

		setProperty(IProgressConstants.KEEP_PROPERTY, Boolean.TRUE);
		setProperty(IProgressConstants.ACTION_PROPERTY, new Action(
				DeployCorePlugin.getDefault().getResourceString(
						"ExecuteUndeploymentJob.action.name")) { //$NON-NLS-1$

			public void run() {
				displayStatus();
			}
		});

		return status;
	}

	public Map getDeployStatusMap() {
		return mDeployStatusMap;
	}

	private void checkCancelled(IProgressMonitor monitor)
			throws InterruptedException {
		if (monitor.isCanceled()) {
			throw new InterruptedException();
		}
	}

	private Map createPackages(IProgressMonitor monitor) throws CoreException,
			InterruptedException {
		SubProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1000);
		IStatus status = mCreatePackages.run(subMonitor);
		if (status.getSeverity() != IStatus.OK) {
			throw new CoreException(status);
		}

		checkCancelled(monitor);

		return mCreatePackages.getPackageDescriptors();
	}

	private void preUndeployCheck(Map packageDescriptors,
			final IProgressMonitor monitor) throws CoreException,
			InterruptedException {
		int deployTaskLength = getDeployTaskLength();
		Map deployCheckStatus = new HashMap();
		int severity = 0;

		for (Iterator it = packageDescriptors.entrySet().iterator(); it
				.hasNext();) {
			Map.Entry entry = (Map.Entry) it.next();
			DeployConfiguration config = (DeployConfiguration) entry.getKey();
			IPackageOutputDescriptor descriptor = (IPackageOutputDescriptor) entry
					.getValue();

			monitor
					.subTask(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"ExecuteDeploymentJob.subtask.preDeployCheck", //$NON-NLS-1$
									new Object[] {
											config.getSourcePackage()
													.getPackageFile(),
											config.getTargetServer()
													.getProfileName()}));

			if (descriptor == null) {
				String exceptionMessage = DeployCorePlugin.getDefault()
						.getResourceString(
								"ExecuteDeploymentJob.error.deployError", //$NON-NLS-1$
								new Object[] {
										config.getSourcePackage()
												.getPackageFile(),
										config.getTargetServer()
												.getProfileName()});
				IStatus status = new Status(
						Status.ERROR,
						DeployCorePlugin.getDefault().getBundle()
								.getSymbolicName(),
						-1,
						DeployCorePlugin.getDefault().getResourceString(
								"ExecuteDeploymentJob.error.log.message"), //$NON-NLS-1$
						new Exception(exceptionMessage));
				throw new CoreException(new MultiStatus(DeployCorePlugin
						.getDefault().getBundle().getSymbolicName(), -1,
						new IStatus[] { status}, DeployCorePlugin
								.getDefault().getResourceString(
										"UndeployStatusDialog.TITLE"), //$NON-NLS-1$
						null));
			}
			else {
				IDeployTarget target = Utilities.adaptToIDeployTarget(config);
				IDeploySession session = target.createDeploySession();

				mDeployConfigToDeployTargetMap.put(config, target);
				mDeployTargetToDeploySessionMap.put(target, session);

				IStatus[] status = session.preUndeployCheck(descriptor);
				deployCheckStatus.put(config, status);
				if (status != null && status.length > 0) {
					for (int index = 0, count = status.length; (severity & IStatus.ERROR) == 0
							&& index < count; ++index) {
						severity |= status[index].getSeverity();
					}
				}
			}

			monitor.worked(deployTaskLength);

			if (severity != 0) {
				Display display = PlatformUI.getWorkbench().getDisplay();
				final PreDeployCheckDialogPage page = new PreDeployCheckDialogPage(
						deployCheckStatus, (severity & IStatus.ERROR) == 0);

				display.syncExec(new Runnable() {

					public void run() {
						boolean disposeShell = false;
						Display display = Display.getCurrent();
						Shell shell = display.getActiveShell();
						if (shell == null) {
							Shell shells[] = display.getShells();
							if (shells.length > 0) {
								shell = shells[shells.length - 1];
							}
							else {
								disposeShell = true;
								shell = new Shell();
							}
						}
						try {
							Dialog dialog = new Dialog(shell, page, SWT.RESIZE);
							if (dialog.open() == Dialog.CANCEL) {
								monitor.setCanceled(true);
							}
						}
						finally {
							if (disposeShell) {
								shell.dispose();
							}
						}
					}
				});
			}

			checkCancelled(monitor);
		}
		checkCancelled(monitor);
	}

	private void undeployPackages(Map packageDescriptors, IProgressMonitor monitor)
			throws CoreException, InterruptedException {
		int deployTaskLength = getDeployTaskLength();

		for (Iterator it = packageDescriptors.entrySet().iterator(); it
				.hasNext();) {
			Map.Entry entry = (Map.Entry) it.next();
			DeployConfiguration config = (DeployConfiguration) entry.getKey();
			IPackageOutputDescriptor descriptor = (IPackageOutputDescriptor) entry
					.getValue();

			monitor
					.subTask(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"ExecuteDeploymentJob.subtask.deploy", //$NON-NLS-1$
									new Object[] {
											config.getSourcePackage()
													.getPackageFile(),
											config.getTargetServer()
													.getProfileName()}));

			if (descriptor == null
					|| !new File(descriptor.getFileName()).exists()) {
				mDeployStatusMap.put(config, new IStatus[] { new Status(
						IStatus.ERROR, DeployCorePlugin.getDefault()
								.getBundle().getSymbolicName(), -1,
						DeployCorePlugin.getDefault().getResourceString(
								"Deploy.ERROR.PackageDoesNotExist", //$NON-NLS-1$
								new Object[] {
										config.getSourcePackage()
												.getPackageFile(),
										config.getTargetServer()
												.getProfileName()}), null)});
				mHasErrorStatus = true;
			}
			else {
				IDeployTarget target = (IDeployTarget) mDeployConfigToDeployTargetMap
						.get(config);
				IDeploySession session = (IDeploySession) mDeployTargetToDeploySessionMap
						.get(target);
				try {
					session.undeploy(descriptor);
					mDeployStatusMap
							.put(
									config,
									new IStatus[] { new Status(
											IStatus.INFO,
											DeployCorePlugin.getDefault()
													.getBundle()
													.getSymbolicName(),
											-1,
											DeployCorePlugin
													.getDefault()
													.getResourceString(
															"Undeploy.SUCCESS.Generic", //$NON-NLS-1$
															new Object[] {
																	config
																			.getSourcePackage()
																			.getPackageFile(),
																	config
																			.getTargetServer()
																			.getProfileName()}),
											null)});
				}
				catch (Exception e) {
					mDeployStatusMap
							.put(
									config,
									new IStatus[] { new Status(
											IStatus.ERROR,
											DeployCorePlugin.getDefault()
													.getBundle()
													.getSymbolicName(),
											-1,
											DeployCorePlugin
													.getDefault()
													.getResourceString(
															"Undeploy.ERROR.Generic", //$NON-NLS-1$
															new Object[] {
																	config
																			.getSourcePackage()
																			.getPackageFile(),
																	target
																			.getConnectionProfile()
																			.getName(),
																	e
																			.getMessage()}),
											e)});
					mHasErrorStatus = true;
				}
				finally {
					if (session != null) {
						session.dispose();
					}
					mDeployTargetToDeploySessionMap.remove(target);
				}
			}
			monitor.worked(deployTaskLength);

			checkCancelled(monitor);
		}
		checkCancelled(monitor);
	}

	private void refreshDSE(IProgressMonitor monitor) {
		IViewPart dseView = null;
		IWorkbenchWindow[] workbenchWindows = DeployCorePlugin
				.getDefault().getWorkbench().getWorkbenchWindows();

		// Find the enterprise browser
		if (workbenchWindows != null) {
			for (int winIndex = 0, winCount = workbenchWindows.length; dseView == null
					&& winIndex < winCount; ++winIndex) {
				IWorkbenchPage workbenchPages[] = workbenchWindows[winIndex]
						.getPages();
				if (workbenchPages != null) {
					IViewReference dseReference = null;
					for (int pageIndex = 0, pageCount = workbenchPages.length; dseReference == null
							&& pageIndex < pageCount; ++pageIndex) {
						dseReference = workbenchPages[pageIndex]
								.findViewReference(DSEPlugin.SERVERS_VIEW_VIEWER_ID);
					}
					if (dseReference != null) {
						dseView = dseReference
								.getView(false);
					}
				}
			}
		}

		// Refresh the browser
		if (dseView != null
				&& dseView instanceof CommonNavigator) {
			int refreshTaskLength = getRefreshTaskLength();
			final CommonNavigator serversView = (CommonNavigator) dseView;
			Display display = serversView.getSite().getPage()
					.getWorkbenchWindow().getShell().getDisplay();
			final TreeViewer viewer = serversView.getCommonViewer();

			// Bring the enterprise browser to the front
			display.syncExec(new Runnable() {

				public void run() {
					serversView.getViewSite().getPage().activate(serversView);
				}
			});

			for (Iterator it = mRoot.getServer().iterator(); it.hasNext();) {
				DeployServer server = (DeployServer) it.next();
				final IConnectionProfile profile = StpProfileManager.getInstance()
						.getProfileByName(server.getProfileName());

				// Refresh the profile
				if (profile.isConnected()) {
					RefreshProfileJob
							.scheduleRefreshProfileJob(profile, viewer);
				}

				monitor.worked(refreshTaskLength);
			}
		}
	}

	private int getDeployTaskLength() {
		int retVal = 0;
		for (Iterator it = mRoot.getPackage().iterator(); it.hasNext();) {
			retVal += ((DeployPackage) it.next()).getTargetConfiguration()
					.size();
		}
		return retVal / 1000 + 1;
	}

	private int getRefreshTaskLength() {
		return mRoot.getServer().size() / 1000 + 1;
	}

	private void logStatus() {
		Collection statuses = new ArrayList();
		for (Iterator it = mDeployStatusMap.values().iterator(); it.hasNext();) {
			statuses.addAll(Arrays.asList((IStatus[]) it.next()));
		}
		DeployCorePlugin.getDefault().log(
				new MultiStatus(DeployCorePlugin.getDefault().getBundle()
						.getSymbolicName(), -1, (IStatus[]) statuses
						.toArray(new IStatus[statuses.size()]),
						DeployCorePlugin.getDefault().getResourceString(
								"UndeployStatusDialog.TITLE"), //$NON-NLS-1$
						null));
	}

	private void displayStatus() {
		Display display = PlatformUI.getWorkbench().getDisplay();
		final DeployStatusDialogPage page = new DeployStatusDialogPage(
				mDeployStatusMap);

		display.syncExec(new Runnable() {

			public void run() {
				boolean disposeShell = false;
				Display display = Display.getCurrent();
				Shell shell = mShell;
				if (shell == null) {
					Shell shells[] = display.getShells();
					if (shells.length > 0) {
						shell = shells[shells.length - 1];
					}
					else {
						disposeShell = true;
						shell = new Shell();
					}
				}
				try {
					Dialog dialog = new Dialog(shell, page, SWT.RESIZE);
					dialog.open();
				}
				finally {
					if (disposeShell) {
						shell.dispose();
					}
				}
			}
		});
	}

}
