/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.core;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.datatools.connectivity.IConnectionProfileProvider;
import org.eclipse.datatools.connectivity.internal.ConnectionProfileManager;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IDeployDriverExtension;
import org.eclipse.stp.soas.deploy.core.IDeploySession;
import org.eclipse.stp.soas.deploy.core.IDeploySessionFactory;
import org.eclipse.stp.soas.deploy.core.IDeployTarget;
import org.eclipse.stp.soas.deploy.core.ILogicalPackage;
import org.eclipse.stp.soas.deploy.core.IPackageConfiguration;
import org.eclipse.stp.soas.deploy.core.IPackageConstructor;
import org.eclipse.stp.soas.deploy.core.IPackageConstructorExtension;
import org.eclipse.stp.soas.deploy.core.IPackageCreationContext;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IPackageValidator;
import org.eclipse.stp.soas.deploy.core.IServerType;
import org.eclipse.stp.soas.deploy.core.ISupportValidator;
import org.eclipse.stp.soas.deploy.core.ISupportedServerType;
import org.eclipse.stp.soas.deploy.core.ISupportedTechnologyType;
import org.eclipse.stp.soas.deploy.core.ITechnologyType;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;


/**
 * @author rcernich
 * 
 * Created on Jan 16, 2004
 */
public class DeployDriverExtension implements IDeployDriverExtension {

	public static final String ATTR_ID = "id"; //$NON-NLS-1$
	public static final String ATTR_PROFILE = "profile"; //$NON-NLS-1$
	public static final String ATTR_CLASS = "class"; //$NON-NLS-1$
	public static final String ATTR_NAME = "name"; //$NON-NLS-1$
	public static final String ATTR_TRANSACTIONAL = "isTransactional"; //$NON-NLS-1$
	public static final String ELEM_SERVER = "supportedServer"; //$NON-NLS-1$
	public static final String ELEM_PACKAGE_CONSTRUCTOR = "packageConstructor"; //$NON-NLS-1$
	public static final String ELEM_SUPPORTED_TECHNOLOGY = "supportedTechnology"; //$NON-NLS-1$
	public static final String ATTR_SUPPORT_VALIDATOR = "supportValidator"; //$NON-NLS-1$
	public static final String ATTR_PACKAGE_VALIDATOR = "packageValidator"; //$NON-NLS-1$

	private String mID;
	private String mName;
	private String mProfile;
	private boolean mTransactional;
	private IDeploySessionFactory mFactory;
	private ISupportedServerType mSupportedServer;
	private Map mIdToPackageConstructor;
	private IConfigurationElement mElement;

	public DeployDriverExtension(IConfigurationElement element)
			throws PackageExtensionException {
		super();
		init(element);
	}

	public IDeploySession createDeploySession(IConnectionProfile profile) {
		try {
			initFactory();
			return mFactory.createDeploySession(profile);
		}
		catch (CoreException e) {
			e.printStackTrace();
			return null;
		}
	}

	public IConnectionProfileProvider getConnectionProfileProvider() {
		return ConnectionProfileManager.getInstance().getProvider(mProfile);
	}

	public String getID() {
		return mID;
	}

	public String getName() {
		return mName;
	}

	public boolean isTransaction() {
		return mTransactional;
	}

	public IDeployTarget adaptProfile(IConnectionProfile profile) {
		IDeployTarget idt;
		if (profile.getProviderId().equals(
				getConnectionProfileProvider().getId())) {
			idt = new DeployTarget(this, profile);
		}
		else {
			idt = null;
		}
		return idt;
	}

	public ISupportedServerType getSupportedServerType() {
		return mSupportedServer;
	}

	public boolean supportsServer(IServerType type) {
		return mSupportedServer.supportsServer(type);
	}

	public List getPackageConstructors() {
		return new ArrayList(mIdToPackageConstructor.values());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IDeployDriverExtension#getPackageConstructor(org.eclipse.stp.soas.deploy.core.ITechnologyType)
	 */
	public IPackageConstructorExtension getPackageConstructor(
			ITechnologyType type) {
		IPackageConstructorExtension retVal = null;
		for (Iterator it = mIdToPackageConstructor.values().iterator(); retVal == null
				&& it.hasNext();) {
			IPackageConstructorExtension tmp = (IPackageConstructorExtension) it
					.next();
			if (tmp.getSupportedTechnologyType().supportsTechnology(type)) {
				retVal = tmp;
			}
		}
		return retVal;
	}

	private void init(IConfigurationElement element)
			throws PackageExtensionException {
		Assert.isTrue(DeploymentExtensionManager.EXT_ELEM_DEPLOY_DRIVER
				.equals(element.getName()));

		mElement = element;

		mID = element.getAttribute(ATTR_ID);
		mName = element.getAttribute(ATTR_NAME);
		mProfile = element.getAttribute(ATTR_PROFILE);
		mTransactional = Boolean.valueOf(
				element.getAttribute(ATTR_TRANSACTIONAL)).booleanValue();

		if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
			System.out
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.trace.processingDeployDriver", //$NON-NLS-1$
									new Object[] { mID}));
			System.out.flush();
		}

		if (getConnectionProfileProvider() == null) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.profileNotSpecified", //$NON-NLS-1$
										new Object[] { mID, mProfile}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.profileNotSpecified")); //$NON-NLS-1$
		}

		IConfigurationElement[] supportedServers = element
				.getChildren(ELEM_SERVER);
		if (supportedServers.length < 1) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.supportedServerNotSpecified", //$NON-NLS-1$
										new Object[] { mID}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.supportedServerNotSpecified")); //$NON-NLS-1$
		}
		else if (supportedServers.length > 1) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.multipleSupportedServerElements", //$NON-NLS-1$
										new Object[] { mID}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.multipleSupportedServerElements")); //$NON-NLS-1$
		}

		processServer(supportedServers[0]);

		processPackageConstructors(element
				.getChildren(ELEM_PACKAGE_CONSTRUCTOR));
	}

	private void processServer(IConfigurationElement element)
			throws PackageExtensionException {
		mSupportedServer = new SupportedServerType(element);
	}

	private void processPackageConstructors(IConfigurationElement[] elements)
			throws PackageExtensionException {
		mIdToPackageConstructor = new TreeMap();
		for (int index = 0, count = elements.length; index < count; ++index) {
			processPackageConstructor(elements[index]);
		}
	}

	private void processPackageConstructor(IConfigurationElement element)
			throws PackageExtensionException {
		IPackageConstructorExtension ipce = new PackageConstructorExtension(
				element);
		if (mIdToPackageConstructor.containsKey(ipce.getID())) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.duplicatePackageConstructor", //$NON-NLS-1$
										new Object[] { mID, ipce.getID()}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.duplicatePackageConstructor")); //$NON-NLS-1$
		}
		else {
			mIdToPackageConstructor.put(ipce.getID(), ipce);
		}
	}

	private void initFactory() throws CoreException {
		if (mFactory == null) {
			mFactory = (IDeploySessionFactory) mElement
					.createExecutableExtension(ATTR_CLASS);
		}
	}

	public class PackageConstructorExtension implements
			IPackageConstructorExtension {

		private IConfigurationElement mElement;
		private String mID;
		private IPackageConstructor mConstructor;
		private ISupportValidator mSupportValidator;
		private IPackageValidator mPackageValidator;
		private ISupportedTechnologyType mSupportedTechnology;

		public PackageConstructorExtension(IConfigurationElement element)
				throws PackageExtensionException {
			super();
			init(element);
		}

		public IConfigurationElement getConfigurationElement() {
			return mElement;
		}

		public String getID() {
			return mID;
		}

		public IPackageOutputDescriptor createPackage(ILogicalPackage pkg,
				IPackageCreationContext context,
				IPackageConfiguration configuration) throws CoreException {
			IPackageOutputDescriptor retVal;
			initConstructor();
			if (mConstructor == null) {
				IStatus status = new Status(
						IStatus.ERROR,
						DeployCorePlugin.getDefault().getBundle()
								.getSymbolicName(),
						-1,
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.error.couldNotLoadPackageConstructor"), null); //$NON-NLS-1$
				throw new CoreException(status);
			}
			else {
				retVal = mConstructor
						.createPackage(pkg, context, configuration);
			}

			return retVal;
		}

		public boolean supportsPackage(ILogicalPackage pkg) {
			boolean retVal;
			retVal = getSupportedTechnologyType().supportsTechnology(
					pkg.getTechnologyType());
			if (retVal) {
				initSupportValidator();
				if (mSupportValidator != null) {
					retVal = retVal
							&& mSupportValidator.supportsFile(pkg.getFile());
				}
			}
			return retVal;
		}

		public IStatus[] validatePackage(ILogicalPackage pkg,
				IPackageConfiguration configuration, IDeployTarget target) {
			IStatus[] retVal;
			initPackageValidator();
			if (mPackageValidator == null) {
				retVal = new IStatus[0];
			}
			else {
				retVal = mPackageValidator.validate(pkg, configuration, target);
			}
			return retVal;
		}

		public ISupportedTechnologyType getSupportedTechnologyType() {
			return mSupportedTechnology;
		}

		public IDeployDriverExtension getDeployDriver() {
			return DeployDriverExtension.this;
		}

		private void init(IConfigurationElement element)
				throws PackageExtensionException {
			mElement = element;

			mID = element.getAttribute(ATTR_ID);

			IConfigurationElement[] supportedTechnologies = element
					.getChildren(ELEM_SUPPORTED_TECHNOLOGY);
			if (supportedTechnologies.length < 1) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeployDriverExtension.trace.error.supportedTechnologyNotSpecified", //$NON-NLS-1$
											new Object[] {
													getDeployDriver().getID(),
													mID}));
					System.err.flush();
				}
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.exception.supportedTechnologyNotSpecified")); //$NON-NLS-1$
			}
			else if (supportedTechnologies.length > 1) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeployDriverExtension.trace.error.multipleSupportedTechnologyElements", //$NON-NLS-1$
											new Object[] {
													getDeployDriver().getID(),
													mID}));
					System.err.flush();
				}
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.exception.multipleSupportedTechnologyElements")); //$NON-NLS-1$
			}

			processTechnology(supportedTechnologies[0]);
		}

		private void processTechnology(IConfigurationElement technology)
				throws PackageExtensionException {
			mSupportedTechnology = new SupportedTechnologyType(technology);
		}

		private void initConstructor() {
			if (mConstructor == null) {
				try {
					mConstructor = (IPackageConstructor) mElement
							.createExecutableExtension(ATTR_CLASS);
				}
				catch (CoreException e) {
					if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
						System.err
								.println(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"DeployDriverExtension.trace.error.couldNotLoadPackageConstructor", //$NON-NLS-1$
												new Object[] {
														getDeployDriver()
																.getID(),
														getID(),
														mElement
																.getAttribute(ATTR_CLASS)}));
						System.err.flush();
					}
				}
			}
		}

		private void initSupportValidator() {
			if (mSupportValidator == null) {
				try {
					mSupportValidator = (ISupportValidator) mElement
							.createExecutableExtension(ATTR_SUPPORT_VALIDATOR);
				}
				catch (CoreException e) {
					if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
						System.err
								.println(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"DeployDriverExtension.trace.error.couldNotLoadSupportValidator", //$NON-NLS-1$
												new Object[] {
														getDeployDriver()
																.getID(),
														getID(),
														mElement
																.getAttribute(ATTR_SUPPORT_VALIDATOR)}));
						System.err.flush();
					}
				}
			}
		}

		private void initPackageValidator() {
			if (mPackageValidator == null) {
				try {
					mPackageValidator = (IPackageValidator) mElement
							.createExecutableExtension(ATTR_PACKAGE_VALIDATOR);
				}
				catch (CoreException e) {
					if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
						System.err
								.println(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"DeployDriverExtension.trace.error.couldNotLoadPackageValidator", //$NON-NLS-1$
												new Object[] {
														getDeployDriver()
																.getID(),
														getID(),
														mElement
																.getAttribute(ATTR_SUPPORT_VALIDATOR)}));
						System.err.flush();
					}
				}
			}
		}
	}

}