/*******************************************************************************

* Copyright (c) 2007 IONA Technologies PLC

* All rights reserved. This program and the accompanying materials

* are made available under the terms of the Eclipse Public License v1.0

* which accompanies this distribution, and is available at

* http://www.eclipse.org/legal/epl-v10.html

* 

* Contributors:

*     IONA Technologies PLC - initial API and implementation

*******************************************************************************/
package org.eclipse.stp.sc.common.runtimeintegrator;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.ExtensionRegUtils;

/**
 * The class is managed the runtime intergator extensions. 
 * it will load all runtime integration extenions during startup 
 * from Extension Registry
 * @author jma
 */
public abstract class RtIntegratorMgr {

    private static final LoggingProxy LOG = LoggingProxy.getlogger(RtIntegratorMgr.class);
    
    public static final String EXT_POINT_RUNTIMEINTEGRATOR = "org.eclipse.stp.sc.common.runtimeIntegrator";
    
    public static final QualifiedName PROPERTY_KEY_RUNTIME_INTEGRATOR_UID = new QualifiedName(
            "stp.sc",
            "runtimeintegrator_id");
    
    protected Hashtable<String, ExtRtIntegrator> runtimeIntTable = 
        new Hashtable<String, ExtRtIntegrator>(); 
       
    public RtIntegratorMgr() {
        loadRuntimeIntegrators();
    }
    
    /**
     * get runtime integrators by server type.
     * For example, we can get all jax-ws runtime integrators by pass "jaxws.runtime" server type 
     * @param serverType
     * @return
     */
    public ExtRtIntegrator[] getRuntimeIntegrators(String serverType) {
        ArrayList<ExtRtIntegrator> integrators = new ArrayList<ExtRtIntegrator>();
        for (ExtRtIntegrator runtimeInt : runtimeIntTable.values()) {
            if (runtimeInt.getTargetServerType().equals(serverType)) {
                integrators.add(runtimeInt);
            }
        }
//        return (ExtRtIntegrator[])integrators.toArray();
        return (integrators.toArray(new ExtRtIntegrator[0]));
    }
    
    private void loadRuntimeIntegrators() {
        IExtensionPoint integratorPoint = ExtensionRegUtils.getExtensionPoint(
                EXT_POINT_RUNTIMEINTEGRATOR);
        IExtension[] extensions = integratorPoint.getExtensions();
        for (IExtension ext : extensions) {
            ExtRtIntegrator runtime;
            try {
                runtime = RtIntegratorFactory.createRuntimeIntegrator(ext);
                runtimeIntTable.put(runtime.getId(), runtime);
                LOG.debug("loaded runtime integrator:" + runtime.toString());
            } catch (CoreException e) {
                LOG.error(e);                
                e.printStackTrace();
            }
        }
    }
    
    /**
     * get generator tool defined in the system by tool id
     * @param toolID
     * @return
     * @throws CoreException 
     */
    public IGeneratorTool getGeneratorTool(String toolID) throws CoreException {
        for (ExtRtIntegrator integrator : runtimeIntTable.values()) {
            for (ExtGeneratorTool tool : integrator.getTools()) {
                if (tool.getToolType().equals(toolID)) {
                    return tool.getToolClass();
                }
            }    
        }
        return null;
    }
    
    
    
    /**
     * get generator tools according runtime extension id and tool type
     * @param runtimeIntegratorID, the runtime integration extension id
     * @param toolType, the tool type id, such as "org.eclipse.stp.sc.jaxws.wsdl2java"
     * @return array of IGeneratorTool
     * @throws CoreException 
     */
    public IGeneratorTool getGeneratorTool(String runtimeIntegratorID, String toolType) throws CoreException {
        ExtRtIntegrator runtimeInt = runtimeIntTable.get(runtimeIntegratorID);
        for (ExtGeneratorTool tool : runtimeInt.getTools()) {
            if (tool.getToolType().equals(toolType)) {
                return tool.getToolClass();
            }
        }
        return null;
    }
    
        
    public IRuntimeKitProcessorTemplate getKitProcessorTemplate(String runtimeIntegratorID) throws CoreException {
        return runtimeIntTable.get(runtimeIntegratorID).getKitTemplate();
    }
    
    public IParameterPage getParameterPage(String runtimeExtID, String toolType) throws CoreException {
        ExtRtIntegrator runtimeInt = runtimeIntTable.get(runtimeExtID);
        for (ExtGeneratorTool tool : runtimeInt.getTools()) {
            if (tool.getToolType().equals(toolType)) {
                return tool.getParameterPage();
            }
        }
        return null;
        
    }
    
    public IParameterPage getParameterPage(IProject project, String toolType) throws CoreException {
        String runtimeId = getRuntimeIntegratorID(project);
        return getParameterPage(runtimeId, toolType);
    }
    
    
    /**
     * get the default runtime integrator extension id from preference page
     * @return
     */
    public abstract String getDefaultRuntimeIntegratorID();
    
    /**
     * get runtime integrator id for project. if the project has setup runtime, then return
     * the project specific one. otherwise, return the default runtime id defined in preference page
     * @param project,
     * @return String, the runtime extension id
     * @throws CoreException 
     */
    public String getRuntimeIntegratorID(IProject project) throws CoreException {
        String projectRuntimeID = getRuntimeIDProjectProperty(project);
        if (projectRuntimeID != null) {
            return projectRuntimeID;
        }
        return getDefaultRuntimeIntegratorID();
    }
    
    protected String getRuntimeIDProjectProperty(IProject project) throws CoreException {
        return project.getPersistentProperty(PROPERTY_KEY_RUNTIME_INTEGRATOR_UID);
    }
    
}
