/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.internal.infrastructure.emf;

import java.util.Arrays;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;

class ScribblerTracker {

   private static final UnreleasedScribblerException[] NO_UNRELEASED_SCRIBBLERS = new UnreleasedScribblerException[0];

   private final WeakHashMap                           scribblerRegistry        = new WeakHashMap();

   protected final Set                                 accessedScribblers       = new HashSet();

   private final Set                                   scribblerDomains         = new HashSet();

   private final Set                                   resourceDescriptors      = new HashSet();

   private IScribblerDomain[]                          domainsArray;

   public IResourceDescriptor[]                        descriptorsArray;

   private final Map                                   referenceCounts          = new HashMap();

   private final String                                editModelLabel;

   public ScribblerTracker(String anEditModelLabel) {
      editModelLabel = anEditModelLabel;
   }

   public synchronized void track(IEditModelScribbler scribbler) {
      Assert.isNotNull(scribbler);
      UnreleasedScribblerException exception = new UnreleasedScribblerException(
            editModelLabel);
      if (!scribblerRegistry.containsKey(scribbler)) {
         scribblerRegistry.put(scribbler, exception);
         accessedScribblers.add(exception);
      }
      updateDomainSet(scribblerRegistry.keySet());

   }

   public synchronized void release(IEditModelScribbler scribbler) {
      Assert.isNotNull(scribbler);
      if (scribblerRegistry.containsKey(scribbler)) {
         UnreleasedScribblerException exception = (UnreleasedScribblerException) scribblerRegistry
               .remove(scribbler);
         accessedScribblers.remove(exception);
      }
      updateDomainSet(scribblerRegistry.keySet());
   }

   public UnreleasedScribblerException[] computeUnreleasedScribblers() {
      Set results = new HashSet();
      UnreleasedScribblerException[] goodScribblers = (UnreleasedScribblerException[]) scribblerRegistry
            .values().toArray(new UnreleasedScribblerException[results.size()]);
      UnreleasedScribblerException[] badScribblers = (UnreleasedScribblerException[]) accessedScribblers
            .toArray(new UnreleasedScribblerException[results.size()]);

      boolean found = false;
      for (int j = 0; j < badScribblers.length; j++) {
         found = false;
         for (int i = 0; i < goodScribblers.length && !found; i++) {
            if (goodScribblers[i] == badScribblers[j]) {
               found = true;
            }
         }
         if (!found) {
            // return this exception
            results.add(badScribblers[j]);
            // remove it from the list
            accessedScribblers.remove(badScribblers[j]);
         }
      }
      if (results.size() == 0)
         return NO_UNRELEASED_SCRIBBLERS;
      return (UnreleasedScribblerException[]) results
            .toArray(new UnreleasedScribblerException[results.size()]);
   }

   /**
    * Returns an unmodifiable set of scribblers. Be warned that the set could
    * change while in use if Scribblers are garbage collected.
    * 
    * @return
    */
   public synchronized Set getScribblers() {
      return Collections.unmodifiableSet(scribblerRegistry.keySet());
   }

   /**
    * 
    * @return the number of valid scribblers currently managed.
    */
   public int getSize() {
      // when a GC cycle occurs, some keys can be nulled out
      scribblerRegistry.remove(null);
      return scribblerRegistry.keySet().size();
   }

   /**
    * @return a Set of
    *         {@link org.eclipse.stp.core.infrastructure.emf.IScribblerDomain}
    */
   public IScribblerDomain[] getScribblerDomains() {
      synchronized (ScribblerTracker.this) {
         if (domainsArray == null)
            domainsArray = (IScribblerDomain[]) scribblerDomains
                  .toArray(new IScribblerDomain[scribblerDomains.size()]);
         return domainsArray;
      }
   }

   public IResourceDescriptor[] getResourceDescriptors() {

      synchronized (this) {
         if (descriptorsArray == null)
            descriptorsArray = (IResourceDescriptor[]) resourceDescriptors
                  .toArray(new IResourceDescriptor[resourceDescriptors.size()]);
         return descriptorsArray;
      }
   }

   public int size() {
      return scribblerRegistry.size();
   }

   public String toString() {
      return NLS.bind(Messages.ScribblerTracker_tracking_, new Integer(size()));
   }

   public boolean isShared(IScribblerDomain aDomain) {
      int[] reference = (int[]) referenceCounts.get(aDomain);
      return reference != null ? reference[0] > 1 : false;
   }

   private synchronized void updateDomainSet(Set scribblers) {

      synchronized (this) {
         domainsArray = null;
         descriptorsArray = null;

         /*
          * A possibility exists that IEditModelScribblers could be garbage
          * collected while the following inner loop is working. In this case, a
          * ConcurrentModificationException could be thrown. In most cases, the
          * outer while loop should see at most one iteration. However, when the
          * garbage collection cycle reclaims IEditModelScribblers, the outer
          * while loop will ensure that the behavior does not affect the proper
          * update of the set of Scribbler Domains.
          */
         boolean done = false;
         int iterationCount = 0;
         scribblerDomains.clear();
         referenceCounts.clear();
         while (!done && iterationCount < 3) {
            try {
               Iterator itr = scribblers.iterator();
               while (itr.hasNext()) {
                  IEditModelScribbler scribbler = (IEditModelScribbler) itr
                        .next();
                  for (Iterator iter = scribbler.getScribblerDomains()
                        .iterator(); iter.hasNext();) {
                     IScribblerDomain element = (IScribblerDomain) iter.next();
                     int[] referenceCount = (int[]) referenceCounts
                           .get(element);
                     if (referenceCount != null) {
                        referenceCount[0]++;
                     } else {
                        referenceCounts.put(element, new int[] { 1 });
                     }
                  }
                  scribblerDomains.addAll(scribbler.getScribblerDomains());
               }
               for (Iterator iter = scribblerDomains.iterator(); iter.hasNext();) {
                  IScribblerDomain element = (IScribblerDomain) iter.next();
                  IResourceDescriptor[] descriptors = element
                        .getResourceDescriptors();
                  resourceDescriptors.addAll(Arrays.asList(descriptors));
               }
               done = true;
            } catch (ConcurrentModificationException cme) {
               scribblerDomains.clear();
               referenceCounts.clear();
            } finally {
               iterationCount++;
            }
         }

      }
   }

}
