/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.tests.infrastructure.emf;

import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.jst.j2ee.common.CommonPackage;
import org.eclipse.jst.j2ee.ejb.EJBJar;
import org.eclipse.stp.core.infrastructure.emf.DefaultScribblerDomain;
import org.eclipse.stp.core.infrastructure.emf.EclipseResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.EditModelEvent;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModel;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IResourceDescriptor;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.internal.infrastructure.emf.DynamicAdapterFactory;
import org.eclipse.stp.core.internal.infrastructure.emf.EditModel;
import org.eclipse.stp.core.internal.infrastructure.emf.ScribblerAdapter;
import org.eclipse.stp.core.internal.infrastructure.emfworkbench.EMFWorkbenchContext;
import org.eclipse.stp.core.tests.infrastructure.util.TestDomain;
import org.eclipse.stp.core.tests.infrastructure.util.TestEditModelListener;
import org.eclipse.stp.core.tests.infrastructure.util.TestHumanInteractionController;
import org.eclipse.stp.core.tests.infrastructure.util.TestOperation;
import org.eclipse.stp.core.tests.infrastructure.util.TestOperationArgs;
import org.eclipse.stp.core.tests.infrastructure.util.TestProgressMonitor;
import org.eclipse.stp.core.tests.infrastructure.util.TestWorkspace;

/**
 * 
 * Tests that should be added:
 * 
 * -Verify that Project close() and open() "reset" any given scribbler -Verify
 * that Edit Models listening to multiple resource sets respond correctly when
 * searching (that if they find the resource in their own list, they don't go
 * back to the resource sets).
 * 
 * <p>
 * This class has not been published as API so long as this comment is present.
 * </p>
 */
public class IEditModelScribblerTest extends TestCase {

   private static final String TEST_EDITMODEL_ID = "test.ejb.editmodel";

   /**
    * @return
    */
   private TestDomain createTestDomain() {
      return TestDomain.createTestDomain(TestWorkspace.getTestProject());
   }

   protected void setUp() throws Exception {
      super.setUp();
      TestWorkspace.init();

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      assertTrue("Ensure there are zero resources loaded.", resourceSet
            .getResources().size() == 0);
   }

   protected void tearDown() throws Exception {
      super.tearDown();

      // Some of the tests ensure that listeners are empty at the beginning
      System.gc();

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      assertTrue("Ensure there are zero resources loaded.", resourceSet
            .getResources().size() == 0);

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), "test.ejb.editmodel");

      // This should be true after the last few lines execute.
      // The notification from loading the resource should
      // trip the stale listeners to remove themselves and
      // autocorrect the resource reference counts
      assertTrue("Ensure there are zero EditModelListeners.",
            ((EditModel) editModel).getListeners().size() == 0);
   }

   public IEditModelScribblerTest(String name) {
      super(name);
   }

   public static TestSuite suite() {
      TestSuite suite = new TestSuite();

      suite.addTestSuite(IEditModelScribblerTest.class);

      // focus on a specific unit test
      // suite.addTest(new
      // IEditModelScribblerTest("testSaveReadOnlyIProgressMonitor_V1"));

      return suite;
   }

   /**
    * Try to fetch a resource by a workspace relative URI.
    * 
    * Methods under test: {@link IEditModelScribbler#getResource(URI)}.
    */
   public void testGetResourceURI() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());

      assertNotNull("A resource should NOT be null", resource);

      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertNotNull("A model object should have been retrived.", jar);
      assertEquals("The displayname should be T.", "T", jar.getDisplayName()); //$NON-NLS-2$

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      resource = scribbler.getResource(URI.createFileURI(""));
      assertNull("A resource should NOT be returned", resource);

      scribbler.close(null);
   }

   /**
    * Verify that isReadOnly returns the correct value.
    * 
    * Methods under test: {@link IEditModelScribbler#isReadOnly()}.
    */
   public void testIsReadOnly() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });
      assertTrue("The scribbler should be marked as read-only.", scribbler
            .isReadOnly());
      scribbler.close(null);

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });
      assertTrue("The scribbler should NOT be marked as read-only.", !scribbler
            .isReadOnly());
      scribbler.close(null);
   }

   /**
    * Try to fetch a resource by a Workspace relative IPath.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#getResource(org.eclipse.core.runtime.IPath)}.
    */
   public void testGetResourceIPath() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(), testDomain);

      Resource resource = scribbler.getResource(testDomain.getTestPath());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertNotNull("A model object should have been retrived.", jar);
      assertEquals("The displayname should be T.", "T", jar.getDisplayName()); //$NON-NLS-2$

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      // let's try a container!
      resource = scribbler.getResource(TestWorkspace.getTestProject()
            .getFullPath());
      assertNull("A resource should NOT be returned", resource);

      scribbler.close(null);

   }

   /**
    * Try to fetch a non-existent resource by a Workspace relative IPath.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#getResource(org.eclipse.core.runtime.IPath)}.
    */
   public void testGetResourceIPathNonExtistent() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      IFile ejbJarFile = TestWorkspace.getTestProject().getFile("ejb-jar.xml");
      Resource resource = scribbler.getResource(ejbJarFile.getFullPath());

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The resource should NOT have been loaded.", !resource
            .isLoaded());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      scribbler.close(null);

   }

   /**
    * Try to fetch a resource by a project relative URI. ALso try to save the
    * resource even though it was loaded for read-only usage.
    */
   public void testGetResourceURI_ReadOnly() {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = null;

      try {
         scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
               TestWorkspace.getTestProject(), testDomain);
      } catch (EditModelException e1) {
         assertNull(e1);
      }

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("Test");

      boolean exceptionThrown = false;
      try {
         scribbler.save(true, null);
      } catch (EditModelException e) {
         exceptionThrown = true;
      }
      assertTrue("An exception should be thrown from the save().",
            exceptionThrown);

      try {
         scribbler.close(null);
      } catch (EditModelException e) {
         assertNull(e);
      }
   }

   /**
    * Ensure that an Edit Model is notified when a resource that it's interested
    * in is loaded through some other means (but still via the
    * ProjectResourceSet).
    * 
    * Verify that when a resource is loaded outside of the edit model, that the
    * edit model will begin tracking the resource if it is interested.
    */
   public void testNotifiactionOnLoad() throws Exception {

      TestDomain testDomain = createTestDomain();

      // We go through the SOA EMFWorkbenchContext here to make sure
      // that the EMFResourceFactoryRegistry.INSTANCE is registered as
      // the resource factory against this project's ProjectResourceSet
      EMFWorkbenchContext context = EMFWorkbenchContext
            .getInstance(TestWorkspace.getTestProject());
      ResourceSet resourceSet = context.getResourceSet();

      /* local test setup: We need to ensure there are no pre-loaded resources */
      Resource resource = resourceSet.getResource(testDomain.getTestURI(),
            false);
      if (resource != null && resource.isLoaded())
         resource.unload();
      resourceSet.getResources().remove(resource);
      /* end setup */

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);

      TestEditModelListener testListener = new TestEditModelListener(
            EditModelEvent.LOADED_RESOURCE);
      editModel.addListener(testListener);

      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { testDomain }, false);

      assertTrue("Ensure there are zero resources loaded.", resourceSet
            .getResources().size() == 0);

      // Force the load of a resource which should prompt a
      // notification to the stale EditModelScribbler's listener
      resource = resourceSet.getResource(testDomain.getTestURI(), true);

      assertTrue("Ensure the edit model is managing the resource.",
            ((EditModel) editModel).isManagingResource(resource));

      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified()); // $NON-NL
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      assertNotNull("A model object should have been retrived.", jar);
      assertEquals("The displayname should be T.", "T", jar.getDisplayName()); //$NON-NLS-2$

      assertTrue(
            "A notification of the loaded resource should have been received.",
            testListener.getEvents().length == 1);
      assertTrue(
            "A notification of the loaded resource should have been received.",
            testListener.getEvents()[0].getEventCode() == EditModelEvent.LOADED_RESOURCE);

      scribbler.close(null);
      editModel.removeListener(testListener);

      resource.unload();
      resourceSet.getResources().remove(resource);
   }

   /**
    * Ensure that an Edit Model is notified when a resource that it's interested
    * in is loaded through some other means (but still via the
    * ProjectResourceSet).
    * 
    * Verify that when a resource is loaded outside of the edit model, that the
    * edit model will begin tracking the resource if it is interested.
    */
   public void testNotifiactionOnSave() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModel editModel = IEditModelFactory.eINSTANCE.getEditModel(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID);

      IEditModelScribbler scribbler = editModel.createScribbler(
            new IScribblerDomain[] { testDomain }, false);

      TestEditModelListener testListener = new TestEditModelListener(
            EditModelEvent.DIRTY);
      editModel.addListener(testListener);

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue("The scribbler should be dirty.", scribbler.isDirty());

      assertTrue(
            "A notification of the loaded resource should have been received.",
            testListener.getEvents().length == 1);
      assertTrue(
            "A notification of the loaded resource should have been received.",
            testListener.getEvents()[0].getEventCode() == EditModelEvent.DIRTY);

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.save(true, monitor);

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler.close(null);
      editModel.removeListener(testListener);
   }

   /**
    * Verify that the resource is not deleted until save() is invoked, and that
    * once save() is invoked, the resource no longer exists.
    * 
    * Method under test:
    * {@link IEditModelScribbler#deleteResource(Resource, IProgressMonitor)
    * 
    */
   public void testDeleteResourceResourceIProgressMonitor()
         throws EditModelException {

      TestDomain testDomain = createTestDomain();

      IFile testFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
            testDomain.getTestPath());
      assertTrue("The IFile we are going to delete should exist.", testFile
            .exists());

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(), testDomain);

      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      Resource resource = scribbler.getResource(testDomain.getTestDescriptor());

      TestProgressMonitor monitor = new TestProgressMonitor(2);
      scribbler.deleteResource(testDomain.getTestDescriptor(), monitor);
      monitor
            .assertMessages("IEditModelScribbler.deleteResource() should provide at least 2 messsages.");

      assertTrue("The IFile should be deleted immediately.", !testFile.exists());

      assertTrue("The scribbler should be dirty after a resource is deleted.",
            scribbler.isDirty());

      assertTrue(
            "Since there is only one scribbler, the write count should have dropped to 0 and unloaded the resource.",
            !resource.isLoaded());

      scribbler.save(true, null);

      assertTrue("The IFile should not exist after save() is invoked.",
            !testFile.exists());

      assertTrue("The scribbler should NOT be dirty after save().", !scribbler
            .isDirty());

      scribbler.close(null);
   }

   /**
    * Verify that the resource is not deleted until save() is invoked, and that
    * once save() is invoked, the resource no longer exists.
    * 
    * Method under test:
    * {@link IEditModelScribbler#deleteResource(IResourceDescriptor, IProgressMonitor)
    * 
    */
   public void testDeleteResource() throws EditModelException {

      TestDomain testDomain = createTestDomain();

      IFile testFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
            testDomain.getTestPath());
      assertTrue("The IFile we are going to delete should exist.", testFile
            .exists());

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(), testDomain);

      Resource resource = scribbler.getResource(testDomain.getTestPath());

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(2);
      scribbler.deleteResource(resource, monitor);
      monitor
            .assertMessages("IEditModelScribbler.deleteResource() should provide at least 2 messsages.");

      assertTrue("The scribbler should be dirty after a resource is deleted.",
            scribbler.isDirty());

      assertTrue("The IFile should be deleted immediately.", !testFile.exists());

      assertTrue(
            "Since there is only one scribbler, the write count should have dropped to 0 and unloaded the resource.",
            !resource.isLoaded());

      scribbler.save(true, null);

      assertTrue("The IFile should not exist after save() is invoked.",
            !testFile.exists());

      assertTrue("The scribbler should NOT be dirty after save().", !scribbler
            .isDirty());

      scribbler.close(null);
   }

   /**
    * Verify that the resource is not deleted until save() is invoked, and that
    * once save() is invoked, the resource no longer exists.
    * 
    * Method under test:
    * {@link IEditModelScribbler#deleteResource(IResourceDescriptor, IProgressMonitor)
    * 
    */
   public void testDeleteResourceWithTwoScribblers() throws EditModelException {

      TestDomain testDomain = createTestDomain();

      IFile testFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
            testDomain.getTestPath());
      assertTrue("The IFile we are going to delete should exist.", testFile
            .exists());

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(), testDomain);

      IEditModelScribbler theOtherScribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(), testDomain);

      Resource resource = scribbler.getResource(testDomain.getTestPath());
      ScribblerAdapter adapter = ScribblerAdapter.findAdapter(resource);
      assertTrue("The resource should have a write count of 2.", adapter
            .getEditReferenceCount() == 2);

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(2);
      scribbler.deleteResource(resource, monitor);
      monitor
            .assertMessages("IEditModelScribbler.deleteResource() should provide at least 2 messsages.");

      assertTrue("The scribbler should be dirty after a resource is deleted.",
            scribbler.isDirty());
      assertTrue(
            "The other scribbler should be dirty after a shared resource is deleted.",
            theOtherScribbler.isDirty());

      assertTrue("The IFile should be deleted immediately.", !testFile.exists());

      assertTrue("The write count should be 1.", adapter
            .getEditReferenceCount() == 1);

      scribbler.save(false, null);

      assertTrue("The IFile should not exist after save() is invoked.",
            !testFile.exists());

      assertTrue(
            "The scribbler should still be dirty after save() since there are two scribblers.",
            scribbler.isDirty());
      assertTrue(
            "The other scribbler should be dirty after a save() since it has not been saved.",
            theOtherScribbler.isDirty());
      scribbler.close(null);
      theOtherScribbler.close(null);
   }

   /**
    * Verify that the resource is not deleted until save() is invoked, and that
    * once save() is invoked, the resource no longer exists.
    * 
    * Method under test:
    * {@link IEditModelScribbler#deleteResource(IResourceDescriptor, IProgressMonitor)
    * 
    */
   public void testDiscardAfterDeleteResource() throws EditModelException {

      TestDomain testDomain = createTestDomain();

      IFile testFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
            testDomain.getTestPath());
      assertTrue("The IFile we are going to delete should exist.", testFile
            .exists());

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(), testDomain);

      Resource resource = scribbler.getResource(testDomain.getTestPath());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertNotNull("A model object should have been retrived.", jar);
      assertEquals("The displayname should be T.", "T", jar.getDisplayName()); //$NON-NLS-2$

      assertTrue("The resource should NOT have a modified flag set to true.",
            !resource.isModified());
      assertTrue("The scribbler should NOT be dirty.", !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(2);
      scribbler.deleteResource(resource, monitor);
      monitor
            .assertMessages("IEditModelScribbler.deleteResource() should provide at least 2 messsages.");

      assertTrue("The scribbler should be dirty after a resource is deleted.",
            scribbler.isDirty());

      assertTrue("The IFile should be deleted immediately.", !testFile.exists());

      scribbler.discard(null);

      assertTrue("The IFile should now exist after discard() is invoked.",
            testFile.exists());

      assertTrue("The scribbler should NOT be dirty after save().", !scribbler
            .isDirty());

   }

   /*
    * Test method for
    * 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.discard(org.eclipse.core.runtime.IProgressMonitor)'
    */
   public void testDiscardIProgressMonitor() throws EditModelException {

      // TODO What does it mean to get for Read and then Revert?
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertEquals(
            "Verify the field to be changed is set to the expected value.", "T", jar //$NON-NLS-2$
                  .getDisplayName());

      jar.setDisplayName("NewDisplayName");

      assertTrue("The scribbler should be dirty after setDisplayName().",
            scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.discard(monitor);

      assertTrue("The scribbler should NOT be dirty after discard().",
            !scribbler.isDirty());

      boolean exceptionThrown = false;
      try {
         scribbler.save(true, null);
      } catch (EditModelException e) {
         exceptionThrown = true;
      }

      assertTrue("The scribbler should NOT have been allowed to save.",
            exceptionThrown);

      monitor
            .assertMessages("IEditModelScribbler.revert() should provide at least 3 messsages.");

      exceptionThrown = false;
      try {
         resource = scribbler.getResource(testDomain.getTestURI());
      } catch (IllegalStateException e) {
         exceptionThrown = true;
      }

      assertTrue(
            "The scribbler should NOT have been allowed to fetch a new resource.",
            exceptionThrown);

      // close is not required as we discarded the scribbler
   }

   /**
    * Load the test resource, change an EMF field, save the resource, reload the
    * resource and ensure the save() was successful. Also ensure that a
    * reasonable number of progress messages are generated.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveBooleanIProgressMonitor_V1() throws Exception {
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue("The scribbler should be dirty.", scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.save(true, monitor);

      assertTrue("The scribbler should NOT be dirty after save().", !scribbler
            .isDirty());

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler.close(null);

      scribbler = null;

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });

      resource = scribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      scribbler.close(null);
   }

   /**
    * Load the test resource, change an EMF field, save the resource, reload the
    * resource and ensure the save() was successful. Also ensure that a
    * reasonable number of progress messages are generated.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveReadOnlyIProgressMonitor_V1() throws Exception {
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestDescriptor(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.save(true, monitor);

      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler.close(null);

      scribbler = null;

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });

      resource = scribbler.getResource(testDomain.getTestURI(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals(
            "Verify the field was not modified since the resource was loaded as read only.",
            "T", jar.getDisplayName());

      scribbler.close(null);
   }

   /**
    * Load the test resource as read only, change an EMF field, save the
    * resource, reload the resource and ensure the save() did not modify the
    * resource. Also ensure that a reasonable number of progress messages are
    * generated.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveReadOnlyIProgressMonitor_V2() throws Exception {
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestPath(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.save(true, monitor);

      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler.close(null);

      scribbler = null;

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });

      resource = scribbler.getResource(testDomain.getTestURI(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals(
            "Verify the field was not modified since the resource was loaded as read only.",
            "T", jar.getDisplayName());

      scribbler.close(null);
   }

   /**
    * Load the test resource, change an EMF field, save the resource, reload the
    * resource and ensure the save() was successful. Also ensure that a
    * reasonable number of progress messages are generated.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveReadOnlyIProgressMonitor_V3() throws Exception {
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.save(true, monitor);

      assertTrue(
            "The scribbler should NOT be dirty as its loaded as read only.",
            !scribbler.isDirty());

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler.close(null);

      scribbler = null;

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });

      resource = scribbler.getResource(testDomain.getTestURI(),
            IEditModelScribbler.OPTION_LOAD_AS_READ_ONLY
                  | IEditModelScribbler.OPTION_LOAD_RESOURCE);
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals(
            "Verify the field was not modified since the resource was loaded as read only.",
            "T", jar.getDisplayName());

      scribbler.close(null);
   }

   /**
    * Create two scribblers, load the test resource, change an EMF field, save
    * the resource, reload the resource and ensure that save(false) on one of
    * the scribblers respects the reference count from the other open scribbler.
    * Ensure that after the save(false), both scribblers are dirty.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveBooleanIProgressMonitor_V2() throws Exception {
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler editModelScribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      IEditModelScribbler operationScribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = operationScribbler.getResource(testDomain
            .getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue("The Operation scribbler should be dirty.", operationScribbler
            .isDirty());
      assertTrue("The Editor scribbler should be dirty.", editModelScribbler
            .isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      operationScribbler.save(false, monitor);

      assertTrue("The Operation scribbler should be dirty after save(false).",
            operationScribbler.isDirty());
      assertTrue("The Editor scribbler should be dirty after save(false).",
            editModelScribbler.isDirty());

      // monitor.assertMessages("IEditModelScribbler.save() should provide at
      // least 3 messsages.");

      operationScribbler.close(null);

      resource = editModelScribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      /*
       * Since no user interface controller is specified, close will call
       * save(false), which will cause a save
       */
      editModelScribbler.close(null);

      /* Verify the resources are saved */

      editModelScribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });
      resource = editModelScribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      editModelScribbler.close(null);
   }

   /**
    * Create two scribblers with similar but non-equivalent scribbler domains ,
    * load the test resource, change an EMF field, save the resource, reload the
    * resource and ensure that save(false) on one of the scribblers respects the
    * reference count from the other open scribbler. Ensure that after the
    * save(false), both scribblers are dirty.
    * 
    * Methods under test:
    * {@link IEditModelScribbler#save(boolean, org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testSaveBooleanIProgressMonitor_V3() throws Exception {
      TestDomain testDomain = createTestDomain();

      IScribblerDomain testDomain2 = new DefaultScribblerDomain() {

         private IResourceDescriptor[] descs = new IResourceDescriptor[] { new EclipseResourceDescriptor(
                                                   TestWorkspace
                                                         .getTestProject()
                                                         .getFile(".project")) };

         public String getEditModelLabel() {
            return TEST_EDITMODEL_ID;
         }

         public IResourceDescriptor[] getResourceDescriptors() {
            return descs;
         }
      };

      IEditModelScribbler editModelScribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain,
                        testDomain2 });

      IEditModelScribbler operationScribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = operationScribbler.getResource(testDomain
            .getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue("The Operation scribbler should be dirty.", operationScribbler
            .isDirty());
      assertTrue("The Editor scribbler should be dirty.", editModelScribbler
            .isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      operationScribbler.save(false, monitor);

      assertTrue(
            "The Operation scribbler should NOT be dirty after save(false).",
            !operationScribbler.isDirty());
      assertTrue("The Editor scribbler should be dirty after save(false).",
            !editModelScribbler.isDirty());

      // monitor.assertMessages("IEditModelScribbler.save() should provide at
      // least 3 messsages.");

      operationScribbler.close(null);

      resource = editModelScribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      /*
       * Since no user interface controller is specified, close will call
       * save(false), which will cause a save
       */
      editModelScribbler.close(null);

      /* Verify the resources are saved */

      editModelScribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });
      resource = editModelScribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      editModelScribbler.close(null);
   }

   /**
    * Load a resource, make a modification, invoke revert, ensure that the
    * modification has been replaced. Also ensure that a reasonable number of
    * progress messages are generated.
    * 
    * 
    * Methods under test:
    * {@link IEditModelScribbler#revert(org.eclipse.core.runtime.IProgressMonitor)},
    * {@link IEditModelScribbler#isDirty()}
    */
   public void testRevertIProgressMonitor() throws Exception {

      // TODO What does it mean to get for Read and then Revert?
      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      assertEquals(
            "Verify the field to be changed is set to the expected value.", "T", jar //$NON-NLS-2$
                  .getDisplayName());

      jar.setDisplayName("NewDisplayName");

      assertTrue("The scribbler should be dirty after setDisplayName().",
            scribbler.isDirty());

      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.revert(monitor);

      assertTrue("The scribbler should NOT be dirty after revert().",
            !scribbler.isDirty());

      monitor
            .assertMessages("IEditModelScribbler.revert() should provide at least 3 messsages.");

      resource = scribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was reverted.", "T", jar //$NON-NLS-2$
            .getDisplayName());

      scribbler.close(null);
   }

   /**
    * Tests the behavior that undo will restore the state of the Scribbler prior
    * to execution of the last command. The Scribbler assumes that the command
    * is intelligent enough to fully undo the modification that it executed.
    * 
    * The test only validates undo across one operation and on one Scribbler.
    * 
    * See testExecuteUndoAcrossScribblers for a test that spans the execution
    * and consistent undo across two Scribblers with equivalent IUndoContexts.
    */
   public void testExecuteUndo() throws Exception {

      IUndoableOperation setDisplayNameOp = new TestOperation();

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      String currentDisplayName = jar.getDisplayName();

      TestOperationArgs args = new TestOperationArgs("ATestDisplayName", jar);

      assertFalse("The display name should be the new value.", args
            .getNewDisplayName().equals(jar.getDisplayName()));

      scribbler.execute(setDisplayNameOp, null, args);

      assertEquals("The display name should be the new value.", args
            .getNewDisplayName(), jar.getDisplayName());

      scribbler.undo(setDisplayNameOp, null, args);

      assertEquals("The display name should be the new value.",
            currentDisplayName, jar.getDisplayName());

      scribbler.close(null);

   }

   /**
    * Verify that an IEditModelScribbler can adapt to the EMF CommandStack
    * interface and execute standard EMF Commands.
    * 
    * Also verify that the command can be undone through the EMF Command Stack.
    * 
    * @see org.eclipse.emf.common.command.CommandStack
    * @see org.eclipse.emf.common.command.Command
    */
   public void testActLikeAnEMFCommandStack() throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      String oldDisplayName = jar.getDisplayName();
      String newDisplayNameValue = "NewDisplayName";

      AdapterFactory adapterFactory = new DynamicAdapterFactory(null);
      Object adaptedType = scribbler.getAdapter(CommandStack.class);
      assertTrue(
            "Verify that the returned object is of type org.eclipse.emf.common.command.CommandStack.",
            adaptedType instanceof CommandStack);
      CommandStack commandStack = (CommandStack) adaptedType;

      EditingDomain domain = new AdapterFactoryEditingDomain(adapterFactory,
            commandStack);

      Command setDisplayNameCmd = SetCommand.create(domain, jar,
            CommonPackage.eINSTANCE
                  .getCompatibilityDescriptionGroup_DisplayName(),
            newDisplayNameValue);

      // execute a set command
      domain.getCommandStack().execute(setDisplayNameCmd);

      assertEquals("The display name should be the new value.",
            newDisplayNameValue, jar.getDisplayName());

      // try to undo the last command
      domain.getCommandStack().undo();

      assertEquals("The display name should be the new value.", oldDisplayName,
            jar.getDisplayName());

      scribbler.close(null);

   }

   /**
    * Verify that an IEditModelScribbler can adapt to the EMF CommandStack
    * interface and execute standard EMF Commands.
    * 
    * Also verify that an undo sent directly to the scribbler will undo the last
    * command.
    * 
    * @see org.eclipse.emf.common.command.CommandStack
    * @see org.eclipse.emf.common.command.Command
    * @see #testActLikeAnEMFCommandStack()
    * 
    */
   public void testActLikeAnEMFCommandStackAndIntegrateScribblerUndo()
         throws Exception {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      String oldDisplayName = jar.getDisplayName();
      String newDisplayNameValue = "NewDisplayName";

      AdapterFactory adapterFactory = new DynamicAdapterFactory(null);
      Object adaptedType = scribbler.getAdapter(CommandStack.class);
      assertTrue(
            "Verify that the returned object is of type org.eclipse.emf.common.command.CommandStack.",
            adaptedType instanceof CommandStack);
      CommandStack commandStack = (CommandStack) adaptedType;

      EditingDomain domain = new AdapterFactoryEditingDomain(adapterFactory,
            commandStack);

      Command setDisplayNameCmd = SetCommand.create(domain, jar,
            CommonPackage.eINSTANCE
                  .getCompatibilityDescriptionGroup_DisplayName(),
            newDisplayNameValue);

      // execute a set command
      domain.getCommandStack().execute(setDisplayNameCmd);

      assertEquals("The display name should be the new value.",
            newDisplayNameValue, jar.getDisplayName());

      // try to undo the last command
      scribbler.undoLast(null, null);

      assertEquals("The display name should be the new value.", oldDisplayName,
            jar.getDisplayName());

      scribbler.close(null);

   }

   /**
    * 
    * Verify that a call to undoLast() will undo the last command executed by
    * the scribbler.
    * 
    * The following test will execute an operation and then request that the
    * last operation executed is undone.
    * 
    * @see #testExecuteUndo
    */
   public void testUndoLast() throws Exception {
      IUndoableOperation setDisplayNameOp = new TestOperation();

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForRead(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      String currentDisplayName = jar.getDisplayName();

      TestOperationArgs args = new TestOperationArgs("ATestDisplayName", jar);

      assertFalse("The display name should be the new value.", args
            .getNewDisplayName().equals(jar.getDisplayName()));

      scribbler.execute(setDisplayNameOp, null, args);

      assertEquals("The display name should be the new value.", args
            .getNewDisplayName(), jar.getDisplayName());

      // test undoLast
      scribbler.undoLast(null, args);

      assertEquals("The display name should be the new value.",
            currentDisplayName, jar.getDisplayName());

      scribbler.close(null);
   }

   /**
    * Test the behavior that close() on a dirty scribbler properly prompts the
    * user.
    * 
    */
   public void testPromptOnClose() {
      // TODO Auto-generated method stub

   }

   /*
    * Test method for
    * 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.close(org.eclipse.core.runtime.IProgressMonitor)'
    */
   public void testCloseIProgressMonitor() throws EditModelException {

      TestDomain testDomain = createTestDomain();

      IEditModelScribbler scribbler = IEditModelFactory.eINSTANCE
            .createScribblerForWrite(TestWorkspace.getTestProject(),
                  TEST_EDITMODEL_ID, new IScribblerDomain[] { testDomain });

      Resource resource = scribbler.getResource(testDomain.getTestURI());
      EJBJar jar = (EJBJar) resource.getContents().get(0);

      jar.setDisplayName("NewDisplayName");

      assertTrue("The resource should have a modified flag set to true.",
            resource.isModified()); // $NON-NL
      assertTrue("The scribbler should be dirty.", scribbler.isDirty());

      TestHumanInteractionController controller = new TestHumanInteractionController(
            Boolean.TRUE);
      scribbler.setUserInterfaceController(controller);
      TestProgressMonitor monitor = new TestProgressMonitor(3);
      scribbler.close(monitor);

      assertTrue("The scribbler should NOT be dirty after close().", !scribbler
            .isDirty());

      monitor
            .assertMessages("IEditModelScribbler.save() should provide at least 3 messsages.");

      scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
            TestWorkspace.getTestProject(), TEST_EDITMODEL_ID,
            new IScribblerDomain[] { testDomain });

      resource = scribbler.getResource(testDomain.getTestURI());
      jar = (EJBJar) resource.getContents().get(0);

      assertEquals("Verify the changed field was modified.", "NewDisplayName",
            jar.getDisplayName());

      scribbler.close(null);

   }

   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.redoLast(org.eclipse.core.runtime.IProgressMonitor)'
   // */
   // public void testRedoLast() {
   //
   // }
   // 
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getScribblerDomains()'
   // */
   // public void testGetScribblerDomains() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.redo(IUndoableOperation,
   // * org.eclipse.core.runtime.IProgressMonitor)'
   // */
   // public void testRedo() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.setUserInterfaceController(IHumanInteractionController)'
   // */
   // public void testSetUserInterfaceController() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getUserInterfaceController()'
   // */
   // public void testGetUserInterfaceController() {
   //
   // }
   //	
   // /**
   // * Verify that executing an operation in one scribbler will be undone from
   // a
   // * separate Scribbler with an equivalent IUndoContext.
   // *
   // */
   // public void testExecuteUndoAcrossScribblers() {
   // // TODO Auto-generated method stub
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getSortedResources()'
   // */
   // public void testGetSortedResources() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getOperationHistory()'
   // */
   // public void testGetOperationHistory() {
   //
   // }
   //
   // /*
   // * Test method for
   // * 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getLabel()'
   // */
   // public void testGetLabel() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.matches(IUndoContext)'
   // */
   // public void testMatches() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.commandStackChanged(EventObject)'
   // */
   // public void testCommandStackChanged() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.createResource(IResourceDescriptor,
   // * org.eclipse.core.runtime.IProgressMonitor)'
   // */
   // public void testCreateResourceIResourceDescriptorIProgressMonitor() {
   //
   // }
   //

   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getResource(IResourceDescriptor)'
   // */
   // public void testGetResourceIResourceDescriptor() {
   //
   // }
   //
   // /*
   // * Test method for
   // *
   // 'com.ibm.soa.infrastructure.emf.internal.EditModelScribbler.getResource(IResourceDescriptor,
   // * int)'
   // */
   // public void testGetResourceIResourceDescriptorInt() {
   //
   // }
   //
   // /**
   // * Test the behavior that an operation executed from one Scribbler can be
   // * undone from another Scribbler with the same set of Scribbler Domains.
   // */
   // public void testOperationExecutionAcrossScribblers() {
   // // TODO Auto-generated method stub
   // }

}
