/*******************************************************************************
 * Copyright (c) 2005-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core.operations;

import java.io.File;
import java.io.IOException;

import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IServiceDescriptor;
import org.eclipse.stp.soas.deploy.core.IPackageConfiguration;
import org.eclipse.stp.soas.deploy.core.IPackageConstructorExtension;
import org.eclipse.stp.soas.deploy.core.IPackageCreationContext;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.internal.deploy.core.PackageCreationContext;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.IProgressConstants;

/**
 * @author rcernich
 * 
 * Created on Jan 26, 2005
 */
public class CreatePackageJob extends Job {

	private IServiceDescriptor mPackage;
	private IPackageConstructorExtension mConstructor;
	private IPackageCreationContext mContext;
	private IPackageConfiguration mConfiguration;

	/**
	 * 
	 */
	public CreatePackageJob(IServiceDescriptor pkg,
							IPackageConstructorExtension constructor,
							IPackageConfiguration configuration) {
		super(DeployCorePlugin.getDefault().getResourceString(
				"CreatePackageJob.task.name")); //$NON-NLS-1$
		mPackage = pkg;
		mConstructor = constructor;
		mConfiguration = configuration;
		setRule(mPackage.getFile().getWorkspace().getRoot());
		setUser(true);
		setSystem(false);
	}

	public CreatePackageJob(IServiceDescriptor pkg,
							IPackageConstructorExtension constructor) {
		this(pkg, constructor, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IStatus run(IProgressMonitor monitor) {
		IStatus status = Status.OK_STATUS;
		monitor.beginTask(DeployCorePlugin.getDefault().getResourceString(
				"CreatePackageJob.task.name"), 1000); //$NON-NLS-1$

		try {
			// Build the workspace
			buildWorkspace(monitor);
			checkCancelled(monitor);

			// Check for errors
			checkForErrors(monitor);
			checkCancelled(monitor);

			// Create the package
			createPackage(monitor);

			setProperty(IProgressConstants.KEEP_PROPERTY, Boolean.TRUE);
			setProperty(IProgressConstants.ACTION_PROPERTY, new Action(
					DeployCorePlugin.getDefault().getResourceString(
							"CreatePackageJob.action.name")) { //$NON-NLS-1$

						public void run() {
							Display display = PlatformUI.getWorkbench()
									.getDisplay();
							display.syncExec(new Runnable() {

								public void run() {
									boolean disposeShell = false;
									Display display = Display.getCurrent();
									Shell shell;
									Shell shells[] = display.getShells();
									if (shells.length > 0) {
										shell = shells[shells.length - 1];
									}
									else {
										disposeShell = true;
										shell = new Shell();
									}
									try {
										MessageDialog
												.openInformation(
														Display
																.getDefault()
																.getActiveShell(),
														DeployCorePlugin
																.getDefault()
																.getResourceString(
																		"CreatePackageFileAction.title.success"), //$NON-NLS-1$
														DeployCorePlugin
																.getDefault()
																.getResourceString(
																		"CreatePackageFileAction.message.success", //$NON-NLS-1$
																		new Object[] { mPackage
																				.getFile()
																				.getFullPath()
																				.toString()}));
									}
									finally {
										if (disposeShell) {
											shell.dispose();
										}
									}
								}
							});
						}
					});
		}
		catch (CoreException e) {
			status = e.getStatus();
		}
		catch (InterruptedException e) {
			status = Status.CANCEL_STATUS;
		}

		monitor.done();

		return status;
	}

	private void buildWorkspace(IProgressMonitor monitor) throws CoreException {
		IProgressMonitor buildMonitor = new SubProgressMonitor(monitor, 400);

		monitor.subTask(DeployCorePlugin.getDefault().getResourceString(
				"CreatePackageJob.subtask.build")); //$NON-NLS-1$

		mPackage.getFile().getWorkspace().build(
				IncrementalProjectBuilder.INCREMENTAL_BUILD, buildMonitor);
		buildMonitor.done();
	}

	private void checkForErrors(IProgressMonitor monitor) throws CoreException {
		monitor.subTask(DeployCorePlugin.getDefault().getResourceString(
				"CreatePackageJob.subtask.errorCheck")); //$NON-NLS-1$
		IStatus[] validateStatus = mConstructor.validatePackage(mPackage,
				mConfiguration, null);

		if ((validateStatus != null) && (validateStatus.length > 0)) {
			MultiStatus status = new MultiStatus(
					DeployCorePlugin.getDefault().getBundle()
							.getSymbolicName(),
					-1,
					validateStatus,
					DeployCorePlugin.getDefault().getResourceString(
							"CreateDeployPackagesJob.message.log.buildPackage"), //$NON-NLS-1$
					null);
			if (status.getSeverity() == IStatus.ERROR) {
				throw new CoreException(status);
			}
			else if (status.getSeverity() == IStatus.WARNING) {
				// RJC: TODO: Do we need to do anything here?
			}
		}

		monitor.worked(100);
	}

	private void createPackage(IProgressMonitor monitor) throws CoreException {
		monitor.subTask(DeployCorePlugin.getDefault().getResourceString(
				"CreatePackageJob.subtask.createPackage")); //$NON-NLS-1$
		mContext = new PackageCreationContext(null, DeploymentExtensionManager
				.getInstance().getOutputFolder(mPackage,
						mConstructor.getName()).getLocation().toFile(),
				createTempFolder());
		if (!mContext.getOutputFolder().exists()) {
			mContext.getOutputFolder().mkdirs();
		}
		mConstructor.createPackage(mPackage, mContext, mConfiguration);
		deleteFile(mContext.getTempFolder());
		monitor.worked(500);
	}

	private void checkCancelled(IProgressMonitor monitor)
			throws InterruptedException {
		if (monitor.isCanceled()) {
			throw new InterruptedException();
		}
	}

	private File createTempFolder() throws CoreException {
		try {
			File tmpFile = File.createTempFile("deploy_temp", ".tmp"); //$NON-NLS-1$ //$NON-NLS-2$
			File tempFolder = new File(tmpFile.getParentFile(), tmpFile
					.getName().substring(0,
							tmpFile.getName().lastIndexOf(".tmp"))); //$NON-NLS-1$
			tempFolder.mkdir();

			tmpFile.deleteOnExit();
			tempFolder.deleteOnExit();

			return tempFolder;
		}
		catch (IOException e) {
			IStatus status = new Status(IStatus.ERROR, DeployCorePlugin
					.getDefault().getBundle().getSymbolicName(), -1,
					DeployCorePlugin.getDefault().getResourceString(
							"CreatePackageJob.error.tempDir"), e);
			throw new CoreException(status);
		}
	}

	private void deleteFile(File file) {
		File[] files = file.listFiles();
		if (files != null && files.length > 0) {
			for (int index = 0, count = files.length; index < count; ++index) {
				deleteFile(files[index]);
			}
		}
		file.delete();
	}
}